/*-----------------------------------------------------------------------------------
Copyright (c)2003 Borland Software Corporation. Patents pending. All rights reserved.
-----------------------------------------------------------------------------------*/

package problem_domain;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

/**
 * This class represents the catalog-like description of the product being sold. <p>
 * If you need to track quantity and aren't selling specific "thing"-like products (that is, ones
 * that are serialized), then add a quantityOnHand attribute to this class.
 * @stereotype description
 * @persistent
 * @version 1.0*/
public class ProductDesc {

    /** Use it if you need to identify Products as specific types. 
     * @rdbLogicalType INTEGER*/
    private int type;

    /** Product name. For example: Goetze's Caramel Cremes 
     * @rdbLogicalType VARCHAR*/
    private String name;

    /** This is the unique identifying number. Something like a UPC for retail products. 
     * @rdbLogicalType VARCHAR*/
    private String itemNumber;

    /** Default price. 
     * @rdbLogicalType NUMERIC*/
    private BigDecimal defaultPrice;

    /** Some prose describing the product in all its glory. 
     * @rdbLogicalType VARCHAR*/
    private String description;

    /**
     *List of prices. If this list has elements, then they are checked. Otherwise, the default price is used. <p>
     * @supplierCardinality 1..*
     * @associates <b>ProductPrice</b>*/
    private Vector priceObjects;

    /**
     * @shapeType Hyperlink 
     */
    /*# private Product attribute1; */

/* ========================================
 * Constructors
 * ======================================== */

    /** Constructor requires all parameters. Type is defaulted to 0 since we aren't using it. */
    public ProductDesc(String anItemNum, String aDesc, String aName, BigDecimal aPrice) {
        type = 0; // not currently used
        itemNumber = anItemNum;
        description = aDesc;
        name = aName;
        defaultPrice = aPrice;
        priceObjects = new Vector();
    } // END ProductDesc(...)

/* ========================================
 * Business Methods
 * ======================================== */

    /**  */
    public int calcQtySold() {
        return 0;
    }

    public BigDecimal calcTotalSales() {
        return new BigDecimal(0);
    }

    /** This will check if there are any Price objects, returning that price, else the default price. */
    public BigDecimal getPrice() {
        // Default the price
        BigDecimal price = defaultPrice;
        // Check to see if we have any associated PriceObjects
        if (priceObjects.size() != 0) {
            // Work through list to find
            // the right price...
            Enumeration iter = listPriceObjects();
            ProductPrice prodPrice; // reference
            Date today = new Date();
            while (iter.hasMoreElements()) {
                prodPrice = (ProductPrice)iter.nextElement();
                if (prodPrice.isValidOnDate(today)) {
                    // Deal with Qty & Qty discounts...
                    price = prodPrice.getPrice();
                    break; // Look no further :=)
                } // ENDIF (valid)
            } // ENDWHILE (price enum)
        } // ENDIF (price objects exist)
        else {
        } // ENDELSE (use dflt price)
        return price;
    } // END calcPriceForQty()

    /** By passing in the qty, we can efficiently do quantity discounts at this level in the object model. */
    public BigDecimal calcPriceForQty(int qty) {
        BigDecimal price = getPrice();
        if (qty > 1) {
            price = price.multiply(new BigDecimal(qty));
        }
        return price;
    }

/* ========================================
 * Class-Level Business Methods
 * ======================================== */

    /**
     *This is a class-level method used to return a list of all available product descriptions currently in
     * the persistent store.
     */
    public static Vector listProductDescs() {
        return new Vector();
    }

/* ========================================
 * Accessor/Mutator Methods
 * ======================================== */

    public void addPriceObject(ProductPrice priceObj) {
        priceObjects.addElement(priceObj);
    }

    public void removePriceObject(ProductPrice priceObj) {
        if (priceObjects.removeElement(priceObj)) {
        }
        else {
            throw new RuntimeException("Could not remove Price: " + priceObj);
        }
    }

    public Enumeration listPriceObjects() {
        return priceObjects.elements();
    }

    public String getItemNumber() {
        return itemNumber;
    }

    public void setItemNumber(String anItemNumber) {
        itemNumber = anItemNumber;
    }

    public String getName() {
        return name;
    }

    public void setName(String aName) {
        name = aName;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String aDescription) {
        description = aDescription;
    }

    public BigDecimal getDefaultPrice() {
        return defaultPrice;
    }

    public void setDefaultPrice(BigDecimal aDefaultPrice) {
        defaultPrice = aDefaultPrice;
    }
} // ENDCLASS ProductDesc

