/*-----------------------------------------------------------------------------------
Copyright (c)2003 Borland Software Corporation. Patents pending. All rights reserved.
-----------------------------------------------------------------------------------*/

package problem_domain;

import java.math.BigDecimal;
import java.sql.SQLException;
import java.text.NumberFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import javax.swing.JOptionPane;

import server.DMServer;
import data_management.SaleDM;

/**
 * This represents the overall transaction record of the sale. For purposes of eliminating the need to re-calc everything, we can redundantly store subtotal and tax to make it easy to get the info.
 * @stereotype moment-interval
 * @persistent
 * @robustness Controller
 * @since
 * @subtitle The entity*/
public class CashSale
{
    private final static int SALE_NEW = 0;
    private final static int SALE_OPEN = 1;
    private final static int SALE_CLOSED = 2;

    /** Poor way to do taxes (but it's easy <g>). */
    public final static double TAX_RATE = 0.06;

    /**
     * This is the list of sale details (line items)
     * @link aggregation
     * @associates <b>problem_domain.CashSaleDetail</b>
     * @supplierCardinality 1..**/
    private Vector detailList;

    /** Timestamp of this sale 
     * @rdbLogicalType DATE*/
    private Date time;

    /**
     * Subtotal of all line items. NOTE: this could be calculated on the fly
     * by iterating through the details, but once we
     * are all done with the transaction, we have decided to purposefully
     * store the result.
     * This is a trade-off based on the fact that this transaction, once
     * completed, is never
     * edited again. By redundantly storing data, we are allowing a quick
     * answer to support
     * the various analysis "runs" made against the data.
     * @rdbLogicalType NUMERIC
     */
    private BigDecimal subtotal;

    /** This represents a discount at the entire sale level. 
     * @rdbLogicalType NUMERIC*/
    private BigDecimal discountAmount;

    /** This is the total tax for this sale. 
     * @rdbLogicalType NUMERIC*/
    private BigDecimal tax;

    /**
     * This allows us to "plug in" different cash sale sequences to support
     * other ways to take in the payment. Examples
     * include: Cash, Credit, Debit, Check.
     * @supplierCardinality 1
     * @rdbLogicalType OTHER*/
    private IMakeCashSale anICashSaleSequencerPlugInPoint;

    /**
     * This is the payment. Based on this value and the total due, we can
     * compute the change due.
     * @supplierCardinality 0..*
     * @clientCardinality 1
     * @rdbLogicalType NUMERIC
     */
    private BigDecimal payment;

    /** Indicates if sale is in process or completed. 
     * @rdbLogicalType INTEGER*/
    private int status = SALE_NEW;

    /**
     * Access to the Data Management Layer.
     * @ NOTE: shown here, but could really just be transiently created in methods that require DM layer...
     * @rdbLogicalType OTHER*/
    private static IDM myDm;
    private DMServer dmServer = null;

    /**
     * This will be used as a unique identifier for CashSale objects.
     */
    private int id = 0;

    /**
     * @shapeType Hyperlink 
     */
    /*# private Sale attribute1; */

    /**
     * @shapeType RobustnessAssociation 
     * @link
     * @label lnkSaleDM
     * @directed
     */
    /*# private SaleDM attribute2; */

    /* ========================================
    * Constructors
    * ======================================== */

    public CashSale()
    {
        clearValues();
    }

    /* ========================================
    * Business Methods
    * ======================================== */

    /**
     * Figure out if the sale and payment match up.
     * Throw an exception if payment is too little.
     */
    public BigDecimal makeCashSale(BigDecimal paymentAmt)
    throws InsuffPaymentException
    {
        payment = paymentAmt;
        // compute change due
        BigDecimal total = calcTotal();
        BigDecimal change = paymentAmt.subtract(total);

        // If change is negative, insufficient payment made!
        if (change.compareTo( new BigDecimal(0.0)) < 0 )
            {
                NumberFormat cf = NumberFormat.getCurrencyInstance();
                String msg = new
                StringBuffer("Payment of ")
                	.append(cf.format(paymentAmt))
                	.append(" less than Total: ")
                	.append(cf.format(total))
                    .toString();
                throw new InsuffPaymentException(msg);
        	}
        // indicate sale is complete
        completed();
        return change;
    } // END makeCashSale()

	/**
     * Calulate the total of all the line items (but no tax).
     */
    public BigDecimal calcSubtotal()
    {
        if ( subtotal.compareTo( new BigDecimal("0.0") ) == 0 )
        {
	        Enumeration iter = getDetails();
	        CashSaleDetail detail = null;
//		Use this line if you want to try to track down a bug
//		Run the audits to see if this line is offensive or not :=)
//        	BigDecimal subtotal = new BigDecimal(0.0);
	        while (iter.hasMoreElements())
	        {
	            detail = (CashSaleDetail)iter.nextElement();
	            subtotal = subtotal.add(detail.calcTotal());
	        } // ENDWHILE
        }  // ENDIF
        return subtotal;
    }

    /** Not implemented */
    public BigDecimal calcDiscountAmount()
    {
        return discountAmount;
    }

    /** INFLEXIBLE IMPLEMENTATION of hard-coded fixed tax rate */
    public BigDecimal calcTax()
    {
        subtotal = calcSubtotal();
        tax = subtotal.multiply( new BigDecimal(TAX_RATE) );
        return tax;
    }

    /**
     * Sum up the subtotal and the tax.
     */
    public BigDecimal calcTotal()
    {
        subtotal = calcSubtotal();
        tax = calcTax();
        BigDecimal total = new BigDecimal(0.0);
        total = total.add(subtotal);
        total = total.add(tax);
        // use this value for easy viewing during debugging
        double dbleTotal = total.floatValue();
        return total;
    }

    public BigDecimal recalcTotal()
    {
        return calcTotal();
    }

    /** Not yet implemented. */
    public boolean verifyAvailability()
    {
        return false;
    }

    /** Return whether the sale is completed or not. */
    public boolean isCompleted()
    {
        if (status == SALE_CLOSED)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /** Indicate if the sale is still pending */
    public boolean isPending()
    {
        if (status == SALE_OPEN)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /** Mark the sale as completed */
    public void completed()
    {
        status = SALE_CLOSED;
    }

	/**
     * Clear/reset the values
     */
    public void clearValues()
    {
        id = 0;
        time = new Date();
        detailList = new Vector();
        discountAmount = new BigDecimal(0.0);
        payment = new BigDecimal(0.0);
        status = SALE_NEW;
        subtotal = new BigDecimal(0.0);
        tax = new BigDecimal(0.0);
    }

    /* ========================================
    * Persistence Methods
    * ======================================== */

    /**
     * Save this sale to persistent storage.
     */
    public void save() throws SQLException
    {
		// Save myself
        myDm.save(this);
    } // END save()

    /* ========================================
    * Class-Level Business Methods
    * ======================================== */

	/**
     * Return a list of all cash sales.
     */
    public Vector listCashSales()
    {
        Vector cashSaleList = null;
		try
        {
            cashSaleList = myDm.getAll();
        }
        catch( SQLException sqe )
        {
            JOptionPane.showMessageDialog(null, sqe.getMessage(),
                "Database Error in Get All", JOptionPane.ERROR_MESSAGE);
        }

        return cashSaleList;
    }

    /** Not implemented */
    public static BigDecimal calcTotalCashSales()
    {
        return new BigDecimal(0);
    }

    /** Not implemented */
    public static int calcTotalCashSalesQty()
    {
        return 0;
    }

    /** Not implemented */
    public static BigDecimal calcAvgTotalCashSale()
    {
        return new BigDecimal(0);
    }

    /** Not implemented */
    public static int calcAvgTotalCashSaleQty()
    {
        return 0;
    }

    /** Not implemented */
    public static int calcCashSalesRate()
    {
        return 0;
    }

    /** Not implemented */
    public static int calcCashSalesQtyRate()
    {
        return 0;
    }

    /* ========================================
    * Accessor/Mutator Methods
    * ======================================== */

    public void addSaleDetail(CashSaleDetail aCashSaleDetail)
    {
        // Change status on first detail addition
        if (status == SALE_NEW)
        {
            status = SALE_OPEN;
        }
        // Do a simple check to see if same item has been scanned
        // before. If so, then increment qty.
        CashSaleDetail detail = null;
        boolean foundIt = false;
        for (int i = 0; i < detailList.size(); ++i)
        {
            detail = (CashSaleDetail)detailList.elementAt(i);
            if (detail.getProductDesc().getItemNumber() ==
            aCashSaleDetail.getProductDesc().getItemNumber())
            {
                foundIt = true;
                detail.setQty(detail.getQty() + 1); // Change the real thing
                break; // Out of for loop
            } // ENDIF (match)
        } // ENDFOR (each element)
        // Add it if not already there
        if (!foundIt)
        {
            detailList.addElement(aCashSaleDetail);
        }
    } // END addSaleDetail()

    public Enumeration getDetails()
    {
        return detailList.elements();
    }

    public Vector getDetailList()
    {
        // May want to return a clone instead of the real things
        return detailList;
    }

    public Date getTime()
    {
        return new Date(time.getTime());
    }

    public void setCompleted(boolean bool)
    {
        if (bool)
        {
            status = SALE_CLOSED;
        }
        else
        {
            status = SALE_OPEN;
        }
    }

    public int getId()
    {
        return id;
    }

    public BigDecimal getSubtotal()
    {
        return subtotal;
    }

    public BigDecimal getTax()
    {
        return tax;
    }

    public void setTax(BigDecimal tax)
    {
        this.tax = tax;
    }

    public void setSubtotal(BigDecimal subtotal)
    {
        this.subtotal = subtotal;
    }

    public void setTime(Date time)
    {
        this.time = time;
    }

    public void setId(int id)
    {
    	this.id = id;
    }

    public void setDm(IDM dm)
    {
        myDm = dm;
    }

    public void setDmServer(DMServer dmServer)
    {
		this.dmServer = dmServer;
		setDm( new SaleDM( dmServer ) );
	}
} // END class CashSale

