/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/threads/SynchronizedWait.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.threads;

import java.lang.reflect.Modifier;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISynchronizedStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;

/**
 * @author d034036
 *
 */
public class SynchronizedWait extends JomTestVisitor {

  /** keep in sync with tests.xml! */

  private static final String NAME = "Wait only in synchronized loops";

  static public final String WAIT_OUTSIDE_SYNC_BLOCK = "sychronizedWait.1"; //$NON-NLS-1$
  static public final String WAIT_OUTSIDE_LOOP = "sychronizedWait.2"; //$NON-NLS-1$

  private Stack loopStack = new Stack();
  private Stack synchronizedStack = new Stack();

  public String getTestName() {
    return NAME;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation)
   */
  public void endVisit(IMethodInvocation node) {
    if (node.getName().equals("wait")) {
      if(!insideSynchronized()) {
        addError(WAIT_OUTSIDE_SYNC_BLOCK, null, node);
      }
      if (!insideLoop()) {
        addError(WAIT_OUTSIDE_LOOP, null, node);
      }
    }
  }

  public void endVisit(IDoStatement node) {
    ((Stack) loopStack.peek()).pop();
  }

  public void endVisit(IForStatement node) {
    ((Stack) loopStack.peek()).pop();
  }

  public void endVisit(ISynchronizedStatement node) {
    ((Stack) synchronizedStack.peek()).pop();
  }

  public void endVisit(IWhileStatement node) {
    ((Stack) loopStack.peek()).pop();
  }

  public boolean visit(IDoStatement node) {
    ((Stack) loopStack.peek()).push(Boolean.TRUE);
    return true;
  }

  public boolean visit(IForStatement node) {
    ((Stack) loopStack.peek()).push(Boolean.TRUE);
    return true;
  }

  public boolean visit(ISynchronizedStatement node) {
    ((Stack) synchronizedStack.peek()).push(Boolean.TRUE);
    return true;
  }

  public boolean visit(IWhileStatement node) {
    ((Stack) loopStack.peek()).push(Boolean.TRUE);
    return true;
  }

  public boolean visit(IMethodDeclaration node) {
    loopStack.push(new Stack());
    Stack newStack = new Stack();
    boolean isSync = Modifier.isSynchronized(node.getModifiers());
    newStack.push(isSync ? Boolean.TRUE : Boolean.FALSE);
    synchronizedStack.push(newStack);
    return true;
  }

  public boolean visit(IInitializer node) {
    loopStack.push(new Stack());
    synchronizedStack.push(new Stack());
    return true;
  }

  public void endVisit(IMethodDeclaration node) {
    loopStack.pop();
    synchronizedStack.pop();
  }

  public void endVisit(IInitializer node) {
    loopStack.pop();
    synchronizedStack.pop();
  }

  private boolean insideLoop() {
    return !loopStack.isEmpty()
      && !((Stack) loopStack.peek()).isEmpty()
      && ((Boolean) ((Stack) loopStack.peek()).peek()).booleanValue();
  }

  private boolean insideSynchronized() {
    return !synchronizedStack.isEmpty()
      && !((Stack) synchronizedStack.peek()).isEmpty()
      && ((Boolean) ((Stack) synchronizedStack.peek()).peek()).booleanValue();
  }

}
