/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/serialization/SerializableTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.serialization;

import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Properties;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayType;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IType;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;
import com.sap.tc.jtools.jlint.jom.util.NameTool;
import com.sap.tc.jtools.jlint.jom.util.PrimitiveTypeTool;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;

/**
 * @author $Author: p4trans $
 *
 * This Jlin test checks some of the oddities when implementing
 * the empty java.io.Serializable interface:
 * 
 *  o  check if all non-static, non-transient and non-primitive fields
 *     of the implementing class implement java.io.Serializable
 *     themselves
 * 
 *  o  check if superclasses implement Serializable; if not, the
 *     first non-serializable superclass must have a zero-args
 *     constructor
 * 
 *  o  check if final long serialVersionUID is declared for
 *     all serializable classes 
 * 
 *  o  if private void writeObject(java.io.ObjectOutputStream out) is
 *     implemented, check that private void readObject(java.io.ObjectInputStream in) 
 *     is implemented, too and vice versa.
 * 
 *  o  read/writeObject() methods and declaration of 
 *     static  final ObjectStreamField[] serialPersistentFields
 *     are mutually exclusive
 */

public class SerializableTest extends JomTestVisitor {

  private static final String NAME = "Serialization";

  /** "magic" field names that influence the serialization process */
  private static final String SER_VER_UID_NAME = "serialVersionUID";
  private static final String SER_PERS_FIELDS_NAME = "serialPersistentFields";

  /** "magic" method names that influence the serialization process */
  private static final String WRITE_OBJECT_NAME = "writeObject";
  private static final String READ_OBJECT_NAME = "readObject";
  private static final String WRITE_REPLACE_NAME = "writeReplace";
  private static final String READ_RESOLVE_NAME = "readResolve";

  /** parameter types of read/writeObject methods */
  private static final String OUT_STREAM_NAME = "java.io.ObjectOutputStream";
  private static final String IN_STREAM_NAME = "java.io.ObjectInputStream";

  /** some fully qualified class names used */
  private static final String OBJECT_NAME = "java.lang.Object";
  private static final String SERIALIZABLE_NAME = "java.io.Serializable";
  private static final String EXTERNALIZABLE_NAME = "java.io.Externalizable";
  private static final String OBJ_STR_FIELD_NAME = "java.io.ObjectStreamField";

  /**
   * message keys and parameter names (see Messages.properties file for values)
   */
  private static final String MSG_KEY_1 = "serialization.1";
  private static final String MSG_KEY_2 = "serialization.2";
  private static final String MSG_KEY_3 = "serialization.3";
  private static final String MSG_KEY_4 = "serialization.4";
  private static final String MSG_KEY_5 = "serialization.5";

  private static final String PARAM_SUPERCLASS = "SUPER_CLASS";

  private static final String INPUT_PARAM_IGNORE_GUI_CLASSES= "IGNORE_GUI_CLASSES";
  private static final String INPUT_PARAM_CHECK_SERIALVERSION_UID = "CHECK_SERIALVERSION_UID";

  /**
   *  enumeration of attributes that are pushed on the 
   *  stack for each TypeDeclaration 
   */
  private class Attributes {
    boolean implementsSerializable = false;
    boolean serialVersionUIDDeclared = false;
    boolean serialPersistentFieldsDeclared = false;
    boolean writeObjectDeclared = false;
    boolean readObjectDeclared = false;
    boolean writeReplaceDeclared = false;
    boolean readResolveDeclared = false;
  }

  private Stack attributeStack = new Stack();
  private boolean ignoreGUIClasses = false;
  private boolean checkSerialVersionUID = false;

  private static final String AWT_PREFIX = "java.awt";
  private static final String SWING_PREFIX = "javax.swing";


  /**
   * @see com.sap.tc.jtools.jlint.jom.JomTestVisitor#getTestName()
   */
  public String getTestName() {
    return NAME;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.TestComponentInterface#setParameters(com.sap.tc.jtools.jtci.interfaces.ParameterInterface[], com.sap.tc.jtools.jtci.TestObject)
   */
  public void setParameters(ParameterInterface[] parameters,
      TestObject testObject) {
    super.setParameters(parameters, testObject);
    this.ignoreGUIClasses = ((Boolean)getInputParameter(INPUT_PARAM_IGNORE_GUI_CLASSES)).booleanValue();    
    this.checkSerialVersionUID= ((Boolean)getInputParameter(INPUT_PARAM_CHECK_SERIALVERSION_UID)).booleanValue();    
  }
  
  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(ITypeDeclaration)
   */
  public boolean visit(ITypeDeclaration node) {
    attributeStack.push(new Attributes());
    IReferenceTypeBinding bnd = (IReferenceTypeBinding) node.resolveBinding();
    if (bnd == null || !bnd.isClass())
      // visit classes only, no interfaces 
      return false;
    if (InheritanceTool.bndImplements(bnd, SERIALIZABLE_NAME)
      && !InheritanceTool.bndImplements(bnd, EXTERNALIZABLE_NAME)) {
      // only visit child nodes  of classes that 
      // implement java.io.Serializable, but not 
      // java.io.Externalizable (Externalizable has its own 
      // serialization mechanism -> we dont check that)
      if (ignoreGUIClasses && isGuiClass(bnd)) {
        return false;
      }
       ((Attributes) attributeStack.peek()).implementsSerializable = true;
      checkForMagicFields(node);
      checkForMagicMethods(node);
      checkSuperClasses(bnd, node);
      checkConsistency(node);
      return true;
    } else
      return false;
  }

  private static boolean isGuiClass(IReferenceTypeBinding bnd) {
    IReferenceTypeBinding currentBnd = bnd;
    String className = null;
    while (currentBnd.getSuperclass() != null) {
      className=currentBnd.getName();
      if (className.startsWith(AWT_PREFIX) ||
          className.startsWith(SWING_PREFIX)) {
        return true;
      }
      currentBnd = currentBnd.getSuperclass();
    }
    return false;
  }

  /**
   * Method checkConsistency.
   */
  private void checkConsistency(Position pos) {
    // read and writeObject methods must both be implemented (if any)
    Attributes attr = (Attributes) attributeStack.peek();
    if (!(attr.readObjectDeclared == attr.writeObjectDeclared)) {
      addError(MSG_KEY_1, null, pos);
    }
    if (attr.serialPersistentFieldsDeclared
      && (attr.readObjectDeclared || attr.writeObjectDeclared)) {
      addError(MSG_KEY_2, null, pos);
    }
  }

  /**
   * the first non-serializable superclass of a serializable class
   * MUST have a no-args constructor
   */
  private void checkSuperClasses(IReferenceTypeBinding bnd, Position pos) {
    if (InheritanceTool.bndImplements(bnd, SERIALIZABLE_NAME)) {
      // go up the inheritance tree
      checkSuperClasses(bnd.getSuperclass(), pos);
    } else {
      // check if there is a no-args constructor
      IMethodBinding[] methods = bnd.getDeclaredMethods();
      boolean constrFound = false;
      methLoop1 : for (int i = 0; i < methods.length; i++) {
        if (methods[i].isConstructor()) {
          constrFound = true;
          break methLoop1;
        }
      }
      // the default constructor is not in the AST, but it's a
      // no-args-constructor
      if (!constrFound)
        return;
      boolean noArgConstrFound = false;
      methLoop2 : for (int i = 0; i < methods.length; i++) {
        if (methods[i].isConstructor()
          && methods[i].getParameterTypes().length == 0) {
          noArgConstrFound = true;
          break methLoop2;
        }
      }
      if (!noArgConstrFound) {
        Properties p = new Properties();
        p.setProperty(PARAM_SUPERCLASS, NameTool.getFullClassName(bnd));
        addError(MSG_KEY_3, p, pos);
      }
    }
  }

  /**
   *  checks if any of the "magic" methods 
   * 
   *     private     void writeObject(java.io.ObjectOutputStream out)
   *     private     void readObject (java.io.ObjectInputStream  in)
   *     *any_modif* Object writeReplace()
   *     *any_modif* Object readResolve()
   * 
   *  are declared
   */
  private void checkForMagicMethods(ITypeDeclaration node) {
    Attributes attr = (Attributes) attributeStack.peek();
    IMethodDeclaration[] methods = node.getMethods();
    for (int i = 0; i < methods.length; i++) {
      String methName = methods[i].getName();
      List params = methods[i].parameters();
      IType returnType = methods[i].getReturnType();

      // writeReplace
      boolean writeReplDecl =
        WRITE_REPLACE_NAME.equals(methName)
          && params.size() == 0
          && OBJECT_NAME.equals(
            NameTool.getFullClassName(returnType.resolveBinding()));
      if (writeReplDecl) {
        attr.writeReplaceDeclared = true;
        continue;
      }

      // readResolve
      boolean readResDecl =
        READ_RESOLVE_NAME.equals(methName)
          && params.size() == 0
          && OBJECT_NAME.equals(
            NameTool.getFullClassName(returnType.resolveBinding()));
      if (readResDecl) {
        attr.readResolveDeclared = true;
        continue;
      }

      // writeObject
      boolean writeObjDecl = false;
      if (WRITE_OBJECT_NAME.equals(methName) && params.size() == 1) {
        IVariableDeclaration param = (IVariableDeclaration) params.get(0);
        writeObjDecl =
          (OUT_STREAM_NAME
            .equals(NameTool.getFullClassName(param.getType().resolveBinding()))
            && Modifier.isPrivate(methods[i].getModifiers())
            && returnType.isPrimitiveType()
            && NameTool.toDotNotation(
              ((ISimpleType) returnType).getName()).equals(
              ISimpleType.VOID));
      }
      if (writeObjDecl) {
        attr.writeObjectDeclared = true;
        continue;
      }

      // readObject
      boolean readObjDecl = false;
      if (READ_OBJECT_NAME.equals(methName) && params.size() == 1) {
        IVariableDeclaration param = (IVariableDeclaration) params.get(0);
        readObjDecl =
          (IN_STREAM_NAME
            .equals(NameTool.getFullClassName(param.getType().resolveBinding()))
            && Modifier.isPrivate(methods[i].getModifiers())
            && returnType.isPrimitiveType()
            && NameTool.toDotNotation(
              ((ISimpleType) returnType).getName()).equals(
              ISimpleType.VOID));
      }
      if (readObjDecl) {
        attr.readObjectDeclared = true;
      }
    }
  }

  /**
   * checks if static final long serialVersionUID
   * or        static final ObjectStreamField[] serialPersistentFields
   * are declared
   */
  private void checkForMagicFields(ITypeDeclaration node) {
    Attributes attr = (Attributes) attributeStack.peek();
    IFieldDeclaration[] decls = node.getFields();
    IType declType;
    for (int i = 0; i < decls.length; i++) {
      int mod = decls[i].getModifiers();
      if (!(Modifier.isStatic(mod) && Modifier.isFinal(mod)))
        continue;
      declType = decls[i].getType();

      //  serialversionUID
      if (!attr.serialVersionUIDDeclared) {
        attr.serialVersionUIDDeclared =
          decls[i].getName().equals(SER_VER_UID_NAME)
            && PrimitiveTypeTool.isLong(declType.resolveBinding());
      }
      // serialPersistentFields
      if (!attr.serialPersistentFieldsDeclared) {
        attr.serialPersistentFieldsDeclared =
          (SER_PERS_FIELDS_NAME.equals(decls[i].getName())
            && declType.isArrayType()
            && ((IArrayType) (declType)).getDimensions() == 1
            && OBJ_STR_FIELD_NAME.equals(
              NameTool.getFullClassName(
                ((IArrayTypeBinding) declType.resolveBinding())
                  .getElementType())));
      }
    }
  }

  /**
   * Method addSerialUIDError.
   */
  private void addSerialUIDWarning(Position pos) {
    addError(MSG_KEY_4, null, pos);
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IFieldDeclaration)
   */
  public boolean visit(IFieldDeclaration node) {
    Attributes attr = (Attributes) attributeStack.peek();
    // we dont check the fields if a customized serialization
    // process is used
    if (attr.serialPersistentFieldsDeclared
      || attr.writeObjectDeclared
      || attr.readObjectDeclared)
      return true;
    // check if non-transient, non-static, non-primitive 
    // fields implement Serializable
    int mod = node.getModifiers();
    if (Modifier.isStatic(mod) || Modifier.isTransient(mod))
      return true;
    IType type = node.getType();
    if (type.isPrimitiveType())
      return true;
    ITypeBinding bnd;
    if (type.isArrayType()) {
      bnd = ((IArrayType) type).getElementType();
      if (bnd.isPrimitive())
        return true;
    } else {
      bnd = type.resolveBinding();
      if (!InheritanceTool
        .bndImplements((IReferenceTypeBinding) bnd, SERIALIZABLE_NAME)) {
        addError(MSG_KEY_5, null, node);
      }
    }
    return true;
  }

  public void endVisit(ITypeDeclaration node) {
    Attributes attr = (Attributes) attributeStack.peek();
    if (checkSerialVersionUID && 
        attr.implementsSerializable &&
        !attr.serialVersionUIDDeclared)
      addSerialUIDWarning(node);
    attributeStack.pop();
  }

}
