/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/rc/RestrictedComponentsTest.java#3 $
 */

package com.sap.tc.jtools.jlint.tests.rc;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.Reader;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.xml.XMLTool;

public class RestrictedComponentsTest extends JomTestVisitor {

  static public final String TEST_NAME = "Restricted Components"; //$NON-NLS-1$

  static public final String PARAMETER_GRAMMAR_FILE = "GRAMMAR_FILE"; //$NON-NLS-1$
  static public final String PARAMETER_ONE_MSG_PER_CU = "ONE_MSG_PER_CU"; //$NON-NLS-1$
  static public final String MSG_KEY_FORBIDDEN_COMPONENT = "rc.1"; //$NON-NLS-1$

  static public final String ERROR_PARAMETER_CALLER = "CALLER"; //$NON-NLS-1$
  static public final String ERROR_PARAMETER_CALLEE = "CALLEE"; //$NON-NLS-1$
  static public final String ERROR_PARAMETER_COMPONENT = "COMPONENT"; //$NON-NLS-1$

  private Stack typeBndStack = new Stack();
  private Set componentNames = new HashSet();
  private Set alreadyCheckedTypes = new HashSet();

  private boolean isMainMethod;
  private boolean oneMsgPerCU = false;
  static private RestrictedComponents rc;

  static private File restrictedComponentsFile;
  static private long fileLastMod;

  /**
   * Jlin infrastructure
   */
  public RestrictedComponentsTest() {
  }

  public String getTestName() {
    return TEST_NAME;
  }

  /**
   * End of Jlin infrastructure
   */
  private void initialize() {
    try {
      oneMsgPerCU =
        ((Boolean) getInputParameter(PARAMETER_ONE_MSG_PER_CU)).booleanValue();
      File tempRestrictedComponentsFile =
        new File((String) getInputParameter(PARAMETER_GRAMMAR_FILE));
      long lastModified = tempRestrictedComponentsFile.lastModified();
      if (!tempRestrictedComponentsFile.equals(restrictedComponentsFile)
        || lastModified > fileLastMod) {
        fileLastMod = lastModified;
        restrictedComponentsFile = tempRestrictedComponentsFile;
        Reader fr = new BufferedReader(new FileReader(restrictedComponentsFile)); 
        rc = new RestrictedComponents(XMLTool.parseReader(fr));
        fr.close();
      }
    } catch (Exception e) {
      logException(e);
    }
  }

  public void setParameters(
    ParameterInterface[] parameters,
    com.sap.tc.jtools.jtci.TestObject testObject) {
    super.setParameters(parameters, testObject);
    initialize();
  }

  public boolean visit(ICompilationUnit compilationUnitDeclaration) {
    componentNames.clear();
    alreadyCheckedTypes.clear();
    return true;
  }

  public boolean visit(IConstructorInvocation consCall) {
    checkRoutine(consCall.resolveConstructorBinding(), consCall);
    return true;
  }

  public boolean visit(ISuperConstructorInvocation consCall) {
    checkRoutine(consCall.resolveConstructorBinding(), consCall);
    return true;
  }

  public boolean visit(IMethodInvocation methCall) {
    IMethodBinding methBind = methCall.resolveMethodBinding();
    if (isMainMethod) {
      // System.exit() is exceptionally allowed in main() methods 
      String name = methCall.getName();
      String callerName = methBind.getDeclaringClass().getName();
      if (name.equals("exit") && callerName.equals("java.lang.System")) {
        return true;
      }
    }
    checkRoutine(methBind, methCall);
    return true;
  }

  public boolean visit(ISuperMethodInvocation methCall) {
    checkType(methCall.resolveTypeBinding(), methCall);
    return true;
  }

  public boolean visit(ISimpleType st) {
    checkType(st.resolveBinding(), st);
    return true;
  }

  public boolean visit(IFieldAccess fa) {
    checkType(fa.resolveTypeBinding(), fa);
    checkField(fa.resolveBinding(), fa);
    return true;
  }

  private void checkField(IVariableBinding varBnd, Position pos) {
    IReferenceTypeBinding currentType =
      (IReferenceTypeBinding) typeBndStack.peek();
    ITypeBinding declClass = varBnd.getDeclaringClass();
    if (!(declClass instanceof IReferenceTypeBinding))
      return; // [].length
    if (((IReferenceTypeBinding) declClass)
      .getPackage()
      .equals(currentType.getPackage())) {
      return; // use within package -> always OK
    }
    RCStatusInfo statusInfo = rc.getStatusInfo(currentType, varBnd);
    if (!statusInfo.getStatus().equals(RCStatusInfo.STRING_STATUS_OK)) {
      error(pos, currentType, varBnd, statusInfo);
    }
  }

  public boolean visit(IClassInstanceCreation node) {
    checkType(node.resolveTypeBinding(), node);
    checkRoutine(node.resolveConstructorBinding(), node);
    return true;
  }

  public boolean visit(IArrayCreation node) {
    checkType(node.resolveTypeBinding(), node);
    return true;
  }

  public boolean visit(IQualifiedName qn) {
    //    String qnString = qn.getName();
    //    if (qnString != null && qnString.charAt(qnString.length() - 1) == '*')
    //      return true;
    try {
      ITypeBinding type = qn.resolveTypeBinding();
      if (type == null) // import declaration
        return true;
      checkType(type, qn);
    } catch (Exception e) {
      logException(e, qn);
    }
    return true;
  }

  public boolean visit(ISimpleName node) {
    ITypeBinding binding = node.resolveTypeBinding();
    if (binding != null) //it could be an import declaration
      checkType(binding, node);
    return true;
  }

  public boolean visit(ITypeDeclaration typeDeclaration) {
    typeBndStack.push(typeDeclaration.resolveBinding());
    return true;
  }

  public void endVisit(ITypeDeclaration typeDeclaration) {
    typeBndStack.pop();
  }

  private void error(
    Position position,
    IBinding caller,
    IBinding callee,
    RCStatusInfo status) {
    String component = status.getComponentName();
    // only one message per component and compilation unit
    if (oneMsgPerCU && componentNames.contains(component))
      return;
    componentNames.add(component);
    Properties errorParameters = new Properties();
    errorParameters.setProperty(
      ERROR_PARAMETER_CALLER,
      convertToHumanReadableSig(RestrictedComponents.getString(caller)));
    errorParameters.setProperty(
      ERROR_PARAMETER_CALLEE,
      convertToHumanReadableSig(RestrictedComponents.getString(callee)));
    errorParameters.setProperty(ERROR_PARAMETER_COMPONENT, component);
    addError(
      MSG_KEY_FORBIDDEN_COMPONENT,
      status.getReason(),
      errorParameters,
      status.getPriority(),
      position);
  }

  private void checkType(ITypeBinding type, Position position) {
    try {
      if (type.isArray()) {
        type = ((IArrayTypeBinding) type).getElementType();
      }
      if (type.isPrimitive())
        return;
      if (alreadyCheckedTypes.contains(type))
        return;
      IReferenceTypeBinding refType = (IReferenceTypeBinding) type;
      IReferenceTypeBinding currentType =
        (IReferenceTypeBinding) typeBndStack.peek();
      if (refType.getPackage() == null && currentType.getPackage() == null) {
        // default package uses default package --> OK
        return;
      }

      if (refType.getPackage() != null
        && refType.getPackage().equals(currentType.getPackage())) {
        alreadyCheckedTypes.add(type);
        return; // use within package -> always OK
      }
      //      String caller = currentType.getName();
      //      String[] callee = RestrictedComponents.getCallee(refType);
      RCStatusInfo statusInfo = rc.getStatusInfo(currentType, refType);
      if (!statusInfo.getStatus().equals(RCStatusInfo.STRING_STATUS_OK)) {
        error(position, currentType, refType, statusInfo);
      }
      alreadyCheckedTypes.add(type);
    } catch (Exception e) {
      logException(e, position);
    }
  }

  private void checkRoutine(IMethodBinding methBind, Position node) {
    if (alreadyCheckedTypes.contains(methBind))
      return;
    IReferenceTypeBinding calledClass = methBind.getDeclaringClass();
    IReferenceTypeBinding currentType =
      (IReferenceTypeBinding) typeBndStack.peek();
    if (calledClass.getPackage().equals(currentType.getPackage())) {
      alreadyCheckedTypes.add(methBind);
      return; // use within package -> always OK
    }
    //    String caller = currentType.getName();
    //    String[] callee = RestrictedComponents.getCallee(methBind);
    RCStatusInfo status = rc.getStatusInfo(currentType, methBind);
    if (!status.getStatus().equals(RCStatusInfo.STRING_STATUS_OK)) {
      error(node, currentType, methBind, status);
    }
    alreadyCheckedTypes.add(methBind);
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
   */
  public boolean visit(IMethodDeclaration node) {
    isMainMethod = node.getName().equals("main");
    return true;
  }

  private static String convertToHumanReadableSig(String internalSignature) {
    int separatorPos = internalSignature.indexOf('@');
    if (separatorPos == -1)
      return internalSignature;
    StringBuffer out = new StringBuffer(internalSignature.length());
    out.append(internalSignature.substring(0, separatorPos));
    out.append('(');
    String args =
      internalSignature.substring(separatorPos + 1).replace(';', ',');
    // remove trailing ","
    if (args.length() > 0) {
      args = args.substring(0, args.length() - 1);
    }
    out.append(args);
    out.append(')');
    return out.toString();
  }

}
