/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/rc/RestrictedComponents.java#3 $
 */

package com.sap.tc.jtools.jlint.tests.rc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

public class RestrictedComponents implements RCConstants {

  public static final String PACKAGE_ROOT_NAME = "";
  public static final String ATTR_DEFAULT_STATUS = "DEFAULT_STATUS";
  public static final String ATTR_DEFAULT_PRIO= "DEFAULT_PRIORITY";
  
  private RCPackage[] packages;

  private RCStatusInfo defaultStatusForRoot;
  private String defaultStatus;
  private int defaultPrio;

  public RestrictedComponents(StructureTree tree) {
    Header topHeader = tree.getHeader();
    this.defaultStatus = topHeader.getParameter(ATTR_DEFAULT_STATUS);
    if (defaultStatus == null) {
      defaultStatus = RCStatusInfo.STRING_STATUS_FORBIDDEN;   
    }
    String defaultPrioString = topHeader.getParameter(ATTR_DEFAULT_PRIO);
    if (defaultPrioString == null) {
      this.defaultPrio = ResultInterface.SEVERITY_ERROR;   
    } else {
      this.defaultPrio = Integer.parseInt(defaultPrioString.trim());   
    }
    this.defaultStatusForRoot = new RCStatusInfo("", this.defaultStatus, this.defaultPrio, null, Collections.EMPTY_LIST); 
    RCPackage pkgRoot =
      new RCPackage(
        "",
        null,
        null,
        defaultStatusForRoot);
    StructureTree[] pkgTrees = tree.getChildren(TAG_PKG);
    packages = new RCPackage[pkgTrees.length];
    for (int i = 0; i < pkgTrees.length; i++) {
      packages[i] = new RCPackage(pkgTrees[i], pkgRoot);
    }
  }

  /**
   * @param caller must be an instance of @see IPackageBinding, 
   *        @see IReferenceTypeBinding or @see IMethodBinding 
   * @param callee must be an instance of @see IPackageBinding, 
   *        @see IReferenceTypeBinding, @see IMethodBinding 
   *        or @see IVariableBinding that is a field 
   *        (@see IVariableBinding#isField() )
   * @return RCStatusInfo
   * @throws IllegalArgumentException if caller or callee are not 
   *         instances of the types specified above
   */
  public RCStatusInfo getStatusInfo(IBinding caller, IBinding callee) {
    // sanity check caller
    if (!((caller instanceof IPackageBinding)
      || (caller instanceof IReferenceTypeBinding)
      || (caller instanceof IMethodBinding)))
      throw new IllegalArgumentException(
        "illegal caller type: " + caller.getClass().getName());
    // sanity check callee
    if (!((callee instanceof IPackageBinding)
      || (callee instanceof IReferenceTypeBinding)
      || (callee instanceof IVariableBinding)
      || (callee instanceof IMethodBinding)))
      throw new IllegalArgumentException(
        "illegal callee type: " + callee.getClass().getName());
    if ((callee instanceof IVariableBinding)
      && !((IVariableBinding) callee).isField())
      throw new IllegalArgumentException(
        "callee must be a field: " + ((IVariableBinding) callee).getName());
    String callerString = getString(caller);
    String[] calleeStringArray = getCalleeArray(callee);
    if (calleeStringArray.length == 0) {
      return defaultStatusForRoot;
    }
    for (int i = 0; i < packages.length; i++) {
      if (calleeStringArray[0].equals(packages[i].getName())) {
        return packages[i].getStatusInfo(callerString, calleeStringArray, 1);
      }
    }
    // no explicit status => default 
      return new RCStatusInfo(
        getCalleeString(callee),
        this.defaultStatus,
        this.defaultPrio,
        null,
        Collections.EMPTY_LIST);
  }

  public static String getString(IBinding bnd) {
    String callerString = null;
    if ((bnd instanceof IPackageBinding)) {
      callerString = getCallerString((IPackageBinding) bnd);
    } else if ((bnd instanceof IReferenceTypeBinding)) {
      callerString = getCallerString((IReferenceTypeBinding) bnd);
    } else if ((bnd instanceof IMethodBinding)) {
      callerString = getCallerString((IMethodBinding) bnd);
    } else if ((bnd instanceof IVariableBinding)) {
      callerString = getCallerString((IVariableBinding) bnd);
    } else {
      throw new IllegalArgumentException();
    }
    return callerString;
  }

  private String getCalleeString(IBinding callee) {
    StringBuffer buf = new StringBuffer();
    String[] components = getCalleeArray(callee);
    for (int i = 0; i < components.length; i++) {
      buf.append(i == 0 ? components[i] : "." + components[i]);
    }
    return buf.toString();
  }

  private static String[] getCalleeArray(IBinding callee) {
    String[] calleeString = null;
    if ((callee instanceof IPackageBinding)) {
      calleeString = getCalleeArray((IPackageBinding) callee);
    } else if ((callee instanceof IReferenceTypeBinding)) {
      calleeString = getCalleeArray((IReferenceTypeBinding) callee);
    } else if ((callee instanceof IMethodBinding)) {
      calleeString = getCalleeArray((IMethodBinding) callee);
    } else if ((callee instanceof IVariableBinding)) {
      calleeString = getCalleeArray((IVariableBinding) callee);
    } else {
      throw new IllegalArgumentException();
    }
    return calleeString;
  }

  private static String[] getCalleeArray(IVariableBinding variable) {
    ITypeBinding declClass = variable.getDeclaringClass();
    if (!(declClass instanceof IReferenceTypeBinding))
      return new String[0];
    List classComponents = getCalleeList((IReferenceTypeBinding) declClass);
    classComponents.add(variable.getName());
    return (String[]) classComponents.toArray(new String[0]);
  }

  private static String[] getCalleeArray(IMethodBinding method) {
    StringBuffer methodKey = new StringBuffer(30);
    if (method.isConstructor()) {
      methodKey.append(method.getDeclaringClass().getIdentifier());
    } else {
      methodKey.append(method.getName());
    }
    methodKey.append(RCMethod.METHOD_SIGNATURE_SEPARATOR);
    ITypeBinding[] paramTypes = method.getParameterTypes();
    for (int k = 0; k < paramTypes.length; k++) {
      methodKey.append(paramTypes[k].getName());
      methodKey.append(RCMethod.PARAMETER_TYPE_SEPARATOR);
    }
    List components = getCalleeList(method.getDeclaringClass());
    components.add(methodKey.toString());
    return (String[]) components.toArray(new String[0]);
  }

  private static String[] getCalleeArray(IReferenceTypeBinding refType) {
    return (String[]) getCalleeList(refType).toArray(new String[0]);
  }

  private static String getCallerString(IReferenceTypeBinding refType) {
    return dotConcat(getCalleeArray(refType));
  }

  private static String getCallerString(IMethodBinding meth) {
    return dotConcat(getCalleeArray(meth));
  }

  private static String getCallerString(IVariableBinding var) {
    return dotConcat(getCalleeArray(var));
  }

  private static String getCallerString(IPackageBinding pkgBnd) {
    return dotConcat(getCalleeArray(pkgBnd));
  }

  private static String[] getCalleeArray(IPackageBinding pkgBnd) {
    return pkgBnd.getNameComponents();
  }

  private static String dotConcat(String[] string) {
    StringBuffer tmp = new StringBuffer();
    for (int i = 0; i < string.length; i++) {
      tmp.append((i == 0 ? "" : ".") + string[i]);
    }
    return tmp.toString();
  }

  private static List getCalleeList(IReferenceTypeBinding refType) {
    List tmpList = new ArrayList();
    IPackageBinding pkgBnd = refType.getPackage();
    if (pkgBnd != null) {
      tmpList.addAll(Arrays.asList(pkgBnd.getNameComponents()));
    }
    List classList = new ArrayList();
    classList.add(refType.getIdentifier());
    IReferenceTypeBinding currentBinding = refType;
    while (currentBinding.getDeclaringClass() != null) {
      classList.add(currentBinding.getDeclaringClass().getIdentifier());
      currentBinding = currentBinding.getDeclaringClass();
    }
    Collections.reverse(classList);
    tmpList.addAll(classList);
    return tmpList;
  }

}