/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/platform/PlatformDependenceTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.platform;

import java.util.List;
import java.util.Properties;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayType;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IBlock;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IStringLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITryStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;

// commented out because regexp lib is not available in build /Jan
//import org.apache.regexp.RE;
//import org.apache.regexp.RESyntaxException;


/**
 * 
 * This JLin test checks (potential) platform-dependent Java code.
 * Overlapping: Features checked by Restricted Components test are also included.
 * 
 * @author d033025
 */


public class PlatformDependenceTest extends JomTestVisitor {
	
	private static final String NAME = "Platform Dependence Test";
	
	private static final String SEPARATOR = ".";
	
	/*
	 * File constructor test
	 * checks if File() is used with the first argment not a "File", i.e., 
	 * File(String) or File(String, String) is used instead of File(File, String).
	 * This test is already covered by checks of OS-specific paths.
	 * 
	private static final String IOFILE_WARNING = 
		"String path may cause problem, use abstract path or " +
		"system properties or file dialog instead";
	private static final String MSG_KEY_IOFILE = "IOFileConstructor";
	private static final String IOFILE = "java.io.File";
	 *
	 */

	
	/*
	 * JDBC test
	 * checks if Class.forName(driver) is used with "driver" 
	 * a String literal field or an initialized String variale;
	 * checks if DriverManager.getConnection(url, ...) is used 
	 * with "url" a String literal field or an initialized String variable
	 */
	private static final String DRIVER_NAME_WARNING = 
		"Hard-coded JDBC driver name may cause problem, " +
		"make driver name configurable by using system property or property file";
	private static final String DATABASE_URL_WARNING = 
		"Hard-coded database URL may cause problem";
	private static final String MSG_KEY_JDBC = "JDBCDependence";
	private static final String LOAD_JDBC_DRIVER = "java.lang.Class.forName";
	private static final String GET_DB_CONNECTION = "java.sql.DriverManager.getConnection";
	
	/*
	 * Platform-dependent literal test
	 * checks if System.getProperty(key, ...) is used with "key"
	 * a String literal field or an initialized String variable with value like "os.name"
	 * or directly a String like "java.version"; 
	 * checks if OS-specific libs (like *.dll) and OS-specific paths (like C:\) are used
	 */
	private static final String PATH_WARNING = 
		"OS-specific path may cause problem";
	private static final String LIB_WARNING = 
		"OS-specific libaray may cause problem";
	private static final String PROPERTY_QUERY_WARNING = 
		"Getting platform-specific property may cause problem";
	private static final String MSG_KEY_LITERAL = "PlatformDependentLiteral";
	private static final String PLATFORM_DEPENDENT_PATH = "\\b([a-zA-Z]{1}):(\\\\|\\|//|/)|" +
														"\\\\localhost|//localhost|//http:";
	private static final String PLATFORM_DEPENDENT_LIB = "(\\.dll|\\.so|\\.o|\\.sl)\\b";
	private static final String PLATFORM_PROPERTY_QUERY = "java.lang.System.getProperty";
	private static final String[] PLATFORM_PROPERTY_KEY = {
			"java.version",
			"java.vendor",
			"java.vendor.url",
			"java.vm.specification.version",
			"java.vm.specification.vendor",
			"java.vm.specification.name",
			"java.vm.version",
			"java.vm.vendor",
			"java.vm.name",
			"java.specification.version", 
			"java.specification.vendor",
			"java.specification.name",
			"java.class.version",
			"os.name",
			"os.arch",
			"os.version" };
	private static final int BAD_PATH = 1;
	private static final int BAD_LIB = 2;
	private static final int BAD_KEY = 3;
	
	/*
	 * Standard I/O stream test
	 * checks if System.in, System.out, and System.err are used
	 */
	private static final String STREAM_WARNING = 
		"Standard I/O stream may cause problem, use Logging if possible";
	private static final String MSG_KEY_STREAM = "StandardIOStream";
	private static final String IN_STREAM = "java.lang.System.in";
	private static final String OUT_STREAM = "java.lang.System.out";
	private static final String ERR_STREAM = "java.lang.System.err";
	private static final int BAD_IN = 4;
	private static final int BAD_OUT = 5;
	private static final int BAD_ERR = 6;
		
	/*
	 * Forbidden API test
	 * checks if forbidden JDK packages, classes, or methods are used
	 */
	private static final String PACKAGE_WARNING = 
		"Using this JDK package may cause problem";
	private static final String CLASS_WARNING = 
		"Using this JDK class may cause problem";
	private static final String METHOD_WARNING = 
		"Using this JDK method may cause problem";
	private static final String MSG_KEY_API = "ForbiddenJavaAPI";
	private static final String[] FORBIDDEN_PKGS = { 
		"com.sun",
		"sun",
		"org.omg.CORBA",
		"org.omg.CORBA_2_3"};
	private static final String[] FORBIDDEN_CLASSES = {};
	private static final String[] FORBIDDEN_METHODS = {
		"java.lang.Thread.setPriority",
		"java.lang.System.load",
		"java.lang.System.loadLibrary",
		"java.lang.Runtime.load",
		"java.lang.Runtime.loadLibrary",
		"java.lang.Runtime.exec",
		"java.io.InputStream.read",
		"java.io.OutputStream.write", 
		"java.sql.Connection.nativeSQL" };
	
	/*
	 * Types of features checked
	 */
	private static final String CHECKED_FEATURE_JDK_API = "JDK_API";
	private static final String CHECKED_FEATURE_METHOD = "METHOD";
	private static final String CHECKED_FEATURE_CONSTRUCTOR = "CONSTRUCTOR";
	private static final String CHECKED_FEATURE_PARAMETER = "PARAMETER";
	private static final String CHECKED_FEATURE_FIELD = "FIELD";
	private static final String CHECKED_FEATURE_VARIABLE = "VARIABLE";
	private static final String CHECKED_FEATURE_VALUE = "VALUE";
	
	/*
	 * Warning/error messages
	 */ 
	private static final String ERROR_PARAMETER_REASON = "REASON";

	
	
	public String getTestName() {
		return NAME;
	}
	
	
	public boolean visit(ITypeDeclaration type) {
		IFieldDeclaration[] fieldList;
		IFieldDeclaration field;
		IMethodDeclaration[] methodList;
		IMethodDeclaration method;
		String fieldName, fieldValue;
		List stmtList, stmtBody;;
		IStatement stmt, cmd;
		IExpression init;
		
		fieldList = type.getFields();
		methodList = type.getMethods();
		fieldName = null;
		fieldValue = null;
		
		// check each String literal field to see if it is an OS-specific lib or an OS-specific path
		// or is used to get a platform-dependent property or to load JDBC driver or to connect DB
		for (int i = 0; i <= fieldList.length-1; i++) {
			field = fieldList[i];
			init = field.getInitializer();
			if (init != null) {
				if (init instanceof IStringLiteral) {
					fieldName = field.getName();
					fieldValue = ((IStringLiteral)init).getLiteralValue();
					
					checkPath(CHECKED_FEATURE_FIELD, fieldName, fieldValue, field);
					checkLib(CHECKED_FEATURE_FIELD, fieldName, fieldValue, field);
					
					// check the use of this String literal field as parameter in each method call
					if (methodList.length >= 1) {
						for (int j = 0; j <= methodList.length-1; j++) {
							method = methodList[j];
							if (method.getBody() != null) {
								stmtList = method.getBody().statements();
								for (int k = 0; k <= stmtList.size()-1; k++) {
									stmt = (IStatement) stmtList.get(k);
									if (isBlockStatement(stmt)) {
										stmtBody = getStatementBody(stmt);
										if (stmtBody != null) 
											for (int m = 0; m <= stmtBody.size()-1; m++) {			
												cmd = (IStatement) stmtBody.get(m);
												checkMethodParameter(cmd, fieldName, fieldValue);
											} 
									}
									else
										checkMethodParameter(stmt, fieldName, fieldValue);					
								}
							}
						}
					}			
						
				}
			}		
		}	
		return true;
	}
	

	public boolean visit(IMethodDeclaration method) {
		IBlock mBody;
		List stmtList;
		IStatement stmt;
		String varName, varValue;
		IExpression init, leftExp, rightExp;
	
		mBody = method.getBody();
		if (mBody != null) {
			stmtList = method.getBody().statements();
		
			// check each variable decalration to see if it is initialized with a String literal;
			// if so, check if this String literal is an OS-specific lib or an OS-specific path
			// or is used to get a platform-dependent property or to load JDBC driver or to connect DB
			for (int i = 0; i <= stmtList.size()-1; i++) {
				stmt = (IStatement) stmtList.get(i);
				if (stmt instanceof IVariableDeclarationStatement) {
					varName = ((IVariableDeclarationStatement)stmt).getName();
					init = ((IVariableDeclarationStatement)stmt).getInitializer();
					if (init != null) {
						// check dubious variable initialization
						if (init instanceof IStringLiteral) {
							varValue = ((IStringLiteral)init).getLiteralValue();						
						
							checkPath(CHECKED_FEATURE_VARIABLE, varName, varValue, stmt);
							checkLib(CHECKED_FEATURE_VARIABLE, varName, varValue, stmt);	
			
							// check the use of this variable as parameter in each method call
							for (int j = i+1; j <= stmtList.size()-1; j++) {
								stmt = (IStatement) stmtList.get(j);
								checkMethodParameter(stmt, varName, varValue);												
							}
						}			
					}
					else {
					// check dubious variable assignment
						for (int j = i+1; j <= stmtList.size()-1; j++) {
							stmt = (IStatement) stmtList.get(j);
							if (stmt instanceof IAssignment) {
								leftExp = ((IAssignment)stmt).getLeftHandSide();
								rightExp = ((IAssignment)stmt).getRightHandSide();
								if (leftExp instanceof IVariableBinding)
									if (!((IVariableBinding)leftExp).isField() && 
									((IVariableBinding)leftExp).getName().equals(varName) &&
									rightExp instanceof IStringLiteral) {
										varValue = ((IStringLiteral)rightExp).getLiteralValue();
									
										checkPath(CHECKED_FEATURE_VARIABLE, varName, varValue, stmt);
										checkLib(CHECKED_FEATURE_VARIABLE, varName, varValue, stmt);	

										// check the use of this variable as parameter in each method call
										for (int k = j+1; k <= stmtList.size()-1; k++) {
											stmt = (IStatement) stmtList.get(k);
											checkMethodParameter(stmt, varName, varValue);												
										}
									}								
							}
						}			
					}			
				}			
			}
		}
		return true;
	}


	public boolean visit(IMethodInvocation mCall) {
		IMethodBinding mBind;
		List argList;
		IExpression arg, firstArg;
		String fullMethodName, fullMethodSignature, argValue;
				
		mBind = mCall.resolveMethodBinding();
		fullMethodName = getFullMethodName(mBind);
		fullMethodSignature = getFullMethodSignature(mBind);
		argList = mCall.arguments();
		
		// check method call with hard-coded parameter value 			
		for (int i = 0; i <= argList.size()-1; i++) {
			arg = (IExpression) argList.get(i);
			if (arg instanceof IStringLiteral) {
				argValue = ((IStringLiteral)arg).getLiteralValue();
						
				checkPath(CHECKED_FEATURE_METHOD, fullMethodSignature, argValue, mCall);
				checkLib(CHECKED_FEATURE_METHOD, fullMethodSignature, argValue, mCall);	
				checkKey(fullMethodName, fullMethodSignature, argValue, mCall);	
			}
		}
		
		
		// check if forbidden methods are used
		checkForbiddenMethods(fullMethodName, mCall);
		
		return true;
	}
	
	
	public boolean visit(IClassInstanceCreation con) {
		IMethodBinding mBind;
		String fullConstructorName, fullConstructorSignature, firstArgType;
		ITypeBinding[] argTypes;
		List argList;
		IExpression arg;
		String argValue;
		
		mBind = con.resolveConstructorBinding();
		fullConstructorName = getFullConstructorName(mBind);
		fullConstructorSignature = getFullConstructorSignature(mBind);
					
		// check if the constructor File() is used with the first parameter not a "File"
		/*
		argTypes = mBind.getParameterTypes();
		if (argTypes.length > 0) {
			firstArgType = argTypes[0].getName();
			if (IOFILE.equals(fullConstructorName) && !IOFILE.equals(firstArgType))
				createError(CHECKED_FEATURE_CONSTRUCTOR, fullConstructorSignature, null, 
				IOFILE_WARNING, MSG_KEY_IOFILE, con);
		}
		*/
				
		// check constructor call with hard-coded parameter value
		argList = con.arguments();			
		for (int i = 0; i <= argList.size()-1; i++) {
			arg = (IExpression) argList.get(i);
			if (arg instanceof IStringLiteral) {
				argValue = ((IStringLiteral)arg).getLiteralValue();
				checkPath(CHECKED_FEATURE_CONSTRUCTOR, fullConstructorSignature, argValue, con);
				checkLib(CHECKED_FEATURE_CONSTRUCTOR, fullConstructorSignature, argValue, con);		
			}
		}
		return true;
	}
	

	public boolean visit(IFieldAccess field) {
		IExpression prefix;
		ITypeBinding fBind;
		String fullFieldName;
		int badValue;
						
		// check if standard I/O streams are used
		prefix = field.getExpression();
		fBind = prefix.resolveTypeBinding();		
		if (fBind != null) {
			fullFieldName = fBind.getName() + SEPARATOR + field.getName();			
			checkIO(fullFieldName, null, field);
		}
		return true;
	}	
	
	
	public boolean visit(IArrayType node) {
	  checkForbiddenComponents(node.resolveBinding(), node);
	  return true;
	}

 
	public boolean visit(ISimpleType node) {
		checkForbiddenComponents(node.resolveBinding(), node);
		return true;
	}
	
	
	public boolean visit(IQualifiedName node) {
	  checkForbiddenComponents(node.resolveTypeBinding(), node);
	  return true;
	}

 
	public boolean visit(ISimpleName node) {
	  checkForbiddenComponents(node.resolveTypeBinding(), node);
	  return true;
	}



	private void checkForbiddenComponents(ITypeBinding bind, Position node) {
		if (bind == null)
			return;
		checkForbiddenPackages(bind, node);
		checkForbiddenClasses(bind, node);
	}


	private void checkForbiddenPackages(ITypeBinding bind, Position node) {
		ITypeBinding elementBinding = bind;
		if (bind.isArray())
			elementBinding = ((IArrayTypeBinding) bind).getElementType();
		if (elementBinding.isPrimitive())
			return;
		for (int i = 0; i <= FORBIDDEN_PKGS.length-1; i++)
			if (getFullPackageName((IReferenceTypeBinding) elementBinding).startsWith(FORBIDDEN_PKGS[i]))
				createError(CHECKED_FEATURE_JDK_API, FORBIDDEN_PKGS[i], null, 
				PACKAGE_WARNING, MSG_KEY_API, node);
	}


	private void checkForbiddenClasses(ITypeBinding bind, Position node) {
		ITypeBinding elementBinding = bind;
		if (bind.isArray())
			elementBinding = ((IArrayTypeBinding) bind).getElementType();
		if (elementBinding.isPrimitive())
			return;
		for (int i = 0; i <= FORBIDDEN_CLASSES.length-1; i++)
			checkClassRecursive( FORBIDDEN_CLASSES[i], (IReferenceTypeBinding) elementBinding, node);
	}


	private void checkClassRecursive(String forbiddenName, IReferenceTypeBinding bind, Position node) {
		if (bindExtends(bind, forbiddenName) || bindImplements(bind, forbiddenName))
			createError(CHECKED_FEATURE_JDK_API, forbiddenName, null, 
			CLASS_WARNING, MSG_KEY_API, node);
	}
	
	
	private void checkForbiddenMethods(String methodName, Position node) {
		for (int i = 0; i <= FORBIDDEN_METHODS.length-1; i++)
			if (methodName.equals(FORBIDDEN_METHODS[i]))
				createError(CHECKED_FEATURE_JDK_API, FORBIDDEN_METHODS[i], null, 
				METHOD_WARNING, MSG_KEY_API, node);
	}
	
	
	/* 
	 * check the use of OS-specific path
	 */
	private void checkPath(String featureType, String feeatureName, String featureValue, Position node) {
		int badValue = checkRegex(featureValue);
		if (badValue == BAD_PATH)
			createError(featureType, feeatureName, featureValue, 
			PATH_WARNING, MSG_KEY_LITERAL, node);
	}
	
	
	/*
	 * check the use of OS-specific lib
	 */
	private void checkLib(String featureType, String feeatureName, String featureValue, Position node) {
		int badValue = checkRegex(featureValue);
		if (badValue == BAD_LIB)
			createError(featureType, feeatureName, featureValue,
			LIB_WARNING, MSG_KEY_LITERAL, node);
	}
	
	
	/*
	 * check the use of standard I/O stream
	 */
	private void checkIO(String fieldName, String fieldValue, Position field) {
		int badValue = checkRegex(fieldName);
		if (badValue == BAD_IN || badValue == BAD_OUT || badValue == BAD_ERR)
			createError(CHECKED_FEATURE_FIELD, fieldName, fieldValue, 
			STREAM_WARNING, MSG_KEY_STREAM, field);
	}
	
		
	/*
	 * check the use of dubious method parameter, such as constant JDBC driver or system property "os.name"
	 */
	private void checkMethodParameter(IStatement stmt, String varName, String varValue) {
		int badValue = checkRegex(varValue);
		if (badValue == BAD_KEY)
			checkProperty(stmt, varName, varValue);
		else
			checkJDBC(stmt, varName);
	}	
	
	
	/*
	 * check method call to get system property with certain key such as "os.name" as parameter
	 */
	private void checkProperty(IStatement stmt, String varName, String varValue) {
		IMethodBinding mBind;
		List argList;
		String fullMethodName, fullMethodSignature, firstArgName;
		IExpression exp, rightExp, init, firstArg;
		
		if (stmt instanceof IExpressionStatement) {
			exp = ((IExpressionStatement)stmt).getExpression();
			if (exp instanceof IAssignment) {					
				rightExp = ((IAssignment)exp).getRightHandSide();
				if (rightExp instanceof IMethodInvocation) {
					mBind = ((IMethodInvocation)rightExp).resolveMethodBinding();
					fullMethodName = getFullMethodName(mBind);
					fullMethodSignature = getFullMethodSignature(mBind);
					argList = ((IMethodInvocation)rightExp).arguments();
					if (!argList.isEmpty()) {
						firstArg = (IExpression) argList.get(0);
						firstArgName = ((ISimpleName)firstArg).getIdentifier();
						if (firstArgName.equals(varName))
							checkKey(fullMethodName, fullMethodSignature, varValue, stmt);		
					}
				}
			}
		}
		if (stmt instanceof IVariableDeclarationStatement) {
			init = ((IVariableDeclarationStatement)stmt).getInitializer();
			if (init != null) {
				if (init instanceof IMethodInvocation) {					
					mBind = ((IMethodInvocation)init).resolveMethodBinding();
					fullMethodName = getFullMethodName(mBind);
					fullMethodSignature = getFullMethodSignature(mBind);
					argList = ((IMethodInvocation)init).arguments();
					if (!argList.isEmpty()) {
						firstArg = (IExpression) argList.get(0);
						if (!(firstArg instanceof IStringLiteral)) {
							firstArgName = ((ISimpleName)firstArg).getIdentifier();
							if (firstArgName.equals(varName))
								checkKey(fullMethodName, fullMethodSignature, varValue, stmt);			 						
						}
					}	
				}
			}
		}				
	}
	
	
	/*
	 * check the use of certain key such as "os.name" in java.lang.System.getProperty()
	 */
	private void checkKey(String fullMethodName, String fullMethodSignature, String argValue, Position node) {
		int badValue = checkRegex(argValue);
		if (badValue == BAD_KEY)
			if (fullMethodName.equals(PLATFORM_PROPERTY_QUERY))
			createError(CHECKED_FEATURE_METHOD, fullMethodSignature, argValue, 
			PROPERTY_QUERY_WARNING, MSG_KEY_LITERAL, node);
	}
	
	
	/*
	 * check method call to load JDBC driver or establish DB connection
	 */
	private void checkJDBC(IStatement stmt, String varName) {
		IMethodBinding mBind;
		String fullMethodName, fullMethodSignature, firstArgName, firstArgValue;
		List argList;
		IExpression exp, rightExp, firstArg;
		
		if (stmt instanceof IExpressionStatement) {
			exp = ((IExpressionStatement)stmt).getExpression();
			if (exp instanceof IAssignment) {
				rightExp = ((IAssignment)exp).getRightHandSide();
				if (rightExp instanceof IMethodInvocation) {									
					mBind = ((IMethodInvocation)rightExp).resolveMethodBinding();
					fullMethodName = getFullMethodName(mBind);
					fullMethodSignature = getFullMethodSignature(mBind);
					if (fullMethodName.equals(GET_DB_CONNECTION)) {		
						argList = ((IMethodInvocation)rightExp).arguments();
						firstArg = (IExpression) argList.get(0);
						if (firstArg instanceof ISimpleName) {
							firstArgName = ((ISimpleName)firstArg).getIdentifier();			
							if (firstArg instanceof IStringLiteral || firstArgName.equals(varName))
								createError(CHECKED_FEATURE_METHOD, fullMethodSignature, firstArgName, 
								DATABASE_URL_WARNING, MSG_KEY_JDBC, rightExp);
						}
					}
				}
			} 
			if (exp instanceof IMethodInvocation) {									
				mBind = ((IMethodInvocation)exp).resolveMethodBinding();
				fullMethodName = getFullMethodName(mBind);
				fullMethodSignature = getFullMethodSignature(mBind);		
				if (fullMethodName.equals(LOAD_JDBC_DRIVER)) {
					argList = ((IMethodInvocation)exp).arguments();
					firstArg = (IExpression) argList.get(0);
					if (firstArg instanceof ISimpleName) {
						firstArgName = ((ISimpleName)firstArg).getIdentifier();	 
						if (firstArg instanceof IStringLiteral || firstArgName.equals(varName))
							createError(CHECKED_FEATURE_METHOD, fullMethodSignature, firstArgName, 
							DRIVER_NAME_WARNING, MSG_KEY_JDBC, exp);
					}
				}
			}
		}
	}


	/*
	 * check regular expressions for special patterns,
	 * using java.util.regexp (JDK1.4)
	 *
	private int checkRegex(String input) {
		Pattern pathP, libP, keyP, inP, outP, errP;
		Matcher pathM, libM, keyM, inM, outM, errM;
		int result = 0;
		
		pathP = Pattern.compile(PLATFORM_DEPENDENT_PATH);
		libP = Pattern.compile(PLATFORM_DEPENDENT_LIB);
		inP = Pattern.compile(IN_STREAM);
		outP = Pattern.compile(OUT_STREAM);
		errP = Pattern.compile(ERR_STREAM);
		pathM = pathP.matcher(input);
		libM = libP.matcher(input);
		inM = inP.matcher(input);
		outM = outP.matcher(input);
		errM = errP.matcher(input);
		if (pathM.find())
			result = BAD_PATH;
		else if (libM.find())
			result = BAD_LIB;
		else if (inM.find())
			result = BAD_IN;
		else if (outM.find())
			result = BAD_OUT;
		else if (errM.find())
			result = BAD_ERR;
		else for (int i = 0; i <= PLATFORM_PROPERTY_KEY.length-1; i++) {
			keyP = Pattern.compile(PLATFORM_PROPERTY_KEY[i]);
			keyM = keyP.matcher(input);
			if (keyM.find())
				result = BAD_KEY;			
			}
		return result;				
	}
	*
	*/			
	
	/*
 	 * check regular expressions for special patterns,
     * using org.apache.regexp
     */
 	private int checkRegex(String input) {
//		RE pathP = null, libP = null, keyP = null, inP = null, outP = null, errP = null;		
//		int result = -1;
//		
//		try {
//			pathP = new RE(PLATFORM_DEPENDENT_PATH);
//			libP = new RE(PLATFORM_DEPENDENT_LIB);
//			inP = new RE(IN_STREAM);
//			outP = new RE(OUT_STREAM);
//			errP = new RE(ERR_STREAM);
//		}
//		catch (RESyntaxException e) {
//			result = 0;
//		}
//
//		if (result != 0 ) {
//			if (pathP.match(input))
//				result = BAD_PATH; 
//			else if (libP.match(input))
//				result = BAD_LIB;
//			else if (inP.match(input))
//				result = BAD_IN;
//			else if (outP.match(input))
//				result = BAD_OUT;
//			else if (errP.match(input))
//				result = BAD_ERR;
//			else for (int i = 0; i <= PLATFORM_PROPERTY_KEY.length-1; i++) {
//				try {
//					keyP = new RE(PLATFORM_PROPERTY_KEY[i]);
//				}
//				catch (RESyntaxException e) {
//					result = 0;
//				}
//				if (result != 0 )
//					if (keyP.match(input))
//						result = BAD_KEY;		
//			}
//		}
    int result = -1;		
		return result;
	}

	
	/*
	 * get qualified method name
	 */
	private String getFullMethodName(IMethodBinding mBind) {
		IReferenceTypeBinding declaringClass = mBind.getDeclaringClass();
		String fullMethodName = declaringClass.getName() + SEPARATOR + mBind.getName();
		return fullMethodName;
	}
	
	
	/*
	 * get qualified constructor name
	 */
	private String getFullConstructorName(IMethodBinding mBind) {
		IReferenceTypeBinding declaringClass = mBind.getDeclaringClass();
		String fullMethodName = declaringClass.getName();
		return fullMethodName;
	}
	
	 	 
	/*
	 * get all parameter types of a method
	 */
	private String mergeMethodSignature(IMethodBinding mBind) {		
		String signature = "(";
		ITypeBinding[] argTypes = mBind.getParameterTypes();
		for (int i = 0; i < argTypes.length; i++) {
			signature = signature + argTypes[i].getName();
			if  (i < (argTypes.length-1))
				signature = signature + ", ";
		}
		signature = signature + ")";
		return signature;
	}
	
	
	/*
	 * get qualified method name with all parameter types
	 */
	private String getFullMethodSignature(IMethodBinding mBind) {		
		return getFullMethodName(mBind) + mergeMethodSignature(mBind);
	}
	
	
	/*
	 * get qualified constructor name with all parameter types
	 */
	private String getFullConstructorSignature(IMethodBinding mBind) {		
		return getFullConstructorName(mBind) + mergeMethodSignature(mBind);
	}
	
	
	/*
	 * get complete package name (with names of all parent packages)
	 */
	private static String getFullPackageName(IReferenceTypeBinding bind) {
		IPackageBinding pBind;
		String[] comp;
		StringBuffer buf;
		String pName;
	  	
		pName = "";
		pBind = bind.getPackage();
		if (pBind != null) {
			comp = pBind.getNameComponents();
			buf = new StringBuffer(comp.length);
			for (int i = 0; i < comp.length; i++)
				buf.append(((i > 0) ? "." : "") + comp[i]);
			pName = buf.toString();	
		}
		return pName;
	}
		
	
	/*
	 * recursively check the inheritance tree of a type binding
	 * to see if it implements an interface
	 */
	private static boolean bindImplements(IReferenceTypeBinding bind, String interfaceName) {
		if (bind == null || interfaceName == null)
			return false;
		if (bind.isInterface() && interfaceName.equals(bind.getName()))
			return true;
	  
		// recursively check all interfaces up the inheritance tree
		IReferenceTypeBinding[] intfs = bind.getInterfaces();
		if (intfs != null)
			for (int i = 0; i < intfs.length; i++)
				if (bindImplements(intfs[i], interfaceName))
					return true;
		//	recursively check super classes up the inheritance tree
		if (bind.getSuperclass() != null)
			return bindImplements(bind.getSuperclass(), interfaceName);
			
		return false;
	}

	
	/*
	 * recursively check the inheritance tree of a type binding
	 * to see if it extends a class
	 */
	private static boolean bindExtends(IReferenceTypeBinding bind, String superClassName) {
		if (bind == null || superClassName == null)
			return false;
		if (bind.isClass() && superClassName.equals(bind.getName()))
			return true;
	  
		// recursively check super classes up the inheritance tree
		if (bind.getSuperclass() != null)
			return bindExtends(bind.getSuperclass(), superClassName);
			
		return false;
	}

	
	/*
	 * check if a statement is a block statement
	 */
	private boolean isBlockStatement(IStatement stmt) {
		boolean result = false;
		if (stmt instanceof ITryStatement) result = true;
		if (stmt instanceof IForStatement) result = true;
		if (stmt instanceof IDoStatement) result = true;
		if (stmt instanceof IWhileStatement) result = true;
		if (stmt instanceof IIfStatement) result = true;
		if (stmt instanceof ISwitchStatement) result = true;
		if (stmt instanceof ICatchClause) result = true;
		return result;
	}
	
	
	/*
	 * get the body (list of statements) of a block statement
	 */
	private List getStatementBody(IStatement stmt) {
		IBlock stmtBlock = null;
		List stmtBody = null;
		List elseBody;
		
		if (stmt instanceof IIfStatement) {
			stmtBody = getStatementBody(((IIfStatement)stmt).getThenStatement());
			elseBody = getStatementBody(((IIfStatement)stmt).getElseStatement());
			if (elseBody != null)
				stmtBody.addAll(elseBody);
		} 
		else if (stmt instanceof ISwitchStatement)
			stmtBody = ((ISwitchStatement)stmt).statements();
		else {
			if (stmt instanceof ITryStatement) {
				stmtBlock = ((ITryStatement)stmt).getBody();
				if (stmtBlock != null)			
					stmtBody = stmtBlock.statements();
			}
			if (stmt instanceof ICatchClause) {
				stmtBlock = (IBlock) ((ICatchClause)stmt).getBody();
				if (stmtBlock != null)			
					stmtBody = stmtBlock.statements();
			}	  
			if (stmt instanceof IForStatement)
				stmtBody = getStatementBody(((IForStatement)stmt).getBody());
			if (stmt instanceof IDoStatement)
				stmtBody = getStatementBody(((IDoStatement)stmt).getBody());
			if (stmt instanceof IWhileStatement)
				stmtBody = getStatementBody(((IWhileStatement)stmt).getBody());
		}
		return stmtBody;
	}
	
	
	/*
	 * build check result message
	 */
	private void createError(
		String checkedFeatureType,
		String checkedFeatureName,
		String checkedFeatureValue,
		String message,
		String messageKey,
		Position position) {

		Properties errorParameters = new Properties();		
		errorParameters.setProperty(checkedFeatureType, checkedFeatureName);
		if (checkedFeatureValue != null)
			if (checkedFeatureType.equals(CHECKED_FEATURE_FIELD) || 
				checkedFeatureType.equals(CHECKED_FEATURE_VARIABLE))
				errorParameters.setProperty(CHECKED_FEATURE_VALUE, checkedFeatureValue);
			else
				errorParameters.setProperty(CHECKED_FEATURE_PARAMETER, checkedFeatureValue);
		if (message != null) {
			errorParameters.setProperty(ERROR_PARAMETER_REASON, message);
		}
	
	   	addError(messageKey, errorParameters, position);
	 }

}