/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/metrics/MaxNestingPerMethod.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.metrics;

import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.List;

import com.sap.tc.jtools.jlint.jom.interfaces.IBlock;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jlint.jom.metrics.Metric;
import com.sap.tc.jtools.jlint.jom.metrics.MetricInfo;
import com.sap.tc.jtools.util.collections.IntStack;

/**
 * @author D037913
 */
public class MaxNestingPerMethod extends JomMetricVisitor {
  private static final String METRIC_NAME = "nestinglevel";

  /** keep in sync with tests.xml! */

  private static final String NAME = "Max nesting level per method";

  private IntStack counterStack = new IntStack();
  private IntStack maxStack = new IntStack();

  public MaxNestingPerMethod() {
    super();
  }

  private void decrementCounter() {
    if (counterStack.isEmpty())
      return;
    counterStack.push(counterStack.pop() - 1);
  }

  public void endVisit(IBlock node) {
    decrementCounter();
  }

  public void endVisit(IMethodDeclaration node) {
    counterStack.pop();
    int maxNestingLevel = maxStack.pop();
    // don't count abstract and interface methods
    if (Modifier.isAbstract(node.getModifiers())
      || node.resolveBinding().getDeclaringClass().isInterface())
      return;
    addMetricValue(METRIC_NAME, maxNestingLevel, node);

  }

  protected MetricInfo[] getMetricInfos() {
    return new MetricInfo[] {
       new MetricInfo(
        METRIC_NAME,
        MetricInfo.AVERAGE | MetricInfo.MAX | MetricInfo.STD_DEV)};
  }

  public String getTestName() {
    return NAME;
  }

  private void incrementCounter() {
    if (counterStack.isEmpty() || maxStack.isEmpty())
      return;
    int maxLevel = maxStack.pop();
    int currentLevel = counterStack.pop() + 1;
    counterStack.push(currentLevel);
    maxStack.push(Math.max(maxLevel, currentLevel));
  }

  public boolean visit(IBlock node) {
    incrementCounter();
    return true;
  }

  public boolean visit(IForStatement node) {
    if (node.getBody() instanceof IBlock)
      return true;
    incrementCounter();
    decrementCounter();
    return true;
  }

  public boolean visit(IIfStatement node) {
    if (!(node.getThenStatement() instanceof IBlock)) {
      incrementCounter();
      decrementCounter();
    }
    IStatement elseStmt = node.getElseStatement();
    if (elseStmt != null && !(elseStmt instanceof IBlock)) {
      incrementCounter();
      decrementCounter();
    }
    return true;
  }

  public boolean visit(IMethodDeclaration node) {
    counterStack.push(-1);
    maxStack.push(-1);
    return true;
  }

  public boolean visit(ISwitchStatement node) {
    List stmts = node.statements();
    for (Iterator iter = stmts.iterator(); iter.hasNext();) {
      if (!(iter.next() instanceof IBlock)) {
        incrementCounter();
        decrementCounter();
      }
    }
    return true;
  }

  public boolean visit(IWhileStatement node) {
    if (node.getBody() instanceof IBlock)
      return true;
    incrementCounter();
    decrementCounter();
    return true;
  }

}
