/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/metrics/CyclomaticComplexity.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.metrics;

import java.lang.reflect.Modifier;

import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jlint.jom.metrics.MetricInfo;
import com.sap.tc.jtools.util.collections.IntStack;

/**
 * This JLin metric measures the cyclomatic complexity of each method
 * in a class.
 * A method without any "decision points" has a cyclomatic complexity of 1.
 * Each time a decision point is encountered within a method, its complexity
 * value is incremented by 1. This implementation considers 
 * <code>if, for, while, case, catch, throw</code> to be decision points.
 * 
 * NOTE: interfaces and local classes are not measured.
 *  
 * @author D037913
 */
public class CyclomaticComplexity extends JomMetricVisitor {

  public static final String TEST_NAME = "Cyclomatic Complexity Metric"; //$NON-NLS-1$
  private static final String METRIC_NAME = "complexity"; //$NON-NLS-1$

  private IntStack complexityCounterStack = new IntStack();

  protected final MetricInfo[] getMetricInfos() {
    return new MetricInfo[] {
       new MetricInfo(
        METRIC_NAME,
        MetricInfo.AVERAGE | MetricInfo.STD_DEV | MetricInfo.MAX)};
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.JomTestVisitor#getTestName()
   */
  public final String getTestName() {
    return TEST_NAME;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
   */
  public final void endVisit(IMethodDeclaration meth) {
    if (complexityCounterStack.isEmpty())
      return;
    int complexity = complexityCounterStack.pop();
    int modifiers = meth.getModifiers();
    if (!(Modifier.isAbstract(modifiers) || Modifier.isNative(modifiers))) {
      addMetricValue(METRIC_NAME, complexity, meth);
    }
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause)
   */
  public final boolean visit(ICatchClause node) {
    incrementCounter();
    return true;
  }

  /* increment complexity counter by 1 */
  private void incrementCounter() {
    if (complexityCounterStack.isEmpty())
      return;
    complexityCounterStack.push(complexityCounterStack.pop() + 1);
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
   */
  public final boolean visit(IForStatement node) {
    incrementCounter();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement)
   */
  public final boolean visit(IIfStatement node) {
    incrementCounter();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
   */
  public final boolean visit(IMethodDeclaration node) {
    complexityCounterStack.push(1);
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase)
   */
  public final boolean visit(ISwitchCase node) {
    incrementCounter();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement)
   */
  public final boolean visit(IThrowStatement node) {
    incrementCounter();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement)
   */
  public final boolean visit(IWhileStatement node) {
    incrementCounter();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration)
   */
  public final boolean visit(ITypeDeclaration node) {
    // interfaces and local classes are ignored
    return !(node.isInterface() || node.isLocalTypeDeclaration());
  }

}
