/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/logging/LoggingAndTracingTest.java#3 $
 */

package com.sap.tc.jtools.jlint.tests.logging;

import java.io.FileReader;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.IStringLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jlint.jom.metrics.MetricInfo;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * @author D037913
 */
public class LoggingAndTracingTest extends JomMetricVisitor {

	private static final String PAR_LOG_METHOD_INFO = "LOG_METHOD_INFO";
	/* keep in sync with tests.xml! */
	private static final String NAME = "Logging and Tracing Test";
	private static final String LOG_OR_TRACE_MSG_METRIC =
		"log or trace method call";

	/* message keys and message parameters */
	private static final String MSG_KEY_1 = "logging.1";
	private static final String MSG_KEY_2 = "logging.2";
	private static final String MSG_KEY_3 = "logging.3";
	private static final String MSG_KEY_4 = "logging.4";
	private static final String MSG_PAR_CAT_USED = "USED_CATEGORY";
	private static final String MSG_PAR_LOG_MSG_USED = "USED_LOG_MESSAGE";
	private static final String MSG_PAR_LOG_CLASS = "LOG_CLASS";
	private static final String MSG_PAR_LOG_METHOD = "LOG_METHOD";
	private static final String MSG_PAR_FORMATTER_NAME = "FORMATTER_NAME";

	/* some logging API class names */
	private static final String LOG = "com.sap.tc.logging.Log";
	private static final String CATEGORY = "com.sap.tc.logging.Category";
	private static final String FORMATTER = "com.sap.tc.logging.Formatter";

	private static final String GET_CATEGORY = "getCategory";

	private static LogMethodInfo[] logMethods;
	
  private static final String PAR_CHECK_TRACE_MSGS = "CHECK_FOR_TRACE_MSGS";
  private static final String PAR_IGNORE_UNKNOWN_MSGS = "IGNORE_STATICALLY_UNKNOWN_MSGS";

	private boolean checkForTraceMsgs = true;
	private boolean ignoreUnknownMsgs = true;
  
	public String getTestName() {
		return NAME;
	}

	public LoggingAndTracingTest() {
		super();
	}

	protected MetricInfo[] getMetricInfos() {
		return new MetricInfo[] {
			 new MetricInfo(LOG_OR_TRACE_MSG_METRIC, MetricInfo.SUM)};
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation)
	 */
	public boolean visit(IMethodInvocation node) {
		IMethodBinding methBnd = node.resolveMethodBinding();
		LogMethodInfo logMethod = getLogOrTraceMethod(methBnd);
		if (logMethod != null) {
			addMetricValue(LOG_OR_TRACE_MSG_METRIC, 1, node);
			checkCall(logMethod, node);
		}
		IReferenceTypeBinding declClass = methBnd.getDeclaringClass();
		if (InheritanceTool.bndExtends(declClass, CATEGORY)) {
			checkCategoryCall(methBnd, node);
		}

		return true;
	}

	private void checkCall(
		LogMethodInfo logMethod,
		IMethodInvocation invocation) {
		String cat = getCategory(logMethod, invocation);
		if (cat != null) {
			Properties p = new Properties();
			p.setProperty(MSG_PAR_CAT_USED, cat);
			addError(MSG_KEY_1, p, invocation);
		}
		String msg = getLogMsg(invocation, logMethod);
//		if (msg == null) //dbg
//			msg = "***";
		if (msg != null) {
			Properties p = new Properties();
			p.setProperty(MSG_PAR_LOG_MSG_USED, msg);
			p.setProperty(MSG_PAR_LOG_CLASS, logMethod.getClasz());
			p.setProperty(MSG_PAR_LOG_METHOD, logMethod.getName()+"@"+logMethod.getSignature());
			if ( logMethod.getType().equals(LogMethodInfo.LOG_TYPE)) {
				addError(MSG_KEY_3, p, invocation);				
			} else {
				if (checkForTraceMsgs) {
				  addError(MSG_KEY_4, p, invocation);
				}
			}

		}
	}

	private String getCategory(
		LogMethodInfo logMethod,
		IMethodInvocation invocation) {
		return getParameterValue(invocation, logMethod.getCategory());
	}

	private String getLogMsg(
		IMethodInvocation invocation,
		LogMethodInfo logMethod) {
		String msg = logMethod.getText();
		return getParameterValue(invocation, msg);
	}

	private String getParameterValue(IMethodInvocation mi, String code) {
		if (code == null)
			return null;
		if (!code.startsWith("$")) {
			return code;
		}
		String msgPosition = code.substring(1);
		try {
			int msgIndex = Integer.parseInt(msgPosition);
			List args = mi.arguments();
			return getString((IExpression) args.get(msgIndex));
		} catch (Exception e) {
			return null;
		}
	}

	private String getString(IExpression expr) {
		if (expr instanceof IStringLiteral) {
			return ((IStringLiteral) expr).getLiteralValue();
		}
		String result = null;
		if (expr instanceof ISimpleName) {
			ISimpleName name = (ISimpleName) expr;
			IBinding binding = name.resolveBinding();
			if (binding instanceof IVariableBinding) {
				IVariableBinding vb = (IVariableBinding) binding;
				result = vb.getValue();
			}
		} else if (expr instanceof IFieldAccess) {
			IFieldAccess name = (IFieldAccess) expr;
			IBinding binding = name.resolveBinding();
			if (binding instanceof IVariableBinding) {
				IVariableBinding vb = (IVariableBinding) binding;
				result = vb.getValue();
			}
		}
		if (!ignoreUnknownMsgs && result == null) {
		  return "<statically unknown>";
		} else {
			return result;
		}
	}

	private static LogMethodInfo getLogOrTraceMethod(IMethodBinding methBnd) {
		String methName = methBnd.getName();
		IReferenceTypeBinding declClass = methBnd.getDeclaringClass();
		ITypeBinding[] parTypes = methBnd.getParameterTypes();
		String methSig = "";
		for (int i = 0; i < parTypes.length; i++) {
			methSig += parTypes[i].getName() + ";";
		}

		for (int i = 0; i < logMethods.length; i++) {
			String cls = logMethods[i].getClasz();
			if (!InheritanceTool.bndImplements(declClass, cls)
				&& !InheritanceTool.bndExtends(declClass, cls)) {
				continue;
			}
			String name = logMethods[i].getName();
			if (!methName.equals(name)) {
				continue;
			}
			String signature = logMethods[i].getSignature();
			if (!signature.equals(methSig)) {
				continue;
			}
			return logMethods[i];
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation)
	 */
	public boolean visit(IClassInstanceCreation node) {
		IMethodBinding constrBnd = node.resolveConstructorBinding();
		if (InheritanceTool.bndExtends(constrBnd.getDeclaringClass(), LOG)) {
			// check for custom formatter as constructor argument
			List args = node.arguments();
			for (Iterator iter = args.iterator(); iter.hasNext();) {
				ITypeBinding argBnd =
					((IExpression) iter.next()).resolveTypeBinding();
				if (argBnd instanceof IReferenceTypeBinding) {
					if (InheritanceTool
						.bndExtends(
							((IReferenceTypeBinding) argBnd),
							FORMATTER)) {
						Properties p = new Properties();
						p.setProperty(
							MSG_PAR_FORMATTER_NAME,
							((IReferenceTypeBinding) argBnd).getName());
						addError(MSG_KEY_2, p, node);
					}
				}
			}
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit)
	 */
	public boolean visit(ICompilationUnit node) {
		// add dummy value to metrics so that sum will show up even if there are
		// no log/trace method calls at all
		addMetricValue(LOG_OR_TRACE_MSG_METRIC, 0, node);
		return true;
	}

	public void setParameters(
		ParameterInterface[] parameters,
		TestObject testObject) {
		super.setParameters(parameters, testObject);
		this.checkForTraceMsgs = ((Boolean)getInputParameter(PAR_CHECK_TRACE_MSGS)).booleanValue();
		this.ignoreUnknownMsgs= ((Boolean)getInputParameter(PAR_IGNORE_UNKNOWN_MSGS)).booleanValue();
		String logMethodFile = (String) getInputParameter(PAR_LOG_METHOD_INFO);
		try {
			StructureTree tree =
				XMLTool.parseReader(new FileReader(logMethodFile));
			StructureTree[] children = tree.getChildren();
			logMethods = new LogMethodInfo[children.length];
			for (int i = 0; i < children.length; i++) {
				logMethods[i] = new LogMethodInfo(children[i]);
			}
		} catch (Exception e) {
			logException(e);
			logMethods = new LogMethodInfo[0];
		}
	}

	private void checkCategoryCall(
		IMethodBinding methBnd,
		IMethodInvocation invocation) {
		List args = invocation.arguments();
		// check for getCategory(String categoryName)
		if (GET_CATEGORY.equals(methBnd.getName())
			&& args.size() == 1) {
			String cat = getParameterValue(invocation, "$0");
			if (cat != null) {
				Properties p = new Properties();
				p.setProperty(
					MSG_PAR_CAT_USED,
					cat);
				addError(MSG_KEY_1, p, invocation);
			}

		}

	}

	private class LogMethodInfo {
		private static final String LOG_METHOD = "log-method";
		private static final String TRACE_METHOD = "trace-method";
		public static final String LOG_TYPE = "log";
		public static final String TRACE_TYPE = "trace";
		private String type;
		private String clasz;
		private String name;
		private String signature;
		private String category;
		private String location;
		private String text;

		public LogMethodInfo(StructureTree tree) {
			String tag = tree.getTag();
			if (tag.equals(LOG_METHOD)) {
				type = LOG_TYPE;
			} else if (tag.equals(TRACE_METHOD)) {
				type = TRACE_TYPE;
			}
			clasz = tree.getParameter("class");
			name = tree.getParameter("name");
			signature = tree.getParameter("signature");
			category = tree.getParameter("category");
			location = tree.getParameter("location");
			text = tree.getParameter("text");
		}

		public String getCategory() {
			return category;
		}

		public String getClasz() {
			return clasz;
		}

		public String getLocation() {
			return location;
		}

		public String getName() {
			return name;
		}

		public String getSignature() {
			return signature;
		}

		public String getText() {
			return text;
		}

		public String getType() {
			return type;
		}
	}
}
