/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/language/SwitchTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.language;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IBreakStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ICastExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IContinueStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.collections.BooleanStack;
import com.sap.tc.jtools.util.collections.IntStack;

/**
 * This JLin test checks for two frequent sources of error when using switch:
 * <ol>
 *   <li>the switch statement has no <code>default</code> case</li>
 *   <li>a case statement "falls through", i.e. has no
 *       <code>break, return, continue</code> or <code>throw</code> statement </li>
 *   <li>the default case is the last case</li>
 * </ol>
 * 
 * @author d037913
 */
public class SwitchTest extends JomTestVisitor {

  public static String NAME = "Switch Test"; //$NON-NLS-1$

  public static String PAR_MAX_CASE_NUMBER = "MAX_CASE_NUMBER"; //$NON-NLS-1$

  private static final String MSG_KEY_1 = "switch.1"; //$NON-NLS-1$
  private static final String MSG_KEY_2 = "switch.2"; //$NON-NLS-1$
  private static final String MSG_KEY_3 = "switch.3"; //$NON-NLS-1$
  private static final String MSG_KEY_4 = "switch.4"; //$NON-NLS-1$

  // stack of Booleans indicating if the current 
  // switch has a default case

  private BooleanStack switchStack = new BooleanStack();
  private IntStack caseCounterStack = new IntStack(20);
  // stack of stack of HashMaps case -> boolean indicating if the
  // case can fall through
  private Stack caseStack = new Stack();

  private int maxCaseNumber;

  public String getTestName() {
    return NAME;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.Visitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase)
   */
  public boolean visit(ISwitchCase node) {
    caseCounterStack.push(caseCounterStack.pop() + 1);
    Stack csStack = (Stack) caseStack.peek();
    Map fallThruMap = new HashMap(1);
    fallThruMap.put(node, Boolean.TRUE);
    csStack.push(fallThruMap);
    if (node.isDefault()) {
      switchStack.pop();
      switchStack.push(true);
    } else {
      // check if default is the last case
      if (switchStack.peek()) {
        addError(MSG_KEY_3, null, node);
      }
    }
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.Visitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement)
   */
  public void endVisit(ISwitchStatement node) {
    if (!switchStack.peek()) {
      addError(MSG_KEY_1, null, node);
    }
    int caseCount = caseCounterStack.pop();
    if (caseCount > maxCaseNumber) {
      Properties p = new Properties();
      p.setProperty(PAR_MAX_CASE_NUMBER, String.valueOf(maxCaseNumber));
      addError(MSG_KEY_4, p, node);
    }
    checkCases(node);
    // reset
    switchStack.pop();
    caseStack.pop();
  }

  private void checkCases(ISwitchStatement switchStmt) {
    Stack csStack = (Stack) caseStack.peek();
    while (!csStack.isEmpty()) {
      Map fallThruMap = (Map) csStack.pop();
      Set deliberateFallThrus = getDeliberateFallThrus(switchStmt);
      Set entries = fallThruMap.entrySet();
      for (Iterator iter = entries.iterator(); iter.hasNext();) {
        Map.Entry entry = (Map.Entry) iter.next();
        ISwitchCase caze = (ISwitchCase) entry.getKey();
        Boolean fallThru = (Boolean) entry.getValue();
        if (fallThru.booleanValue() && !deliberateFallThrus.contains(caze)) {
          addFallThroughError(caze);
        }
      }
    }
  }

  /**
   * @return Set of ISwitchCase that are empty 
   * => we suppose they deliberately fall thru
   */
  private Set getDeliberateFallThrus(ISwitchStatement switchStmt) {
    Set out = new HashSet();
    List stmts = switchStmt.statements();
    IStatement lastStmt = null;
    ISwitchCase lastSwitchCase = null;
    for (Iterator iter = stmts.iterator(); iter.hasNext();) {
      IStatement stmt = (IStatement) iter.next();
      if (stmt instanceof ISwitchCase) {
        lastSwitchCase = (ISwitchCase) stmt;
      }
      if (lastStmt != null
        && (lastStmt instanceof ISwitchCase)
        && (stmt instanceof ISwitchCase)) {
        // lastStmt was empty
        out.add(lastStmt);
      }
      lastStmt = stmt;
    }
    // last case falling through is OK
    if (lastSwitchCase != null) {
      out.add(lastSwitchCase);
    }
    return out;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.Visitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement)
   */
  public boolean visit(ISwitchStatement node) {
    switchStack.push(false);
    caseStack.push(new Stack());
    caseCounterStack.push(0);
    return true;
  }

  private void addFallThroughError(Position pos) {
    addError(MSG_KEY_2, null, pos);
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IBreakStatement)
   */
  public boolean visit(IBreakStatement node) {
    if (caseStack.isEmpty()) {
      return true;
    }
    markCaseNoFallThru();
    return true;
  }

  private void markCaseNoFallThru() {
    Stack csStack = (Stack) caseStack.peek();
    Map fallThruMap = (Map) csStack.peek();
    Set entries = fallThruMap.entrySet();
    for (Iterator iter = entries.iterator(); iter.hasNext();) {
      Map.Entry entry = (Map.Entry) iter.next();
      fallThruMap.put(entry.getKey(), Boolean.FALSE);
    }
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IContinueStatement)
   */
  public boolean visit(IContinueStatement node) {
    if (caseStack.isEmpty()) {
      return true;
    }
    markCaseNoFallThru();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement)
   */
  public boolean visit(IReturnStatement node) {
    if (caseStack.isEmpty()) {
      return true;
    }
    markCaseNoFallThru();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement)
   */
  public boolean visit(IThrowStatement node) {
    if (caseStack.isEmpty()) {
      return true;
    }
    markCaseNoFallThru();
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.TestComponentInterface#setParameters(com.sap.tc.jtools.jtci.interfaces.ParameterInterface[], com.sap.tc.jtools.jtci.TestObject)
   */
  public void setParameters(
    ParameterInterface[] parameters,
    TestObject testObject) {
    super.setParameters(parameters, testObject);
    maxCaseNumber =
      ((Integer) getInputParameter(PAR_MAX_CASE_NUMBER)).intValue();
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit)
   */
  public boolean visit(ICompilationUnit node) {
    // clear stacks, just to be sure...
    switchStack.clear();
    caseCounterStack.clear();
    caseStack.clear();
    return true;
  }

}
