/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/language/PathLengthTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.language;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IAnonymousClassDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * This JLin test checks if the packageroot-relative class file name 
 * corresponding to each type declaration exceeds a maximum character length.
 * (E.g. on Windows 2000, there is a general path length limit of 256 characters).
 * 
 * The maximum character length is specified by the integer 
 * input parameter MAX_PATH_LENGTH. 
 *  
 * @author D037913
 */
public class PathLengthTest extends JomTestVisitor {

  /* keep in sync with tests.xml! */
  private static final String NAME = "Path Length";
  private static final String PAR_MAX_LENGTH = "MAX_PATH_LENGTH";
  private static final String CLASS_FILE_NAME = "CLASS_FILE_NAME";
  private static final String PACKAGE_SEPARATOR = File.separator;
  private static final String INNER_CLASS_SEPARATOR = "$";
  private static final String CLASS_FILE_SUFFIX = ".class";
  private static final String MSG_KEY = "pathlength.1";

  private int maxPathLength;
  private int anonClassCounter = 0;
  private Properties msgProps;
  private StringBuffer nameBuf = new StringBuffer();
  private Map localClassMap = new HashMap();

  public String getTestName() {
    return NAME;
  }

  public void setParameters(
    ParameterInterface[] parameters,
    TestObject testObject) {
    super.setParameters(parameters, testObject);
    maxPathLength = ((Integer) getInputParameter(PAR_MAX_LENGTH)).intValue();
    msgProps = new Properties();
    msgProps.setProperty(PAR_MAX_LENGTH, String.valueOf(maxPathLength));
  }

  public boolean visit(ITypeDeclaration node) {
    if (node.resolveBinding().isTopLevel()) {
      anonClassCounter = 0;
      localClassMap.clear();
    }
    checkName(node.resolveBinding(), node);
    return true;
  }

  private void checkName(IReferenceTypeBinding bnd, Position pos) {
    String name = getClassFileName(bnd);
    if (name.length() > maxPathLength) {
      msgProps.setProperty(CLASS_FILE_NAME, name);
      addError(MSG_KEY, msgProps, pos);
    }
  }

  public boolean visit(IAnonymousClassDeclaration node) {
    checkName(node.resolveBinding(), node);
    return true;
  }

  private void descendClassDecl(IReferenceTypeBinding bnd) {
    if (bnd.isAnonymous()) {
      nameBuf.insert(0, INNER_CLASS_SEPARATOR + ++anonClassCounter);
      descendLocalorAnonClassDecl(bnd.getDeclaringClass());
    } else if (bnd.isLocal()) {
      String name = bnd.getIdentifier();
      Integer nameCount = (Integer) localClassMap.get(name);
      if (nameCount != null) {
        nameCount = new Integer(nameCount.intValue() + 1);
      } else {
        nameCount = new Integer(1);
      }
      localClassMap.put(name, nameCount);
      nameBuf.insert(
        0,
        INNER_CLASS_SEPARATOR
          + nameCount.toString()
          + INNER_CLASS_SEPARATOR
          + bnd.getIdentifier());
      descendLocalorAnonClassDecl(bnd.getDeclaringClass());
    } else {
      IReferenceTypeBinding declClass = bnd.getDeclaringClass();
      if (declClass == null) {
        nameBuf.insert(0, bnd.getIdentifier());
        IPackageBinding pkg = bnd.getPackage();
        if (!pkg.isUnnamed()) {
          String[] pkgComps = pkg.getNameComponents();
          for (int i = pkgComps.length - 1; i > -1; i--) {
            nameBuf.insert(0, pkgComps[i] + PACKAGE_SEPARATOR);
          }
        }
      } else {
        nameBuf.insert(0, INNER_CLASS_SEPARATOR + bnd.getIdentifier());
        descendClassDecl(declClass);
      }
    }
  }

  private void descendLocalorAnonClassDecl(IReferenceTypeBinding binding) {
    IReferenceTypeBinding declClass = binding.getDeclaringClass();
    if (declClass != null) {
      descendLocalorAnonClassDecl(declClass);
    } else {
      descendClassDecl(binding);
    }
  }

  private String getClassFileName(IReferenceTypeBinding bnd) {
    // clear buffer
    nameBuf.delete(0, nameBuf.length());
    nameBuf.append(CLASS_FILE_SUFFIX);
    descendClassDecl(bnd);
    return nameBuf.toString();
  }

}