/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/language/AssignmentTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.language;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IPostfixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IPrefixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;

/**
 *  This JLin test checks for questionable assignments, i.e.
 * 
 *   o assignments within conditional expressions (except in while conditions)
 *   o assignmnents to for loop variables outside the loop initializer
 *  
 *  @author d037913
 */
public class AssignmentTest extends JomTestVisitor {

	private static final String NAME = "Assignment Test";

	private static final String MSG_KEY_1 = "assign.1";
	private static final String MSG_KEY_2 = "assign.2";
	private static final String MSG_KEY_3 = "assign.3";

	// stack containing the IForStatement's
	private Stack forLoops = new Stack();
	// stack containing the IMethodDeclaration's
	private Stack methodStack = new Stack();

	private Set okAssignments = new HashSet();

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.JomTestVisitor#getTestName()
	 */
	public String getTestName() {
		return NAME;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
	 */
	public boolean visit(IForStatement node) {
		forLoops.push(node);
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
	 */
	public void endVisit(IForStatement node) {
		forLoops.pop();
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IAssignment)
	 */
	public boolean visit(IAssignment node) {
		IExpression leftSide = node.getLeftHandSide();
		// check if it's an assignment to a loop variable
		checkForLoopVariable(leftSide);
		// check if it's an assignment to a method parameter
		//		checkForParameter(leftSide);
		if (ISimpleType.BOOLEAN.equals(node.resolveTypeBinding().getName())) {
			if (!okAssignments.contains(node)) {
				addError(MSG_KEY_1, null, node);
			}
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IPostfixExpression)
	 */
	public boolean visit(IPostfixExpression node) {
		// postfix expressions (--, ++) are shorthands for assignments
		checkForLoopVariable(node.getOperand());
		//		checkForParameter(node.getOperand());
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IPrefixExpression)
	 */
		public boolean visit(IPrefixExpression node) {
			// incrementors and decrementors are actually 
			// assignments, too
			if (IPrefixExpression.INCREMENT.equals(node.getOperator())
				|| IPrefixExpression.DECREMENT.equals(node.getOperator())) {
				checkForLoopVariable(node.getOperand());
//				checkForParameter(node.getOperand());
			}
			return true;
		}

	//	private void checkForParameter(IExpression expr) {
	//		if (methodStack.isEmpty() || !(expr instanceof IName))
	//			return;
	//		IBinding bnd = ((IName) expr).resolveBinding();
	//		IMethodDeclaration methDecl = (IMethodDeclaration) methodStack.peek();
	//		List params = methDecl.parameters();
	//		// loop over all parameters and check if the binding is
	//		// a binding to a method parameter
	//		for (Iterator iter = params.iterator(); iter.hasNext();) {
	//			IVariableDeclaration varDecl = (IVariableDeclaration) iter.next();
	//			if (varDecl.resolveBinding().equals(bnd)) {
	//				addAssignmentToParameterError(expr);
	//			}
	//		}
	//	}

	private void checkForLoopVariable(IExpression expr) {
		if (forLoops.isEmpty() || !(expr instanceof ISimpleName))
			return;
		ISimpleName name = (ISimpleName) expr;
		for (Iterator forLoopIt = forLoops.iterator(); forLoopIt.hasNext();) {
			IForStatement forStmt = (IForStatement) forLoopIt.next();
			int bodyPos = forStmt.getBody().getStartPosition();
			IStatement[] initializers = forStmt.getInitializers();
			for (int i = 0; i < initializers.length; i++) {
				IStatement o = initializers[i];
				if (o instanceof IVariableDeclarationStatement) {
					IVariableDeclarationStatement declStmt =
						(IVariableDeclarationStatement) o;
					if (declStmt.resolveBinding().equals(name.resolveBinding())
						&& expr.getStartPosition() > bodyPos) {
						addAssignmentToLoopVarError(expr);
					}
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
	 */
	public boolean visit(IMethodDeclaration node) {
		methodStack.push(node);
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
	 */
	public void endVisit(IMethodDeclaration node) {
		methodStack.pop();
	}

	private void addAssignmentToParameterError(Position pos) {
		addError(MSG_KEY_2, null, pos);
	}

	private void addAssignmentToLoopVarError(Position pos) {
		addError(MSG_KEY_3, null, pos);
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement)
	 */
	public void endVisit(IExpressionStatement node) {
		okAssignments.clear();
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement)
	 */
	public boolean visit(IExpressionStatement node) {
		IExpression expr = node.getExpression();
		if (!ISimpleType.BOOLEAN.equals(expr.resolveTypeBinding().getName())) {
			return true;
		}
		// a = b = c; is OK
		while (expr instanceof IAssignment) {
			IAssignment assgnmnt = ((IAssignment) expr);
			okAssignments.add(expr);
			expr = assgnmnt.getRightHandSide();
		}
		return true;
	}

}
