/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/jdkc/JomJdkcTest.java#3 $
 */

package com.sap.tc.jtools.jlint.tests.jdkc;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.Reader;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IAnonymousClassDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IStringLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.ITryStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.tests.rc.RCStatusInfo;
import com.sap.tc.jtools.jlint.tests.rc.RestrictedComponents;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * This JLin test checks most of the source incompatibilities between JDK 1.3 and 1.4.
 * 
 * The cases listed in http://java.sun.com/j2se/1.4/compatibility.html#source are handled
 * as follows:
 * 
 * case 1:
 *   check if javax.swing.tree.DefaultTreeModel(javax.swing.tree.TreeNode) or
 *   javax.swing.tree.DefaultTreeModel.setRoot(javax.swing.tree.TreeNode) is called
 *   within a try { } catch (IllegalArgumentException e) { } block.
 *  
 * case 2:
 *   check if a there are any occurences of class literals (e.g. "Foo.class") 
 *   anywhere within serializable nested classes that don't declare 
 *   static final long serialVersionUID.
 *  
 * case 3:
 *   check for assignments to javax.swing.text.DefaultHighlighter.DefaultPainter.
 *   
 * case 4:
 *   check if javax.swing.text.html.HTMLWriter is used.
 *   
 * case 5:
 *   check if java.awt.event.MouseEvent.MOUSE_LAST is used. 
 *   
 * case 6:
 *   package org.omg is generally forbidden by restricted components test (except for
 *   callers from com.sap.engine.* ) => we don't check this.
 *   
 * case 7:
 *   check if any class that extends java.io.ObjectOutputStream overrides putFields().
 *   check if any class that extends java.io.ObjectInputStream overrides readFields().
 *   
 * case 8:
 *   these changes affect the compile-time only => we don't check this because the
 *   1.4 compiler will generate error messages.
 *   
 * case 9:
 *   check if an abstract class which implements Serializable (and doesn't declare
 *   static final long serialVersionUID) does not implement one or more methods from 
 *   another interface it directly implements.
 *   
 * case 10:
 *   package java.sql is forbidden by default in restricted components test => we don't check this.
 *   
 * case 11:
 *   check if java.net.URLConnection.getInputStream() is used within a 
 *   try { } catch (FileNotFoundException e) {} block.
 *   
 * case 12:
 *   check if any type, variable, method or field is named "assert".
 *   
 * case 13:
 *   check if any class implements java.applet.AppletContext (package java.applet is forbidden by
 *   restricted components test).
 *   
 * case 14:
 *   check for any non-abstract class that extends java.net.SocketImpl.
 *   
 * case 15:
 *   check if System.getProperty(String), System.getProperty(String, String) or System.setProperty(String,String)
 *   is used with the first argument a String literal "com.sun.net.ssl.dhKeyExchangeFix".
 *   
 * case 16:
 *   java security policy files are user-specific files that affect the runtime of the VM and thus cannot be
 *   checked statically.
 *   
 * case 17:
 *   check for any class that implements java.awt.dnd.DropTargetListener and has non-abstract methods
 *   drop(DropTargetDropEvent e) or dragExit(DropTargetEvent e).  
 *   
 * case 18:
 *   cannot be checked statically.
 *   
 * case 19:
 *   check for usage of java.awt.Window.toFront() and java.awt.Window.toBack().
 *   
 * case 20:
 *   cannot be checked statically.
 *
 *  
 * @author d027734, d037913, d034036
 *
 */

public class JomJdkcTest extends JomTestVisitor {

  static public final String SER_VER_UID_NAME = "serialVersionUID";
  static public final String TEST_NAME = "JDK 1.3 - 1.4 compatibility test"; //$NON-NLS-1$

  static public final String PARAMETER_GRAMMAR_FILE = "GRAMMAR_FILE"; //$NON-NLS-1$

  static public final String COMPATIBILITY_URL = "http://java.sun.com/j2se/1.4/compatibility.html#source"; //$NON-NLS-1$

  static public String DEFAULT_HIGHLIGHTER = "javax.swing.text.DefaultHighlighter"; //$NON-NLS-1$
  static public String JSSE_PROPERTY = "com.sun.net.ssl.dhKeyExchangeFix"; //$NON-NLS-1$

  static public final String JAVA_LANG_OBJECT = "java.lang.Object"; //$NON-NLS-1$
  static public final String DEFAULT_PAINTER = "DefaultPainter"; //$NON-NLS-1$
  static public final String OBJECT_OUTPUT_STREAM = "java.io.ObjectOutputStream"; //$NON-NLS-1$
  static public final String OBJECT_INPUT_STREAM = "java.io.ObjectInputStream"; //$NON-NLS-1$
  static public final String SERIALIZABLE = "java.io.Serializable"; //$NON-NLS-1$
  static public final String METHOD_PUT_FIELDS = "putFields"; //$NON-NLS-1$ 
  static public final String METHOD_READ_FIELDS = "readFields"; //$NON-NLS-1$
  static public final String INTERFACE_APPLET_CONTEXT = "java.applet.AppletContext"; //$NON-NLS-1$
  static public final String INTERFACE_DROPTARGET_LISTENER = "java.awt.dnd.DropTargetListener"; //$NON-NLS-1$
  static public final String CLASS_SOCKET_IMPL = "java.net.SocketImpl"; //$NON-NLS-1$

  static public final String CLASS_SOCKET_ADDRESS = "java.net.SocketAddress"; //$NON-NLS-1$
  static public final String EXCEPTION_ILLEGAL_ARGUMENT = "java.lang.IllegalArgumentException"; //$NON-NLS-1$
  static public final String EXCEPTION_FILE_NOT_FOUND = "java.io.FileNotFoundException"; //$NON-NLS-1$
  static public final String EXCEPTION_RUNTIME = "java.lang.RuntimeException"; //$NON-NLS-1$
  static public final String EXCEPTION_IO = "java.io.IOException"; //$NON-NLS-1$
  static public final String EXCEPTION = "java.lang.Exception"; //$NON-NLS-1$
  static public final String THROWABLE = "java.lang.Throwable"; //$NON-NLS-1$
  static public final String CLASS_NETWORK_INTERFACE = "java.net.NetworkInterface"; //$NON-NLS-1$
  static public final String CLASS_DATAGRAM_PACKET = "java.net.DatagramPacket"; //$NON-NLS-1$
  static public final String CLASS_INET_ADDRESS = "java.net.InetAddress"; //$NON-NLS-1$

  static public final String METHOD_CONNECT = "connect"; //$NON-NLS-1$
  static public final String METHOD_JOIN_GROUP = "joinGroup"; //$NON-NLS-1$
  static public final String METHOD_LEAVE_GROUP = "leaveGroup"; //$NON-NLS-1$
  static public final String METHOD_PEEK_DATA = "peekData"; //$NON-NLS-1$
  static public final String METHOD_SEND_URGENT_DATA = "sendUrgentData"; //$NON-NLS-1$
  static public final String METHOD_DRAG_EXIT = "dragExit"; //$NON-NLS-1$
  static public final String METHOD_DROP = "drop"; //$NON-NLS-1$

  static public final String KEYWORD_ASSERT = "assert"; //$NON-NLS-1$

  static private final String JDKC_STATUS_1 = "jdkc.1"; //$NON-NLS-1$
  static private final String JDKC_STATUS_2 = "jdkc.2"; //$NON-NLS-1$
  static private final String JDKC_STATUS_3 = "jdkc.3"; //$NON-NLS-1$
  static private final String JDKC_STATUS_7_1 = "jdkc.7.1"; //$NON-NLS-1$
  static private final String JDKC_STATUS_7_2 = "jdkc.7.2"; //$NON-NLS-1$
  static private final String JDKC_STATUS_9 = "jdkc.9"; //$NON-NLS-1$
  static private final String JDKC_STATUS_11 = "jdkc.11"; //$NON-NLS-1$
  static private final String JDKC_STATUS_12 = "jdkc.12"; //$NON-NLS-1$
  static private final String JDKC_STATUS_13 = "jdkc.13"; //$NON-NLS-1$
  static private final String JDKC_STATUS_14 = "jdkc.14"; //$NON-NLS-1$
  static private final String JDKC_STATUS_15 = "jdkc.15"; //$NON-NLS-1$
  static private final String JDKC_STATUS_17 = "jdkc.17"; //$NON-NLS-1$

  private Stack contextStack = new Stack();
  private Stack envStack = new Stack();
  private Stack enclCompUnitStack = new Stack();
  private Stack tryCatchStacks = new Stack();
  private Stack typeStack = new Stack();
  private RestrictedComponents rc;

  private HashSet checkedInterfaces;

  public String getTestName() {
    return TEST_NAME;
  }

  private void initialize() {
    String tempRestrictedComponentsList =
      (String) getInputParameter(PARAMETER_GRAMMAR_FILE);
    try {
      Reader fr = new BufferedReader(new FileReader(tempRestrictedComponentsList));
      rc = new RestrictedComponents(XMLTool.parseReader(fr));
      fr.close();
    } catch (Exception e) {
      logException(e);
    }
  }

  public void setParameters(
    ParameterInterface[] parameters,
    com.sap.tc.jtools.jtci.TestObject testObject) {
    super.setParameters(parameters, testObject);
    initialize();
  }

  public boolean visit(ITypeDeclaration typeDeclaration) {
    typeStack.push(typeDeclaration.resolveBinding());
    contextStack.push(new HashSet());
    tryCatchStacks.push(new Stack());
    envStack.push(typeDeclaration.resolveBinding());
    testTypeDeclaration(
      (IReferenceTypeBinding) typeDeclaration.resolveBinding(),
      typeDeclaration);
    if ((typeDeclaration.getName().getIdentifier() != null)
      && KEYWORD_ASSERT.equals(typeDeclaration.getName().getIdentifier())) {
      incompatError(typeDeclaration, JDKC_STATUS_12);
    }
    return true;
  }

  public void endVisit(ITypeDeclaration typeDeclaration) {
    typeStack.pop();
    contextStack.pop();
    envStack.pop();
    tryCatchStacks.push(new Stack());
  }

  public boolean visit(ICompilationUnit cU) {
    enclCompUnitStack.push(cU);
    contextStack.push(new HashSet());
    if (cU.getPackage() != null) {
      envStack.push(cU.getPackage().resolveBinding());
    }

    return true;
  }

  public boolean visit(IAnonymousClassDeclaration typeDeclaration) {
    typeStack.push(typeDeclaration.resolveBinding());
    contextStack.push(new HashSet());
    envStack.push(typeDeclaration.resolveBinding());
    tryCatchStacks.push(new Stack());
    testTypeDeclaration(typeDeclaration.resolveBinding(), typeDeclaration);
    return true;
  }

  public boolean visit(IVariableDeclaration varDecl) {
    if (KEYWORD_ASSERT.equals(varDecl.getName())) {
      incompatError(varDecl, JDKC_STATUS_12);
    }
    return true;
  }

  public boolean visit(ISimpleType st) {
    checkType(st.resolveBinding(), st);
    return true;
  }

  public void endVisit(IAnonymousClassDeclaration typeDeclaration) {
    typeStack.pop();
    contextStack.pop();
    envStack.pop();
    tryCatchStacks.pop();
  }

  public void endVisit(ICompilationUnit compilationUnitDeclaration) {
    contextStack.pop();
    if (compilationUnitDeclaration.getPackage() != null) {
      envStack.pop();
    }
    enclCompUnitStack.pop();
  }

  public boolean visit(IAssignment ass) {
    IExpression lhs = ass.getLeftHandSide();
    if (!(lhs instanceof IFieldAccess)) {
      return true;
    }
    IFieldAccess fAcc = (IFieldAccess) lhs;
    String accessedClass = fAcc.getExpression().resolveTypeBinding().getName();
    String selector = fAcc.getName();
    if (DEFAULT_PAINTER.equals(selector)
      && DEFAULT_HIGHLIGHTER.equals(accessedClass)) {
      incompatError(ass, JDKC_STATUS_3);
    }
    return true;
  }

  public boolean visit(IConstructorInvocation constCall) {
    try {
      IMethodBinding methBind = constCall.resolveConstructorBinding();
      RCStatusInfo status = getStatus(methBind);
      if (!status.getStatus().equals(RCStatusInfo.STRING_STATUS_OK)) {
        error(constCall, status);
      }
    } catch (Exception e) {
      logException(e, constCall);
      return false;
    }
    return true;
  }

  public boolean visit(ITypeLiteral classLiteral) {
    /*<???>*/
    ITypeBinding currClass = classLiteral.resolveTypeBinding();
    if (currClass.isArray()) {
      currClass = ((IArrayTypeBinding) currClass).getElementType();
    }
    if (currClass.isPrimitive())
      return true;

    if (isSubtypeOf((IReferenceTypeBinding) currClass, SERIALIZABLE)
      && ((IReferenceTypeBinding) currClass).isNested()) {
      incompatError(classLiteral, JDKC_STATUS_2);
    }
    /*</???>*/
    currClass = (ITypeBinding) typeStack.peek();
    if (currClass.isArray()) {
      currClass = ((IArrayTypeBinding) currClass).getElementType();
    }
    if (currClass.isPrimitive())
      return true;

    if (isSubtypeOf((IReferenceTypeBinding) currClass, SERIALIZABLE)
      && ((IReferenceTypeBinding) currClass).isNested()) {
      if (!declaresSerialVersionUID((IReferenceTypeBinding) currClass)
        || (isSubtypeOf(((IReferenceTypeBinding) currClass).getDeclaringClass(),
          SERIALIZABLE)
          && !declaresSerialVersionUID(((IReferenceTypeBinding) currClass)
            .getDeclaringClass()))) {
        incompatError(classLiteral, JDKC_STATUS_2);
      }
    }
    return true;
  }

  public boolean visit(IMethodInvocation methCall) {
    IVariableDeclaration handler = null;
    try {
      IMethodBinding methBind = methCall.resolveMethodBinding();
      RCStatusInfo statusInfo = getStatus(methBind);
      String status = statusInfo.getStatus();
      if (!status.equals(RCStatusInfo.STRING_STATUS_OK)) {
        if (status.equals(JDKC_STATUS_1)) {
          handler = checkCatch(EXCEPTION_ILLEGAL_ARGUMENT);
          if (null != handler) {
            incompatError(handler, status);
          }
        } else if (status.equals(JDKC_STATUS_11)) {
          handler = checkCatch(EXCEPTION_FILE_NOT_FOUND);
          if (null != handler) {
            incompatError(handler, status);
          }
        } else if (status.equals(JDKC_STATUS_15)) {
          List args = methCall.arguments();
          IExpression firstArg = (IExpression) args.get(0);
          if (firstArg instanceof IStringLiteral) {
            if (JSSE_PROPERTY
              .equals(((IStringLiteral) firstArg).getLiteralValue())) {
              incompatError(methCall, JDKC_STATUS_15);
            }
          }
        } else {
          error(methCall, statusInfo);
        }
      }
    } catch (Exception e) {
      logException(e, methCall);
      return false;
    }
    return true;
  }

  public boolean visit(ITryStatement tryStmt) {
    Stack lclTryCatchStack = (Stack) tryCatchStacks.peek();
    lclTryCatchStack.push(tryStmt);
    return true;
  }

  public void endVisit(ITryStatement tryStmt) {
    Stack lclTryCatchStack = (Stack) tryCatchStacks.peek();
    lclTryCatchStack.push(tryStmt);
  }

  public boolean visit(IMethodDeclaration methDecl) {
    Stack lclTryCatchStack = (Stack) tryCatchStacks.peek();
    lclTryCatchStack.clear();
    if (KEYWORD_ASSERT.equals(methDecl.getName())) {
      incompatError(methDecl, JDKC_STATUS_12);
    }
    return true;
  }

  public boolean visit(IFieldDeclaration fieldDeclaration) {
    if (KEYWORD_ASSERT.equals(fieldDeclaration.getName())) {
      incompatError(fieldDeclaration, JDKC_STATUS_12);
    }
    return true;
  }

  public boolean visit(IFieldAccess fa) {
    checkType(fa.resolveTypeBinding(), fa);
    try {
      IVariableBinding varBind = fa.resolveBinding();
      if (varBind.getDeclaringClass() == null) // [].length
        return true;
      RCStatusInfo statusInfo = getStatus(varBind);
      String status = statusInfo.getStatus();
      if (!status.equals(RCStatusInfo.STRING_STATUS_OK))
        error(fa, statusInfo);
    } catch (Exception e) {
      logException(e, fa);
      return false;
    }
    return true;
  }

  public boolean visit(IInitializer init) {
    Stack lclTryCatchStack = (Stack) tryCatchStacks.peek();
    lclTryCatchStack.clear();
    return true;
  }

  public boolean visit(IQualifiedName qn) {
    String qnString = qn.getName();
    if (qnString != null && qnString.charAt(qnString.length() - 1) == '*')
      return true;
    try {
      ITypeBinding type = qn.resolveTypeBinding();
      if (type == null) // import declaration
        return true;
      checkType(type, qn);
    } catch (Exception e) {
      logException(e, qn);
    }
    return true;
  }

  public boolean visit(ISimpleName node) {
    ITypeBinding binding = node.resolveTypeBinding();
    if (binding != null) //it could be an import declaration
      checkType(binding, node);
    return true;
  }

  private void checkType(ITypeBinding type, Position pos) {
    try {
      if (type.isArray()) {
        type = ((IArrayTypeBinding) type).getElementType();
      }
      if (type.isPrimitive())
        return;
      RCStatusInfo statusInfo = getStatus((IReferenceTypeBinding) type);
      String status = statusInfo.getStatus();
      if (!status.equals(RCStatusInfo.STRING_STATUS_OK))
        error(pos, statusInfo);
    } catch (Exception e) {
      logException(e, pos);
    }
  }

  private void testTypeDeclaration(
    IReferenceTypeBinding binding,
    Position pos) {
    try {
      if (isSubtypeOf(binding, OBJECT_OUTPUT_STREAM)) {
        IMethodBinding[] methods = binding.getDeclaredMethods();
        for (int i = 0; i < methods.length; i++) {
          String methName = methods[i].getName();

          if (METHOD_PUT_FIELDS.equals(methName)
            && 0 == methods[i].getParameterTypes().length) {
            incompatError(pos, JDKC_STATUS_7_1);
          }
        }
      } else {
        if (isSubtypeOf(binding, OBJECT_INPUT_STREAM)) {
          IMethodBinding[] methods = binding.getDeclaredMethods();

          for (int i = 0; i < methods.length; i++) {

            String methName = methods[i].getName();

            if (METHOD_READ_FIELDS.equals(methName)
              && 0 == methods[i].getParameterTypes().length) {

              incompatError(pos, JDKC_STATUS_7_2);
            }
          }
        }
      }
      IReferenceTypeBinding[] interfaces = binding.getInterfaces();
      if (implementsInterface(binding, SERIALIZABLE)
        && !declaresSerialVersionUID(binding)
        && Modifier.isAbstract(binding.getModifiers())
        && !binding.isInterface()) {
        IMethodBinding[] allMeths = binding.getDeclaredMethods();
        for (int j = 0; j < allMeths.length; j++) {
          if (!Modifier.isAbstract(allMeths[j].getModifiers())) {
            continue;
          }
          boolean found = false;
          // is abstract method from _directly_ implemented interface?
          intfloop : for (int i = 0; i < interfaces.length; i++) {
            IMethodBinding[] intfMeths = interfaces[i].getDeclaredMethods();
            for (int k = 0; k < intfMeths.length; k++) {
              if (allMeths[j].getName().equals(intfMeths[k].getName())) {
                found = true;
                break intfloop;
              }
            }
          }
          if (found)
            incompatError(pos, JDKC_STATUS_9);
        }
      }
      for (int i = 0; i < interfaces.length; i++) {
        if (interfaces[i].getName().equals(INTERFACE_APPLET_CONTEXT)) {
          incompatError(pos, JDKC_STATUS_13);
        }
      }
      if (!Modifier.isAbstract(binding.getModifiers())
        && isSubtypeOf(binding, CLASS_SOCKET_IMPL)) {
        incompatError(pos, JDKC_STATUS_14);
      }
      if (isSubtypeOf(binding, INTERFACE_DROPTARGET_LISTENER)) {
        IMethodBinding[] allMeths = binding.getDeclaredMethods();
        for (int i = 0; i < allMeths.length; i++) {
          if (!Modifier.isAbstract(allMeths[i].getModifiers())
            && (METHOD_DROP.equals(allMeths[i].getName())
              || METHOD_DRAG_EXIT.equals(allMeths[i].getName()))) {
            incompatError(pos, JDKC_STATUS_17);
          }
        }
      }
    } catch (Exception e) {
      logException(e, pos);
    }
  }

  private void incompatError(Position pos, String status) {
    addError(status, COMPATIBILITY_URL, null, pos);
  }

  private boolean isSubtypeOf(IReferenceTypeBinding clType, String name) {
    IReferenceTypeBinding parent = clType.getSuperclass();
    if (parent != null) {
      if (name.equals(parent.getName()))
        return true;
      if (isSubtypeOf(parent, name)) {
        return true;
      }
    }
    IReferenceTypeBinding[] interfaces = clType.getInterfaces();
    for (int i = 0; i < interfaces.length; i++) {
      if (name.equals(interfaces[i].getName()))
        return true;
      if (isSubtypeOf(interfaces[i], name)) {
        return true;
      }
    }
    return false;
  }

  private IVariableDeclaration checkCatch(String excName) {
    IVariableDeclaration argument;
    Stack lclTryCatchStack = (Stack) tryCatchStacks.peek();
    int stackLen = lclTryCatchStack.size();
    for (int i = stackLen - 1; i >= 0; i--) {
      List catchClauseList =
        ((ITryStatement) lclTryCatchStack.get(i)).catchClauses();
      if (catchClauseList != null || !catchClauseList.isEmpty()) {
        Iterator it = catchClauseList.iterator();
        while (it.hasNext()) {
          ICatchClause catchClause = ((ICatchClause) it.next());
          argument = catchClause.getException();
          ITypeBinding caughtExc =
            (ITypeBinding) argument.getType().resolveBinding();
          if (caughtExc.getName().equals(excName)) {
            return argument;
          }
          String caughtExcName = caughtExc.getName();
          if (EXCEPTION_FILE_NOT_FOUND.equals(excName)) {
            if (EXCEPTION_IO.equals(caughtExcName)
              || EXCEPTION.equals(caughtExcName)
              || THROWABLE.equals(caughtExcName)) {
              return null;
            } else if (EXCEPTION_ILLEGAL_ARGUMENT.equals(excName)) {
              if (EXCEPTION_RUNTIME.equals(caughtExcName)
                || EXCEPTION.equals(caughtExcName)
                || THROWABLE.equals(caughtExcName)) {
                return null;
              }
            }
          }
        }
      }
    }
    return null;
  }

  private void error(Position position, RCStatusInfo status) {
    addError(
      status.getStatus(),
      COMPATIBILITY_URL,
      null,
      position);
  }

  private boolean implementsInterface(
    IReferenceTypeBinding type,
    String interfaceFullName) {
    checkedInterfaces = new HashSet();
    return implementsInterfaceInt(type, interfaceFullName);
  }

  private boolean implementsInterfaceInt(
    IReferenceTypeBinding type,
    String interfaceName) {
    IReferenceTypeBinding[] interfaces = type.getInterfaces();
    if (interfaceName == null || interfaces.length == 0)
      return false;
    for (int i = 0; i < interfaces.length; i++) {
      if (checkedInterfaces.contains(interfaces[i])) {
        continue;
      }
      checkedInterfaces.add(interfaces[i]);
      if (interfaceName.equals(interfaces[i].getName())) {
        return true;
      } else {
        if (implementsInterfaceInt(interfaces[i], interfaceName)) {
          return true;
        }
      }

    }
    return false;
  }

  private boolean declaresSerialVersionUID(IReferenceTypeBinding bnd) {
    IVariableBinding[] fields = bnd.getDeclaredFields();
    boolean serialVerDeclared = false;
    fieldloop : for (int i = 0; i < fields.length; i++) {
      int mod = fields[i].getModifiers();
      if (!(fields[i].getName().equals(SER_VER_UID_NAME)
        && Modifier.isStatic(mod)
        && Modifier.isFinal(mod)))
        continue;
      ITypeBinding fieldType = fields[i].getType();
      if (fieldType.isPrimitive()
        && ISimpleType.LONG.equals(fieldType.getName())) {
        serialVerDeclared = true;
        break fieldloop;
      }
    }
    return serialVerDeclared;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation)
   */
  public boolean visit(IClassInstanceCreation node) {
    IVariableDeclaration handler = null;
    try {
      IMethodBinding methBind = node.resolveConstructorBinding();
      RCStatusInfo statusInfo = getStatus(methBind);
      String status = statusInfo.getStatus();
      if (!status.equals(RCStatusInfo.STRING_STATUS_OK)) {
        if (status.equals(JDKC_STATUS_1)) {
          handler = checkCatch(EXCEPTION_ILLEGAL_ARGUMENT);
          if (null != handler) {
            incompatError(handler, status);
          }
        } else {
          error(node, statusInfo);
        }
      }
    } catch (Exception e) {
      logException(e, node);
      return false;
    }
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement)
   */
  public boolean visit(IVariableDeclarationStatement node) {
    if (KEYWORD_ASSERT.equals(node.getName())) {
      incompatError(node, JDKC_STATUS_12);
    }
    return true;
  }

  private RCStatusInfo getStatus(IBinding bnd) {
    return rc.getStatusInfo((IBinding) envStack.peek(), bnd);
  }

}
