/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/exception/rewrite/CatchBlockTest.java#1 $
 */
package com.sap.tc.jtools.jlint.tests.exception.rewrite;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IBlock;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IEmptyStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;

/**
 * @author D037913
 */
public class CatchBlockTest extends JomTestVisitor {

  private static final String NAME                              = "New Catch Block Test";
  private Stack               catchBlockInfoStack               = new Stack();
  private static final String THROWABLE                         = Throwable.class
                                                                    .getName();
  private static final String OBJECT                            = Object.class
                                                                    .getName();
  private static final String SYSTEM                            = System.class
                                                                    .getName();
  private static final String MSG_KEY_EXC_NOT_NESTED            = "newcatchblock.1";
  private static final String MSG_KEY_EXC_NO_GOOD_PATTERN_FOUND = "newcatchblock.2";
  private static final String MSG_KEY_EMPTY_CATCH_BLOCK         = "newcatchblock.3";
  private static final String METH_PRINTSTACKTRACE              = "printStackTrace";
  private static final String METH_TOSTRING                     = "toString";
  private static final String METH_GETCLASS                     = "getClass";
  private static final String METH_EQUALS                       = "equals";
  private static final String METH_HASHCODE                     = "hashCode";
  private static final String METH_WAIT                         = "wait";
  private static final String METH_NOTIFY                       = "notify";
  private static final String METH_NOTIFYALL                    = "notifyAll";
  private static final String METH_GETMESSAGE                   = "getMessage";
  private static final String METH_GETLOCMESSAGE                = "getLocalizedMessage";
  private static final Set    BAD_METH_NAMES                    = new HashSet();
  static {
    BAD_METH_NAMES.addAll(Arrays.asList(new String[]{METH_PRINTSTACKTRACE,
      METH_TOSTRING, METH_GETCLASS, METH_EQUALS, METH_HASHCODE, METH_WAIT,
      METH_NOTIFY, METH_NOTIFYALL, METH_GETMESSAGE, METH_GETLOCMESSAGE}));
  }
  

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IArrayCreation)
   */
  public boolean visit(IArrayCreation node) {
    if (!isUsageCheckNecessary()) {
      return true;
    }
    IArrayInitializer arrayInit = node.getInitializer();
    if (arrayInit == null) {
      return true;
    }
    List exprs = arrayInit.expressions();
    for (Iterator iter = exprs.iterator(); iter.hasNext();) {
      IExpression expr = (IExpression) iter.next();
      if (extendsThrowableOrThrowableArray(expr.resolveTypeBinding())) {
        getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
        return true;
      }
    }
    return true;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IAssignment)
   */
  public boolean visit(IAssignment node) {
    if (!isUsageCheckNecessary()) {
      return true;
    }
    ITypeBinding bnd = node.getRightHandSide().resolveTypeBinding();
    if (!(bnd instanceof IReferenceTypeBinding)) {
      return true;
    }
    if (extendsThrowableOrThrowableArray((IReferenceTypeBinding) bnd)) {
      getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
    }
    return true;
  }

  private boolean isUsageCheckNecessary() {
    return insideCatchBlock()
        && !getCurrentCatchBlockInfo().isAnyUsagePatternFound();
  }

  private static boolean extendsThrowableOrThrowableArray(ITypeBinding binding) {
    if (!(binding instanceof IReferenceTypeBinding)) {
      return false;
    }
    IReferenceTypeBinding refBnd = (IReferenceTypeBinding) binding;
    if (refBnd.isArray()) {
      ITypeBinding elementBnd = ((IArrayTypeBinding) refBnd).getElementType();
      if (!(elementBnd instanceof IReferenceTypeBinding)) {
        return false;
      } else {
        refBnd = (IReferenceTypeBinding) elementBnd;
      }
    }
    return InheritanceTool.bndExtends(refBnd, THROWABLE);
  }

  private static boolean isObjectOrObjectArray(ITypeBinding binding) {
    if (!(binding instanceof IReferenceTypeBinding)) {
      return false;
    }
    IReferenceTypeBinding refBnd = (IReferenceTypeBinding) binding;
    if (refBnd.isArray()) {
      ITypeBinding elementBnd = ((IArrayTypeBinding) refBnd).getElementType();
      if (!(elementBnd instanceof IReferenceTypeBinding)) {
        return false;
      } else {
        refBnd = (IReferenceTypeBinding) elementBnd;
      }
    }
    return OBJECT.equals(refBnd.getName());
  }

  private static boolean extendsThrowable(ITypeBinding binding) {
    if (!(binding instanceof IReferenceTypeBinding)) {
      return false;
    }
    IReferenceTypeBinding refBnd = (IReferenceTypeBinding) binding;
    if (refBnd.isArray()) {
      return false;
    }
    return InheritanceTool.bndExtends(refBnd, THROWABLE);
  }

  public boolean visit(IClassInstanceCreation node) {
    if (!isUsageCheckNecessary()) {
      return true;
    }
    checkMethodOrConstructorInvocation(node.resolveConstructorBinding(), node
        .arguments());
    return true;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit)
   */
  public boolean visit(ICompilationUnit node) {
    // just to be sure to have a clear stack even if
    // exceptions occured before
    catchBlockInfoStack.clear();
    return true;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation)
   */
  public boolean visit(IMethodInvocation node) {
    if (!isUsageCheckNecessary()) {
      return true;
    }
    checkIfGoodUsageMethodOfThrowable(node);
    checkMethodOrConstructorInvocation(node.resolveMethodBinding(), node
        .arguments());
    return true;
  }

  /*
   * return whether the method invocation is a "good" method invoked on a
   * Throwable. "good" means NOT any of the following methods:
   */
  private void checkIfGoodUsageMethodOfThrowable(IMethodInvocation node) {
    IExpression expr = node.getExpression();
    if (expr == null || !extendsThrowable(expr.resolveTypeBinding())) {
      return;
    }
    IMethodBinding methBnd = node.resolveMethodBinding();
    String methName = node.getName();
    if (!BAD_METH_NAMES.contains(methName)) {
      getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
      return;
    } else {
      if (METH_PRINTSTACKTRACE.equals(methName)) {
        // check for printStackTrace() or
        // printStackTrace(System.out/System.err)
        // if not -> good usage found
        if (!isPrintStackTraceSystemOutOrErr(node)) {
          getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
          return;
        }
      } else if (METH_TOSTRING.equals(methName)) {
      } else if (METH_GETCLASS.equals(methName)) {
      } else if (METH_EQUALS.equals(methName)) {
      } else if (METH_HASHCODE.equals(methName)) {
      } else if (METH_WAIT.equals(methName)) {
      } else if (METH_NOTIFY.equals(methName)) {
      } else if (METH_NOTIFYALL.equals(methName)) {
      } else if (METH_GETMESSAGE.equals(methName)) {
      } else if (METH_GETLOCMESSAGE.equals(methName)) {
      }
    }
  }

  private boolean isPrintStackTraceSystemOutOrErr(IMethodInvocation node) {
    IMethodBinding methBnd = node.resolveMethodBinding();
    ITypeBinding[] paramTypes = methBnd.getParameterTypes();
    if (paramTypes.length == 0) {
      // printStackTrace() with no args goes to System.err
      return true;
    } else if (paramTypes.length == 1) {
      // check for printStackTrace(System.out) or
      // printStackTrace(System.err)
      ITypeBinding param = methBnd.getParameterTypes()[0];
      if (!(param instanceof IReferenceTypeBinding)) {
        return false;
      } else {
        if (InheritanceTool.bndExtends((IReferenceTypeBinding) param,
            PrintStream.class.getName())) {
          IExpression actualParam = (IExpression) node.arguments().get(0);
          if (actualParam instanceof IFieldAccess) {
            IFieldAccess fAcc = (IFieldAccess) actualParam;
            String fieldName = fAcc.getName();
            if ("out".equals(fieldName) || "err".equals(fieldName)) {
              ITypeBinding bnd = fAcc.getExpression().resolveTypeBinding();
              if (SYSTEM.equals(bnd.getName())) {
                return true;
              }
            }
          }
        }
      }
    }
    return false;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement)
   */
  public boolean visit(IReturnStatement node) {
    if (!isUsageCheckNecessary()) {
      return true;
    }
    IExpression expr = node.getExpression();
    if (expr != null
        && extendsThrowableOrThrowableArray(expr.resolveTypeBinding())) {
      getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
    }
    return true;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement)
   */
  public boolean visit(IThrowStatement node) {
    if (!insideCatchBlock()) {
      return true;
    }
    if (isBadThrowUsage(node)) {
      getCurrentCatchBlockInfo().setBadUsagePatternFound(true);
      addError(MSG_KEY_EXC_NOT_NESTED, null, node);
    } else {
      getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
    }
    return true;
  }

  /*
   * return whether the throw statements is the bad usage pattern throw new
   * Throwable(...) with no Throwables as arguments (i.e. thrown exception
   * doesn't nest causing one)
   */
  private boolean isBadThrowUsage(IThrowStatement node) {
    IExpression expr = node.getExpression();
    if (!(expr instanceof IClassInstanceCreation)) {
      return false;
    }
    List args = ((IClassInstanceCreation) expr).arguments();
    for (Iterator iter = args.iterator(); iter.hasNext();) {
      IExpression argExpr = (IExpression) iter.next();
      if (extendsThrowableOrThrowableArray(argExpr.resolveTypeBinding())) {
        return false;
      }
    }
    return true;
  }

  public String getTestName() {
    return NAME;
  }

  public void endVisit(ICatchClause node) {
    CatchBlockInfo cbInfo = (CatchBlockInfo) catchBlockInfoStack.pop();
    if (cbInfo.isAnyUsagePatternFound()) {
      // bad usage pattern found -> message already generated
      // good pattern found -> nothing to complain
      return;
    } else {
      // no good pattern found -> complain
      addError(MSG_KEY_EXC_NO_GOOD_PATTERN_FOUND, null, node);
    }
  }

  public boolean visit(ICatchClause node) {
    catchBlockInfoStack.push(new CatchBlockInfo());
    checkForEmptyCatchBlock(node);
    return true;
  }

  private void checkForEmptyCatchBlock(ICatchClause node) {
    IBlock catchBlock = node.getBody();
    List stmts = catchBlock.statements();
    if (stmts.size() == 0
        || (stmts.size() == 1 && (stmts.get(0) instanceof IEmptyStatement))) {
      getCurrentCatchBlockInfo().setBadUsagePatternFound(true);
      Properties p = new Properties();
      p.setProperty("EXCEPTION", node.getException().getType().resolveBinding()
          .getName());
      addError(MSG_KEY_EMPTY_CATCH_BLOCK, p, node);
    }
  }

  private boolean insideCatchBlock() {
    return !catchBlockInfoStack.isEmpty();
  }

  private CatchBlockInfo getCurrentCatchBlockInfo() {
    return (CatchBlockInfo) catchBlockInfoStack.peek();
  }

  private void checkMethodOrConstructorInvocation(IMethodBinding bnd,
      List arguments) {
    ITypeBinding[] formalParams = bnd.getParameterTypes();
    int i = 0;
    for (Iterator iter = arguments.iterator(); iter.hasNext(); i++) {
      IExpression arg = (IExpression) iter.next();
      if (extendsThrowableOrThrowableArray(arg.resolveTypeBinding())
          && !isObjectOrObjectArray(formalParams[i])) {
        getCurrentCatchBlockInfo().setGoodUsagePatternFound(true);
        return;
      }
    }
  }
}
