/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/exception/ExceptionTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.exception;

import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.collections.BooleanStack;

/**
 * This JLin test checks for some exception-related antipatterns:
 * 
 *  o throw  or catch Throwables that are not subclasses of java.lang.Exception
 *  o define a subclass of Throwable that is not a subclass of java.lang.Exception
 *  o catch an exception and throw a new one that doesn't preserve the causing exception's stacktrace 
 *  o (optionally) throw, catch or declare too generic exceptions (i.e. java.lang.Exception or java.lang.RuntimeException)
 *  o (optionally) declare exceptions to be thrown but don't throw them.
 * 
 * @author D037913
 */
public class ExceptionTest extends JomTestVisitor {

  /* keep in sync with tests.xml! */
  private static final String NAME = "ExceptionTest";

  private static final String MSG_KEY_2 = "exc.2";
  private static final String MSG_KEY_3 = "exc.3";
  private static final String MSG_KEY_4 = "exc.4";
  private static final String MSG_KEY_5 = "exc.5";
  private static final String MSG_KEY_6 = "exc.6";
  private static final String MSG_KEY_7 = "exc.7";
  private static final String MSG_KEY_8 = "exc.8";
//  private static final String MSG_KEY_9 = "exc.9";

  private static final String THROWABLE = "java.lang.Throwable";
  private static final String EXCEPTION = "java.lang.Exception";
  private static final String RUNTIME_EXCEPTION = "java.lang.RuntimeException";

  private static final String PAR_CHECK_DECARED_EXC = "CHECK_DECLARED_EXC";
  private static final String PAR_CHECK_TOO_GENERIC_EXC =
    "CHECK_TOO_GENERIC_EXC";

  private static final String MSG_PAR_EXCEPTION = "EXCEPTION";
  private static final String MSG_PAR_THROWABLE = "THROWABLE";

  private Stack throwsStack = new Stack();

  private BooleanStack inCatchBlockStack = new BooleanStack();

  private boolean checkTooGenericExceptions = false;
  private boolean checkIfDeclaredExcReallyThrown = false;

  public String getTestName() {
    return NAME;
  }

  public boolean visit(ICatchClause node) {
    inCatchBlockStack.pop();
    inCatchBlockStack.push(true);
    ITypeBinding excType = node.getException().resolveBinding().getType();
    if (!(excType instanceof IReferenceTypeBinding))
      return true;
    IReferenceTypeBinding excRefType = (IReferenceTypeBinding) excType;
    if (InheritanceTool.bndExtends(excRefType, THROWABLE)
      && !InheritanceTool.bndExtends(excRefType, EXCEPTION)) {
      addError(MSG_KEY_3, null, node);
    }
    String excName = excRefType.getName();
    if (checkTooGenericExceptions
      && (EXCEPTION.equals(excName) || RUNTIME_EXCEPTION.equals(excName))) {
      Properties p = new Properties();
      p.setProperty(MSG_PAR_EXCEPTION, excName);
      addError(MSG_KEY_7, p, node);
    }
    return true;
  }

  public boolean visit(IThrowStatement node) {
    IReferenceTypeBinding excBnd =
      (IReferenceTypeBinding) node.getExpression().resolveTypeBinding();
    if (InheritanceTool.bndExtends(excBnd, THROWABLE)
      && !InheritanceTool.bndExtends(excBnd, EXCEPTION)) {
      addError(MSG_KEY_4, null, node);
    }
    String excName = excBnd.getName();
    if (checkTooGenericExceptions
      && (EXCEPTION.equals(excName) || RUNTIME_EXCEPTION.equals(excName))) {
      Properties p = new Properties();
      p.setProperty(MSG_PAR_EXCEPTION, excName);
      addError(MSG_KEY_8, p, node);
    }
    if (!throwsStack.isEmpty()) {
      Map excMap = (Map) throwsStack.peek();
      Set declExcNames = excMap.keySet();
      for (Iterator iter = declExcNames.iterator(); iter.hasNext();) {
        String declExcName = (String) iter.next();
        if (InheritanceTool.bndExtends(excBnd, declExcName)) {
          excMap.put(declExcName, Boolean.TRUE);
        }
      }
    }
    if (inCatchBlockStack.peek())
      checkForLostStackTrace(node);
    return true;
  }

  private void checkForLostStackTrace(IThrowStatement node) {
    // check for throw new ...Exception() that doesn't 
    // preserve the causing exception (stacktrace loss)
    if (!(node.getExpression() instanceof IClassInstanceCreation))
      return;
    List args = ((IClassInstanceCreation) node.getExpression()).arguments();
    boolean throwableArgumentFound = false;
    for (Iterator iter = args.iterator();
      iter.hasNext() && !throwableArgumentFound;
      ) {
      IExpression argument = (IExpression) iter.next();
      ITypeBinding argumentBnd = argument.resolveTypeBinding();
      if (!(argumentBnd instanceof IReferenceTypeBinding))
        continue;
      throwableArgumentFound =
        InheritanceTool.bndExtends(
          (IReferenceTypeBinding) argumentBnd,
          THROWABLE);
    }
    // TODO consolidate with catchBlockTest
//    if (!throwableArgumentFound) {
//      addError(MSG_KEY_9, null, node);
//    }
  }

  public boolean visit(ITypeDeclaration node) {
    inCatchBlockStack.push(false);
    if (InheritanceTool.bndExtends(node.resolveBinding(), THROWABLE)
      && !InheritanceTool.bndExtends(node.resolveBinding(), EXCEPTION)) {
      addError(MSG_KEY_5, null, node);
    }
    return true;
  }

  public boolean visit(IMethodDeclaration node) {
    IReferenceTypeBinding[] excs = node.resolveBinding().getExceptionTypes();
    Map excMap = new HashMap();
    for (int i = 0; i < excs.length; i++) {
      String excName = excs[i].getName();
      if (checkTooGenericExceptions
        && (EXCEPTION.equals(excName) || RUNTIME_EXCEPTION.equals(excName))) {
        Properties p = new Properties();
        p.setProperty(MSG_PAR_EXCEPTION, excName);
        addError(MSG_KEY_2, p, node);
      }
      excMap.put(excName, Boolean.FALSE);
    }
    throwsStack.push(excMap);
    return true;
  }

  public void endVisit(IMethodDeclaration node) {
    Map excMap = (Map) throwsStack.pop();
    if (!checkIfDeclaredExcReallyThrown
      || Modifier.isAbstract(node.resolveBinding().getModifiers()))
      return;
    Set keys = excMap.keySet();
    for (Iterator iter = keys.iterator(); iter.hasNext();) {
      String excName = (String) iter.next();
      Boolean reallyThrown = (Boolean) excMap.get(excName);
      if (!reallyThrown.booleanValue()) {
        Properties p = new Properties();
        p.setProperty(MSG_PAR_THROWABLE, excName);
        addError(MSG_KEY_6, p, node);
      }
    }
  }

  public boolean visit(IClassInstanceCreation node) {
    addThrownExceptions(node.resolveConstructorBinding());
    return true;
  }

  private void addThrownExceptions(IMethodBinding binding) {
    if (throwsStack.isEmpty())
      return;
    Map excMap = (Map) throwsStack.peek();
    Set declExcNames = excMap.keySet();
    IReferenceTypeBinding[] excTypes = binding.getExceptionTypes();
    for (int i = 0; i < excTypes.length; i++) {
      for (Iterator iter = declExcNames.iterator(); iter.hasNext();) {
        String declExcName = (String) iter.next();
        if (InheritanceTool.bndExtends(excTypes[i], declExcName)) {
          excMap.put(declExcName, Boolean.TRUE);
        }
      }
    }
  }

  public boolean visit(IMethodInvocation node) {
    addThrownExceptions(node.resolveMethodBinding());
    return true;
  }

  public boolean visit(ISuperConstructorInvocation node) {
    addThrownExceptions(node.resolveConstructorBinding());
    return true;
  }

  public boolean visit(ISuperMethodInvocation node) {
    addThrownExceptions(node.resolveMethodBinding());
    return true;
  }

  public void setParameters(
    ParameterInterface[] parameters,
    TestObject testObject) {
    super.setParameters(parameters, testObject);
    checkIfDeclaredExcReallyThrown =
      ((Boolean) getInputParameter(PAR_CHECK_DECARED_EXC)).booleanValue();
    checkTooGenericExceptions =
      ((Boolean) getInputParameter(PAR_CHECK_TOO_GENERIC_EXC)).booleanValue();
  }

  public void endVisit(ITypeDeclaration node) {
    inCatchBlockStack.pop();
  }

  public void endVisit(ICatchClause node) {
    inCatchBlockStack.pop();
    inCatchBlockStack.push(false);
  }

}
