/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/exception/ExceptionHandlingMetric.java#3 $
 */

package com.sap.tc.jtools.jlint.tests.exception;

import java.util.ArrayList;
import java.util.List;

import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jlint.jom.metrics.MetricInfo;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * @author C5050508
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class ExceptionHandlingMetric extends JomMetricVisitor {

  /** keep in sync with tests.xml! */
  private static final String NAME = "Exception Handling Metric";

  private static String TOTAL_NUMBER_OF_THROWABLE =
    "classes extending java.lang.Throwable";
  private static String TOTAL_NUMBER_OF_THROW_STATEMENTS = "throw statements";
  private static String TOTAL_NUMBER_OF_CATCH_BLOCKS = "catch blocks";

  private static String[] exceptionClasses =
    {
      "com.sap.exception.BaseException",
      "com.sap.exception.BaseRuntimeException",
      java.sql.SQLException.class.getName(),
      java.lang.RuntimeException.class.getName(),
      java.lang.Exception.class.getName(),
      java.lang.Throwable.class.getName(),
      java.lang.Error.class.getName()};

  private static String[] exceptionInterfaces =
    { "com.sap.exception.IBaseException" };

  /**
   * 
   */
  public ExceptionHandlingMetric() {
    super();
  }

  public String getTestName() {
    return NAME;
  }

  public boolean visit(IThrowStatement node) {
    IExpression expression = node.getExpression();
    IReferenceTypeBinding binding = null;

    if (expression instanceof IClassInstanceCreation) {
      IClassInstanceCreation classCreation =
        (IClassInstanceCreation) expression;
      ITypeBinding typeBinding = classCreation.getName().resolveTypeBinding();

      if (typeBinding instanceof IReferenceTypeBinding)
        binding = (IReferenceTypeBinding) typeBinding;
      else
        return true;
    } else
      binding = (IReferenceTypeBinding) expression.resolveTypeBinding();

    addMetricValue(TOTAL_NUMBER_OF_THROW_STATEMENTS, 1, node);

    String bindingName = binding.getName();
    boolean foundClass = false;

    for (int i = 0; i < exceptionClasses.length; i++) {
      if (bindingName.equals(exceptionClasses[i]) == true) {
        addThrowMetricValue(exceptionClasses[i], true, 1.0f, node);
        foundClass = true;

        break;
      }
    }

    if (foundClass == false) {
      for (int i = 0; i < exceptionClasses.length; i++) {
        if (JomBindingUtil
          .checkExtendsClassIndirectly(binding, exceptionClasses[i])
          == true) {
          addThrowMetricValue(exceptionClasses[i], false, 1.0f, node);
          foundClass = true;

          break;
        }
      }
    }

    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause)
   */
  public boolean visit(ICatchClause node) {
    String exceptionTypeName =
      node.getException().resolveBinding().getType().getName();

    addMetricValue(TOTAL_NUMBER_OF_CATCH_BLOCKS, 1, node);

    for (int i = 0; i < exceptionClasses.length; i++) {
      if (exceptionTypeName.equals(exceptionClasses[i]) == true) {
        addMetricValue(
          "caught exceptions of type " + exceptionClasses[i],
          1,
          node);
        break;
      }
    }
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration)
   */
  public boolean visit(ITypeDeclaration node) {
    IReferenceTypeBinding binding =
      (IReferenceTypeBinding) node.resolveBinding();

    IReferenceTypeBinding superClass = binding.getSuperclass();
    IReferenceTypeBinding[] interfaces = binding.getInterfaces();

    if (JomBindingUtil
      .checkExtendsClass(binding, java.lang.Throwable.class.getName())
      == true)
      addMetricValue(TOTAL_NUMBER_OF_THROWABLE, 1, node);

    if (superClass != null) {
      String superClassName = superClass.getName();
      boolean foundDirectSuperClass = false;

      for (int i = 0; i < exceptionClasses.length; i++) {
        if (superClassName.equals(exceptionClasses[i]) == true) {
          addExtendsMetricValue(exceptionClasses[i], true, 1.0f, node);

          foundDirectSuperClass = true;
          break;
        }
      }

      if (foundDirectSuperClass == false) {
        for (int i = 0; i < exceptionClasses.length; i++) {
          if (JomBindingUtil
            .checkExtendsClassIndirectly(superClass, exceptionClasses[i])
            == true) {
            addExtendsMetricValue(exceptionClasses[i], false, 1.0f, node);

            break;
          }
        }
      }

    }

    boolean foundDirectImplementingInterface = false;

    for (int i = 0; i < interfaces.length; i++) {
      IReferenceTypeBinding _interface = interfaces[i];
      String interfaceName = _interface.getName();

      for (int j = 0; j < exceptionInterfaces.length; j++) {
        if (interfaceName.equals(exceptionInterfaces[j]) == true) {
          addImplementsMetricValue(exceptionInterfaces[j], true, 1.0f, node);
          foundDirectImplementingInterface = true;
          break;
        }
      }
    }

    if (foundDirectImplementingInterface == false) {
      for (int i = 0; i < exceptionInterfaces.length; i++) {
        if (JomBindingUtil
          .checkImplementsInterface(binding, exceptionInterfaces[i])
          == true) {
          addImplementsMetricValue(exceptionInterfaces[i], false, 1.0f, node);
          break;
        }
      }
    }

    return true;
  }

  protected void addExtendsMetricValue(
    String exceptionClass,
    boolean direct,
    float value,
    Position pos) {
    if (direct)
      addMetricValue(
        "classes extending " + exceptionClass + " directly",
        value,
        pos);
    else
      addMetricValue(
        "classes extending " + exceptionClass + " indirectly",
        value,
        pos);
  }

  protected void addImplementsMetricValue(
    String interfaceClass,
    boolean direct,
    float value,
    Position pos) {
    if (direct)
      addMetricValue(
        "interfaces implementing " + interfaceClass + " directly",
        value,
        pos);
    else
      addMetricValue(
        "interfaces implementing " + interfaceClass + " indirectly",
        value,
        pos);
  }

  protected void addThrowMetricValue(
    String exceptionClass,
    boolean direct,
    float value,
    Position pos) {
    if (direct)
      addMetricValue(
        "throw-statements throwing an exception class which extends "
          + exceptionClass
          + " directly",
        value,
        pos);
    else
      addMetricValue(
        "throw-statements throwing an exception class which extends "
          + exceptionClass
          + " indirectly",
        value,
        pos);
  }

  protected MetricInfo[] getMetricInfos() {
    List metricInfo = new ArrayList(20);

    for (int i = 0; i < exceptionClasses.length; i++) {
      metricInfo.add(
        new MetricInfo(
          "classes extending " + exceptionClasses[i] + " directly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
      metricInfo.add(
        new MetricInfo(
          "classes extending " + exceptionClasses[i] + " indirectly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
    }

    for (int i = 0; i < exceptionInterfaces.length; i++) {
      metricInfo.add(
        new MetricInfo(
          "interfaces implementing " + exceptionInterfaces[i] + " directly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
      metricInfo.add(
        new MetricInfo(
          "interfaces implementing " + exceptionInterfaces[i] + " indirectly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
    }

    for (int i = 0; i < exceptionClasses.length; i++) {
      metricInfo.add(
        new MetricInfo(
          "throw-statements throwing an exception class which extends "
            + exceptionClasses[i]
            + " directly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
      metricInfo.add(
        new MetricInfo(
          "throw-statements throwing an exception class which extends "
            + exceptionClasses[i]
            + " indirectly",
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
    }

    for (int i = 0; i < exceptionClasses.length; i++) {
      metricInfo.add(
        new MetricInfo(
          "caught exceptions of type " + exceptionClasses[i],
          MetricInfo.SUM,
          ResultInterface.SEVERITY_IGNORE));
    }

    metricInfo.add(new MetricInfo(TOTAL_NUMBER_OF_THROWABLE, MetricInfo.SUM));
    metricInfo.add(
      new MetricInfo(TOTAL_NUMBER_OF_THROW_STATEMENTS, MetricInfo.SUM));
    metricInfo.add(
      new MetricInfo(TOTAL_NUMBER_OF_CATCH_BLOCKS, MetricInfo.SUM));

    return (MetricInfo[]) metricInfo.toArray(new MetricInfo[0]);
  }

}
