package com.sap.tc.jtools.jlint.tests.exception;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Stack;

import com.sap.tc.jtools.jlint.jom.interfaces.*;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jlint.jom.metrics.MetricInfo;

public class CatchBlockTest extends JomMetricVisitor
{
	private static final String EMPTY_CATCH_BLOCK_METRIC = "empty catch blocks" ;
	private static final String INSUFFICENT_EXCEPTION_HANDLING_METRIC = "Missing any exception handling" ;
	private static final String CATCH_BLOCKS_METRIC = "catch blocks" ;

	private static final String MSG_PARAMETER_EXCEPTION = "exceptionName";
	
	private Stack blockStack;

	/* keep in sync with tests.xml! */

	private static final String NAME = "Catch Block Test";

	protected boolean visitCatchClause;
	
	protected boolean visitThrowStatement;

	protected List statements;	
	
	protected IExpressionStatement currentExpressionStatement;
	protected ICatchClause currentCatchClause;
	
	protected HashMap exceptionLifecycleMap;
	
	protected List simpleMethodInvocations;
	
	protected int currentLevel;
	
	protected IBlock currentBlock;

	public String getTestName()
	{
		return NAME;
	}
	
	protected MetricInfo[] getMetricInfos()
	{
		List metricInfo = new ArrayList(20);

		metricInfo.add(new MetricInfo(INSUFFICENT_EXCEPTION_HANDLING_METRIC, MetricInfo.SUM | MetricInfo.AVERAGE));
		metricInfo.add(new MetricInfo(EMPTY_CATCH_BLOCK_METRIC, MetricInfo.SUM));
		metricInfo.add(new MetricInfo(CATCH_BLOCKS_METRIC, MetricInfo.SUM));

		return (MetricInfo[]) metricInfo.toArray(new MetricInfo[0]);
	}

	/**
	 * 
	 */
	public CatchBlockTest()
	{
		statements = new ArrayList();
		blockStack = new Stack();
		
		exceptionLifecycleMap = new HashMap();
		simpleMethodInvocations = new ArrayList();
		
		visitCatchClause = false;
	}

	protected void reset()
	{
		statements.clear();
		
		exceptionLifecycleMap.clear();
		simpleMethodInvocations.clear();
		
		visitCatchClause = false;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause)
	 */
	public boolean visit(ICatchClause node)
	{
		visitCatchClause = true;
		currentCatchClause = node;
		currentLevel = 1;
		
		ExceptionLifecycle exceptionLifecycle = new ExceptionLifecycle(node.getException().resolveBinding(), node.getException());
		exceptionLifecycleMap.put(exceptionLifecycle.getBinding(), exceptionLifecycle);

		return super.visit(node);
	}
	
	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause)
	 */
	public void endVisit(ICatchClause node)
	{
		String nodeName = node.getException().getName(); 
		if(nodeName.equals("s9") == true)
		{
			int t = 0;
		}
		analyseCatchBlockLifecyle(node);
		
		reset();
		currentCatchClause = null;
		
		super.endVisit(node);
	}	

	private void analyseCatchBlockLifecyle(ICatchClause node)
	{	
		CatchBlockLifecycle catchBlockLifecycle = new CatchBlockLifecycle(exceptionLifecycleMap, simpleMethodInvocations, node);

		CatchBlockLifecycleAnalyser catchBlockLifecycleAnalyser = new CatchBlockLifecycleAnalyser(catchBlockLifecycle);
		catchBlockLifecycleAnalyser.analyse();
		
		boolean isPrio1IncorrectStatement = false;

		if(catchBlockLifecycleAnalyser.hasEmptyCatchClause())
		{  
			Properties properties = new Properties();
			properties.setProperty(MSG_PARAMETER_EXCEPTION, node.getException().resolveBinding().getType().getName());
			addError("catchblocktestMsg.7", properties, node);
			isPrio1IncorrectStatement = true;
		}
		else if(catchBlockLifecycleAnalyser.hasThrowStatements() == false)
		{
			if(catchBlockLifecycleAnalyser.hasOnlyReturnStatements() == true)
			{
				addError("catchblocktestMsg.5", null, node);
			}
			else if(catchBlockLifecycleAnalyser.hasOnlyEmptyStatements() == true)
			{
				addError("catchblocktestMsg.6", null, node);
				isPrio1IncorrectStatement = true;
			}
			else if(catchBlockLifecycleAnalyser.hasOnlyWrongHandlingMethods() == true)
			{
				String[][] wrongHandlingMethods = catchBlockLifecycleAnalyser.getWrongHandlingMethods();
				String methodNames = "";
			 
				boolean firstCall = true;
				for(int i = 0; i < wrongHandlingMethods.length; i++)
				{
					if(wrongHandlingMethods[i][2].equals("1") == true)
					{
						if(firstCall == true)
							firstCall = false;						
						else
							methodNames += "; ";
							
						methodNames += wrongHandlingMethods[i][1] + "." + wrongHandlingMethods[i][0] + "()";
					}
				}
					
				Properties properties = new Properties();
				properties.setProperty("methodNames", methodNames);

				addError("catchblocktestMsg.4", properties, node);
				isPrio1IncorrectStatement = true;
			}
			else if(catchBlockLifecycleAnalyser.hasValidExceptionLifecycle() == false)
			{
			  // comment out because itdoesn't work reliably
//				addError("catchblocktestMsg.3", null, node);
			}
		}
		else if(catchBlockLifecycleAnalyser.getIncorrectThrowStatements().size() > 0)
		{
			Iterator incorrectThrowStatementsIt = catchBlockLifecycleAnalyser.getIncorrectThrowStatements().iterator();
			
			while(incorrectThrowStatementsIt.hasNext())
			{
				IThrowStatement throwStatement = (IThrowStatement) incorrectThrowStatementsIt.next();
			
				addError("catchblocktestMsg.2", null, throwStatement);
			}
		}

		if(isPrio1IncorrectStatement)
			addMetricValue(INSUFFICENT_EXCEPTION_HANDLING_METRIC, 1, node);
		else  
			addMetricValue(INSUFFICENT_EXCEPTION_HANDLING_METRIC, 0, node);

	  /* use these statements for see all correct catch blocks in the output
			Properties properties = new Properties();
			properties.setProperty("method", "catch block");
			addError("catchblocktestInfo.1", properties, node);
	  */
		
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IAssertStatement)
	 */
	public boolean visit(IAssertStatement node)
	{
		if(visitCatchClause)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IBreakStatement)
	 */
	public boolean visit(IBreakStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IContinueStatement)
	 */
	public boolean visit(IContinueStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement)
	 */
	public boolean visit(IDoStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
			currentLevel++;
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement)
	 */
	public void endVisit(IDoStatement node)
	{
		if(visitCatchClause == true)
		{
			currentLevel--;
		}
	}


	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IEmptyStatement)
	 */
	public boolean visit(IEmptyStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement)
	 */
	public boolean visit(IExpressionStatement node)
	{
		if(visitCatchClause == true)
		{
			currentExpressionStatement = node;
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
	 */
	public boolean visit(IForStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));
			currentLevel++;	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
	 */
	public void endVisit(IForStatement node)
	{
		if(visitCatchClause == true)
		{
			currentLevel--;	
		}
	}


	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement)
	 */
	public boolean visit(IIfStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
			currentLevel++;
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IForStatement)
	 */
	public void endVisit(IIfStatement node)
	{
		if(visitCatchClause == true)
		{
			currentLevel--;
		}
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement)
	 */
	public boolean visit(ISwitchStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
			currentLevel++;
		}
		
		return true;
	}
	
	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement)
	 */
	public void endVisit(ISwitchStatement node)
	{
		if(visitCatchClause)
		{
			currentLevel--;
		}
	}
	

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ILabeledStatement)
	 */
	public boolean visit(ILabeledStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement)
	 */
	public boolean visit(IReturnStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
			
			analyseExitStatement(node, node.getExpression());
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ISynchronizedStatement)
	 */
	public boolean visit(ISynchronizedStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclarationStatement)
	 */
	public boolean visit(ITypeDeclarationStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement)
	 */
	public boolean visit(IVariableDeclarationStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
			
			if(JomBindingUtil.checkExtendsClass(node.getType().resolveBinding(), java.lang.Throwable.class.getName()) == true)
			{
				// ExceptionLifecyle exceptionLifecycle = new ExceptionLifecyle(currentBlock, node);
				ExceptionLifecycle exceptionLifecycle = new ExceptionLifecycle(node.resolveBinding(), node);
				
				exceptionLifecycleMap.put(exceptionLifecycle.getBinding(), exceptionLifecycle);
			}
		}
		
		return true;
	}

	public boolean visit(IAssignment node)
	{
		if(visitCatchClause == true)
		{
			IExpression rhAssignmentExpression = (IExpression) node.getRightHandSide();
			IExpression lhAssignmentExpression = (IExpression) node.getLeftHandSide();
					
			String rhBindingName = rhAssignmentExpression.resolveTypeBinding().getName();
			String lhBindingName = lhAssignmentExpression.resolveTypeBinding().getName();
			/*
			if(rhAssignmentExpression instanceof IMethodInvocation)
			{
				analyseMethodInvocation((IMethodInvocation) rhAssignmentExpression);	
			}
			*/
			if(JomBindingUtil.checkExtendsClass(lhAssignmentExpression.resolveTypeBinding(), java.lang.Throwable.class.getName()) == true)
			{
				// find the appropriate ExceptionLifeCycle instance
				if(lhAssignmentExpression instanceof ISimpleName)
				{
					ISimpleName simpleName = (ISimpleName) lhAssignmentExpression; 
					String identifier = simpleName.getIdentifier();
					
					ExceptionLifecycle exceptionLifecyle = findExceptionLifecyle(simpleName.resolveBinding());
					if(exceptionLifecyle != null)
					{
						exceptionLifecyle.addAssignment(rhAssignmentExpression);
					}
				}
			}
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation)
	 */
	public boolean visit(IMethodInvocation node)
	{
		if(visitCatchClause == true && visitThrowStatement == false)
		{
			analyseMethodInvocation(node);
		}
		
		return true;
	}

	public boolean visit(IClassInstanceCreation node) {
	  return true;
	}
	
	protected void analyseMethodInvocation(IMethodInvocation node)
	{
			
		IMethodBinding methodBinding = node.resolveMethodBinding();
		String methodBindingName = methodBinding.getName();
		IReferenceTypeBinding declaringClass = node.resolveMethodBinding().getDeclaringClass();
		IExpression expression = node.getExpression();
			
		// check if the respective class is the right one
		if(expression != null &&
		   JomBindingUtil.checkExtendsClass(declaringClass, java.lang.Throwable.class.getName()) == true)	
		{
			if(expression instanceof ISimpleName)
			{
				// find 'e' in e.initCause()
				ISimpleName simpleName = (ISimpleName) expression; 
				String identifier = simpleName.getIdentifier();
						
				ExceptionLifecycle exceptionLifecyle = findExceptionLifecyle(simpleName.resolveBinding());
				if(exceptionLifecyle != null)
				{
					exceptionLifecyle.addMethodInvocation(node);
				}
			}
		}
		else
		{
			IExpression argument = analyseMethodArguments(node, node.arguments(), java.lang.Throwable.class.getName());			
			// check for Throwable as argument

			if(argument != null)
			{
				if(argument instanceof ISimpleName)
				{
					ISimpleName simpleName = (ISimpleName) argument; 
					String identifier = simpleName.getIdentifier();
						
					ExceptionLifecycle exceptionLifecyle = findExceptionLifecyle(simpleName.resolveBinding());
					if(exceptionLifecyle != null)
					{
						exceptionLifecyle.addArgumentInvocation(node);
					}
				}
			}
		}
		
		// a method a called which throws Exceptions
		IReferenceTypeBinding[] exceptionTypes = node.resolveMethodBinding().getExceptionTypes();
		if(exceptionTypes.length > 0) 
		{
			// TODO: Generate one exitpoint for every exception type???
			ExceptionLifecycle exceptionLifecycle = new ExceptionLifecycle(exceptionTypes[0], exceptionTypes[0].getName(), expression);
			exceptionLifecycle.addExitPoint(currentExpressionStatement);
					
			exceptionLifecycleMap.put(exceptionTypes[0], exceptionLifecycle);
		}
		
		 simpleMethodInvocations.add(node);
	}
	
	public IExpression analyseMethodArguments(IMethodInvocation node, List arguments, String argumentTypeName)
	{
		Iterator argumentIt = arguments.iterator(); 
			
		while(argumentIt.hasNext())
		{
			IExpression argument = (IExpression) argumentIt.next();

			ITypeBinding argumentType = argument.resolveTypeBinding();
			if (!(argumentType instanceof IReferenceTypeBinding))
			  continue;

			IReferenceTypeBinding argumentBindung = (IReferenceTypeBinding) argumentType;
			
			if(argument instanceof ISimpleName)
			{
				ISimpleName simpleArgument = (ISimpleName) argument;
				
				// retreive the name of the identifier of the argument
				String identifier = simpleArgument.getIdentifier();
				
				if (JomBindingUtil.checkExtendsClass(argumentBindung, argumentTypeName))
					return argument;
			}
			else if(argument instanceof IMethodInvocation)
			{
				IMethodInvocation innerMethodInvocation = (IMethodInvocation) argument; 
			
				// check if one of the argument types extends the class argumentTypeName
				if (JomBindingUtil.checkExtendsClass(innerMethodInvocation.resolveMethodBinding().getDeclaringClass(), argumentTypeName))
					return argument;
			}
			else if(argument instanceof IClassInstanceCreation)
			{
				ExceptionLifecycle exceptionLifecycle = handleClassInstanceCreation((IClassInstanceCreation) argument, currentExpressionStatement);
				if(exceptionLifecycle != null)
				{
					exceptionLifecycle.addArgumentInvocation(node);				
				}
				
				if (JomBindingUtil.checkExtendsClass(argumentBindung, argumentTypeName))
					return argument;
			}
		}

		return null;
	}	

	protected ExceptionLifecycle handleClassInstanceCreation(IClassInstanceCreation classCreation, IStatement statement)
	{
		String classCreationTypeName = classCreation.getName().resolveTypeBinding().getName();

		if (JomBindingUtil.checkExtendsClass(classCreation.getName().resolveTypeBinding(), java.lang.Throwable.class.getName()) == true)
		{
			ExceptionLifecycle exceptionLifecycle = new ExceptionLifecycle(classCreation.resolveTypeBinding(), classCreationTypeName, classCreation);
			exceptionLifecycle.addExitPoint(statement);
					
			exceptionLifecycleMap.put(classCreation.resolveTypeBinding(), exceptionLifecycle);
			return exceptionLifecycle;
		}
		
		return null;
	}

	protected void analyseExitStatement(IStatement statement, IExpression expression)
	{
		if(expression instanceof IClassInstanceCreation)
		{
			handleClassInstanceCreation((IClassInstanceCreation) expression, statement);
		}

		if(expression instanceof ISimpleName)
		{
			ISimpleName simpleName = (ISimpleName) expression; 
			String identifier = simpleName.getIdentifier();
					
			ExceptionLifecycle exceptionLifecyle = findExceptionLifecyle(simpleName.resolveBinding());
			if(exceptionLifecyle != null)
			{
				exceptionLifecyle.addExitPoint(statement);
			}
		}
				
		if(expression instanceof IMethodInvocation)
		{
			IMethodInvocation methodInvocation = (IMethodInvocation) expression; 
						
			// if return type of MethodInvocation is equal to ???
			ITypeBinding returnTypeBinding = methodInvocation.resolveMethodBinding().getReturnType();
			if(returnTypeBinding != null) 
			{
				String returnTypeName = returnTypeBinding.getName();

				ExceptionLifecycle exceptionLifecycle = new ExceptionLifecycle(returnTypeBinding, returnTypeName, expression);
				exceptionLifecycle.addExitPoint(statement);
					
				exceptionLifecycleMap.put(returnTypeBinding, exceptionLifecycle);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement)
	 */
	public boolean visit(IThrowStatement node)
	{
		visitThrowStatement = true;

		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));
			
			analyseExitStatement(node, node.getExpression());
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement)
	 */
	public void endVisit(IThrowStatement node)
	{
		visitThrowStatement = false;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement)
	 */
	public boolean visit(IWhileStatement node)
	{
		if(visitCatchClause == true)
		{
			statements.add(new StatementInfo(node, currentLevel));	
		}
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement)
	 */
	public void endVisit(IWhileStatement node)
	{
		if(visitCatchClause == true)
		{
			currentLevel--;
		}
	}

	public class StatementInfo
	{
		IStatement statement;
		
		int level;
		
		public StatementInfo(IStatement statement, int level)
		{
			this.statement = statement;
			this.level = level;
		}

	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IBlock)
	 */
	public boolean visit(IBlock node)
	{
		blockStack.push(node);
		currentBlock = node;
		
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IBlock)
	 */
	public void endVisit(IBlock node)
	{
		blockStack.pop();
	}

	protected ExceptionLifecycle findExceptionLifecyle(IBinding binding)
	{
		ExceptionLifecycle exceptionLifecyle = (ExceptionLifecycle) exceptionLifecycleMap.get(binding);
		if(exceptionLifecyle != null)
			return exceptionLifecyle;

		return null;
	}
}
