package com.sap.tc.jtools.jlint.tests.exception;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.IEmptyStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;

public class CatchBlockLifecycleAnalyser
{
	protected CatchBlockLifecycle catchBlockLifecycle;
	
	boolean onlyWrongHandlingMethods = false;
	boolean emptyCatchClause = false;
	boolean onlyReturnStatements = false;
	boolean onlyEmptyStatements = false;
	boolean handlingMethod = false;
	boolean validExceptionLifecycle = false;

	List incorrectThrowStatements;
	
	int throwStatementCount = 0;
	int returnStatementCount = 0;
	int emptyStatementCount = 0;
	
	private static String[][] wrongHandlingMethods =
		{ { "println", "java.io.PrintStream", "" },
		  { "printStackTrace", "java.lang.Throwable", "" } };

	private static String[][] insufficentHandlingMethods =
		{ { "errorT", "com.sap.tc.logging.Location" },
		  { "getMessage", "java.lang.Throwable" },
		  { "toString", "java.lang.Throwable" } };
	
	
	public CatchBlockLifecycleAnalyser(CatchBlockLifecycle catchBlockLifecycle)
	{
		this.catchBlockLifecycle = catchBlockLifecycle;
		
		incorrectThrowStatements = new ArrayList();
	}

	public void analyse()
	{
		if(catchBlockLifecycle == null)
			throw new NullPointerException("Attribute catchBlockLifecyle is null");
		
		ICatchClause catchClause = catchBlockLifecycle.getCatchClause();
		
		List statements = catchClause.getBody().statements();
		
		if(statements == null || statements.size() == 0)
		{
			emptyCatchClause = true;
			return;
		}		
		
		countStatements();

		analyseSimpleHandlingMethods();
		
		analyseExceptionLifecycle();
	}
	
	protected void analyseSimpleHandlingMethods()
	{
		if(wrongHandlingMethods == null)
			return;
		
		// analyse handling methods
		int wrongMethodCallCount = 0;

		ICatchClause catchClause = catchBlockLifecycle.getCatchClause();
		List statements = catchClause.getBody().statements();
		Iterator statementIt = statements.iterator();
		
		for(int i = 0; i < wrongHandlingMethods.length; i++)
		{
			wrongHandlingMethods[i][2] = "0";

			IMethodInvocation calledMethod = JomMethodUtil.checkMethodInvocation(statements, wrongHandlingMethods[i][0], 
																				wrongHandlingMethods[i][1], false);
													
			if(calledMethod != null)
			{
				wrongMethodCallCount++;
				wrongHandlingMethods[i][2] = "1";
			}
		}

		if(returnStatementCount == statements.size())
			onlyReturnStatements = true;
		else if(emptyStatementCount == statements.size())
		{
			onlyEmptyStatements = true;
		}
		else
		{
			if(wrongMethodCallCount == statements.size() - returnStatementCount)
				onlyWrongHandlingMethods = true;
		}
	}
	
	protected void analyseExceptionLifecycle()
	{
		HashMap exceptionLifecycleMap = catchBlockLifecycle.getExceptionLifecycleMap(); 
		
		Set keys = exceptionLifecycleMap.keySet();
		Iterator keyIt = keys.iterator();
		int handlingMethodCount = 0;
		int argumentMethodCount = 0;

		int throwExitPointCount = 0;
		int returnExitPointCount = 0;
		int expressionExitPointCount = 0;
		int wrappedCount = 0;

		
		while(keyIt.hasNext())
		{
			IBinding key = (IBinding) keyIt.next();
			ExceptionLifecycle exceptionLifecycle = (ExceptionLifecycle) exceptionLifecycleMap.get(key);
			
			throwExitPointCount += exceptionLifecycle.getThrowExitPointCount();
			returnExitPointCount += exceptionLifecycle.getReturnExitPointCount();
			expressionExitPointCount += exceptionLifecycle.getExpressionExitPointCount();
						
			ExceptionLifecycleAnalyser analyser = new ExceptionLifecycleAnalyser(exceptionLifecycle, catchBlockLifecycle.catchClause.getException());
			analyser.analyse();

			if(analyser.hasHandlingMethod() == true)
			{
				handlingMethodCount++;
				handlingMethod = true;
			}

			if(analyser.isWrapped() == true)
				wrappedCount++;

			incorrectThrowStatements.addAll(analyser.getIncorrectThrowStatements());
//			for DEBUG purposes
//			System.out.println(exceptionLifecycle);
		}
		
		// correct handling means at least one valid exit point or one correct handling method

		// check for a valid handling method
		if(handlingMethodCount > 0 || wrappedCount > 0)
			validExceptionLifecycle = true;
		else
		{
			// check for a valid exit point
			if(throwExitPointCount + expressionExitPointCount > 0)
			{
				// check if at least one of the thrown exception was created correctly
				// find the throwStatement and check the it 
			 
				validExceptionLifecycle = true;
			}
		}
		
	}
	
	protected void countStatements()
	{
		ICatchClause catchClause = catchBlockLifecycle.getCatchClause();
		
		List statements = catchClause.getBody().statements();
		Iterator statementIt = statements.iterator();

		while (statementIt.hasNext())
		{
			IStatement statement = (IStatement) statementIt.next();

			if(statement instanceof IReturnStatement)
				returnStatementCount++;
			else if(statement instanceof IEmptyStatement)
				emptyStatementCount++;				
			else if(statement instanceof IThrowStatement)			
				throwStatementCount++;
		}
		
		if(returnStatementCount == statements.size())
			onlyReturnStatements = true;
		else if(emptyStatementCount == statements.size())
			onlyEmptyStatements = true;
	}
	
	public List getIncorrectThrowStatements()
	{
		return incorrectThrowStatements; 
	}
		
	/**
	 * @return
	 */
	public boolean hasOnlyWrongHandlingMethods()
	{
		return onlyWrongHandlingMethods;
	}
	
	/**
	 * @return
	 */
	public boolean hasEmptyCatchClause()
	{
		return emptyCatchClause;
	}
	
	/**
	 * @return
	 */
	public boolean hasThrowStatements()
	{
		return throwStatementCount > 0;
	}
	
	/**
	 * @return
	 */
	public boolean hasOnlyReturnStatements()
	{
		return onlyReturnStatements;
	}

	/**
	 * @return
	 */
	public boolean hasOnlyEmptyStatements()
	{
		return onlyEmptyStatements;
	}

	public boolean hasHandlingMethod()
	{
		return handlingMethod;
	}
	
	public boolean hasValidExceptionLifecycle()
	{
		return validExceptionLifecycle;
	}
	
	/**
	 * @return
	 */
	public String[][] getWrongHandlingMethods()
	{
		return wrongHandlingMethods;
	}
}
