/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/ejb/RemoteInterfaceTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.ejb;

import java.util.Iterator;
import java.util.List;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IType;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;

/**
 *
 * This JLin test checks if all method parameters and the return type
 * declared on the remote interface of an EJB (i.e. an interface 
 * that extends javax.ejb.EJBObject) implement java.io.Serializable
 * 
 * @author d037913
 */

public class RemoteInterfaceTest extends JomTestVisitor {

  private static final String NAME = "EJB serializable parameters"; //$NON-NLS-1$

  private static final String EJB_OBJECT_NAME = "javax.ejb.EJBObject"; //$NON-NLS-1$
  private static final String SERIALIZABLE_NAME = "java.io.Serializable"; //$NON-NLS-1$

  private static final String MSG_KEY_1 = "ejbremote.1"; //$NON-NLS-1$

  /**
   * @see com.sap.tc.jtools.jlint.jom.JomTestVisitor#getTestName()
   */
  public String getTestName() {
    return NAME;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(ITypeDeclaration)
   */
  public boolean visit(ITypeDeclaration node) {
    IReferenceTypeBinding bnd = node.resolveBinding();
    if (bnd == null || !bnd.isInterface())
      return false;
    // only visit child nodes of interfaces that 
    // extend javax.ejb.EJBObject
    return InheritanceTool.bndImplements(bnd, EJB_OBJECT_NAME);
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IMethodDeclaration)
   */

  public boolean visit(IMethodDeclaration node) {
    List params = node.parameters();
    if (params != null) {
      // all remote interface method parameters must be serializable
      for (Iterator iter = params.iterator(); iter.hasNext();) {
        IVariableDeclaration var = (IVariableDeclaration) iter.next();
        ITypeBinding varBnd = var.resolveBinding().getType();
        checkIfSerializable(varBnd, var);
      }
    }
    // return type must be serializable
    IType returnType = node.getReturnType();
    ITypeBinding bnd = returnType.resolveBinding();
    if (bnd != null) {
      checkIfSerializable(bnd, returnType);
    }
    return true;
  }

  /**
   * Method checkBinding.
   * @param varBnd
   */
  private void checkIfSerializable(ITypeBinding bnd, Position pos) {
    if (bnd.isPrimitive()) {
      // primitive types are serializable per se 
      return;
    }
    if (bnd.isArray()) {
      bnd = ((IArrayTypeBinding) bnd).getElementType();
      if (bnd.isPrimitive())
        return;
    }
    if (!InheritanceTool
      .bndImplements((IReferenceTypeBinding) bnd, SERIALIZABLE_NAME)) {
      addError(MSG_KEY_1, null, pos);
    }
  }

}
