/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/tests/ejb/ImplementationTest.java#2 $
 */

package com.sap.tc.jtools.jlint.tests.ejb;

import java.lang.reflect.Modifier;
import java.util.Properties;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayType;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.util.InheritanceTool;
import com.sap.tc.jtools.jlint.jom.util.NameTool;

/**
 * @author $Author: p4trans $
 *
 * This JLin test checks EJB programming restrictions (see EJB spec)
 * It is triggered for all classes that implement 
 * javax.ejb.EnterpriseBean
 * 
 */
public class ImplementationTest extends JomTestVisitor {

  private static final String NAME = "EJB programming restrictions"; //$NON-NLS-1$

  private static final String MSG_KEY_1 = "ejbimpl.1"; //$NON-NLS-1$
  private static final String MSG_KEY_2 = "ejbimpl.2"; //$NON-NLS-1$
  private static final String MSG_KEY_3 = "ejbimpl.3"; //$NON-NLS-1$
  private static final String MSG_KEY_4 = "ejbimpl.4"; //$NON-NLS-1$
  private static final String MSG_KEY_5 = "ejbimpl.5"; //$NON-NLS-1$
  private static final String MSG_KEY_6 = "ejbimpl.6"; //$NON-NLS-1$
  private static final String MSG_KEY_7 = "ejbimpl.7"; //$NON-NLS-1$
  private static final String MSG_KEY_8 = "ejbimpl.8"; //$NON-NLS-1$
  private static final String MSG_KEY_9 = "ejbimpl.9"; //$NON-NLS-1$

  private static final String PKG_PARAM = "PACKAGE"; //$NON-NLS-1$
  private static final String METHOD_PARAM = "METHOD"; //$NON-NLS-1$
  private static final String FIELD_PARAM = "FIELD"; //$NON-NLS-1$
  private static final String CLASS_PARAM = "CLASS"; //$NON-NLS-1$

  private static final String EJB_CLASS_NAME = "javax.ejb.EnterpriseBean"; //$NON-NLS-1$
  private static final String FINALIZE_METH = "finalize"; //$NON-NLS-1$

  /** fully qualified names of forbidden packages, classes and methods */

  private static final String[] FORBIDDEN_PKGS = { "java.awt", "javax.swing", "java.lang.reflect", "java.lang.ref" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

  private static final String[] FORBIDDEN_CLASSES = { "java.lang.ClassLoader",
    //$NON-NLS-1$
    "java.io.File", //$NON-NLS-1$
    "java.io.FileDescriptor", //$NON-NLS-1$
    "java.lang.SecurityManager", //$NON-NLS-1$
    "java.lang.RuntimePermission", //$NON-NLS-1$
    "java.lang.Thread", //$NON-NLS-1$
    "java.lang.ThreadGroup", //$NON-NLS-1$
    "java.lang.ThreadLocal", //$NON-NLS-1$
    "java.lang.Runtime", //$NON-NLS-1$
    "java.lang.Process", //$NON-NLS-1$
    "java.lang.Compiler" }; //$NON-NLS-1$

  private static final String[] FORBIDDEN_STATIC_METHODS =
    { "java.lang.System.exit",
    //$NON-NLS-1$
    "java.lang.System.setErr", //$NON-NLS-1$
    "java.lang.System.setIn", //$NON-NLS-1$
    "java.lang.System.setOut", //$NON-NLS-1$
    "java.lang.System.load", //$NON-NLS-1$
    "java.lang.System.loadLibrary", //$NON-NLS-1$
    "java.lang.System.gc" }; //$NON-NLS-1$

  private static final String[] FORBIDDEN_STATIC_FIELDS = { "java.lang.System.out", "java.lang.System.err", "java.lang.System.in" }; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

  private boolean isEJB = false;
  private boolean hasConstructor = false;
  private boolean hasNoArgsConstructor = false;

  /**
   * @see com.sap.tc.jtools.jlint.jom.JomTestVisitor#getTestName()
   */
  public String getTestName() {
    return NAME;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(ITypeDeclaration)
   */
  public boolean visit(ITypeDeclaration node) {
    IReferenceTypeBinding bnd = node.resolveBinding();
    if (bnd == null || !bnd.isClass())
      return false;
    if (InheritanceTool.bndImplements(bnd, EJB_CLASS_NAME)) {
      // only visit child nodes of classes that 
      // implement javax.ejb.EnterpriseBean
      isEJB = true;
      checkModifiers(node);
      return true;
    } else
      return false;
  }

  /**
   * checks if bean classes are declared public and not abstract nor final 
   * @param node
   */
  private void checkModifiers(ITypeDeclaration node) {
    int mod = node.getModifiers();
    if (!Modifier.isPublic(mod)) {
      addError(MSG_KEY_1, null, node);
    }
    if (Modifier.isFinal(mod)) {
      addError(MSG_KEY_2, null, node);
    }
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IFieldDeclaration)
   */
  public boolean visit(IFieldDeclaration node) {
    // static fields should be made final
    if (!Modifier.isStatic(node.getModifiers()))
      return true;
    if (!Modifier.isFinal(node.getModifiers())) {
      addError(MSG_KEY_3, null, node);
    }
    return true;
  }

  private void checkForbiddenComponents(ITypeBinding bnd, Position pos) {
    if (bnd == null)
      return;
    checkForbiddenPackages(bnd, pos);
    checkForbiddenClasses(bnd, pos);
  }

  private void checkForbiddenPackages(ITypeBinding bnd, Position pos) {
    ITypeBinding elementBinding = bnd;
    if (bnd.isArray()) {
      elementBinding = ((IArrayTypeBinding) bnd).getElementType();
    }
    if (elementBinding.isPrimitive())
      return;
    for (int i = 0; i < FORBIDDEN_PKGS.length; i++) {
      if (NameTool
        .getFullPackageName((IReferenceTypeBinding) elementBinding)
        .startsWith(FORBIDDEN_PKGS[i])) {
        Properties p = new Properties();
        p.setProperty(PKG_PARAM, FORBIDDEN_PKGS[i]);
        addError(MSG_KEY_4, p, pos);
      }
    }
  }

  private void checkForbiddenClasses(ITypeBinding bnd, Position pos) {
    ITypeBinding elementBinding = bnd;
    if (bnd.isArray()) {
      elementBinding = ((IArrayTypeBinding) bnd).getElementType();
    }
    if (elementBinding.isPrimitive())
      return;
    for (int i = 0; i < FORBIDDEN_CLASSES.length; i++) {
      checkClassRecursive(
        FORBIDDEN_CLASSES[i],
        (IReferenceTypeBinding) elementBinding,
        pos);
    }
  }

  /**
   * Method checkClassRecursive.
   * @param bnd
   */
  private void checkClassRecursive(
    String forbiddenName,
    IReferenceTypeBinding bnd,
    Position pos) {
    if (InheritanceTool.bndExtends(bnd, forbiddenName)
      || InheritanceTool.bndImplements(bnd, forbiddenName)) {
      Properties p = new Properties();
      p.setProperty(CLASS_PARAM, forbiddenName);
      addError(MSG_KEY_5, p, pos);
    }
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IMethodInvocation)
   */
  public boolean visit(IMethodInvocation node) {
    String fullyQalifiedMethod =
      NameTool.getFullClassName(
        node.resolveMethodBinding().getDeclaringClass())
        + "."
        + node.resolveMethodBinding().getName();
    for (int i = 0; i < FORBIDDEN_STATIC_METHODS.length; i++) {
      if (fullyQalifiedMethod.equals(FORBIDDEN_STATIC_METHODS[i])) {
        Properties p = new Properties();
        p.setProperty(METHOD_PARAM, FORBIDDEN_STATIC_METHODS[i]);
        addError(MSG_KEY_6, p, node);
      }
    }
    return true;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IArrayType)
   */
  public boolean visit(IArrayType node) {
    checkForbiddenComponents(node.resolveBinding(), node);
    return true;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IQualifiedName)
   */
  public boolean visit(IQualifiedName node) {
    checkForbiddenComponents(node.resolveTypeBinding(), node);
    return true;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(ISimpleName)
   */
  public boolean visit(ISimpleName node) {
    checkForbiddenComponents(node.resolveTypeBinding(), node);
    return true;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(ISimpleType)
   */
  public boolean visit(ISimpleType node) {
    checkForbiddenComponents(node.resolveBinding(), node);
    return true;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(IFieldAccess)
   */
  public boolean visit(IFieldAccess node) {
    IVariableBinding bnd = node.resolveBinding();
    if (!Modifier.isStatic(node.resolveBinding().getModifiers()))
      return true;
    String fullName =
      NameTool.getFullClassName(bnd.getDeclaringClass()) + "." + node.getName();
    for (int i = 0; i < FORBIDDEN_STATIC_FIELDS.length; i++) {
      if (FORBIDDEN_STATIC_FIELDS[i].equals(fullName)) {
        Properties p = new Properties();
        p.setProperty(FIELD_PARAM, FORBIDDEN_STATIC_FIELDS[i]);
        addError(MSG_KEY_7, p, node);
      }
    }
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration)
   */
  public boolean visit(IMethodDeclaration node) {
    checkFinalize(node);
    // no constructor declared means default constructor is implicitly there
    if (node.isConstructor())
      hasConstructor = true;
    // check for public no-args constructor
    if (node.isConstructor()
      && node.parameters().size() == 0
      && Modifier.isPublic(node.getModifiers())) {
      hasNoArgsConstructor = true;
    }
    return true;
  }

  /**
   * check if node is a finalize method  
   * @param node
   */
  private void checkFinalize(IMethodDeclaration node) {
    if (FINALIZE_METH.equals(node.getName())
      && Modifier.isPublic(node.getModifiers())
      && node.getReturnType().isSimpleType()
      && ISimpleType.VOID.equals(
        ((ISimpleType) node.getReturnType()).getName()[0])
      && node.parameters().size() == 0) {
      addError(MSG_KEY_8, null, node);
    }
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration)
   */
  public void endVisit(ITypeDeclaration node) {
    if (isEJB && hasConstructor && !hasNoArgsConstructor) {
      addError(MSG_KEY_9, null, node);
    }
    // reset
    hasConstructor = false;
    hasNoArgsConstructor = false;
    isEJB = false;
  }

}
