/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/javadiff/tests/APIDiff.java#2 $
 */
 
/** 
 * Open: Create a message if API-Classes are deleted. EndOfCompile-Event is missing, Check at end of package.
 *   Still open: When package is deleted, we do not have an event. Event is only available in nightly-build.
 *   Write perl-tool (diff), generate required input (list of API-classes) here.
 * Jochen will provide DCs that reference all API-Classes. Missing classes are detected there.
 * 
 * Test if a API Class inherits from (depends on?) a non-API Class.
 * The base classes can be tested in any order. Only at the end of compilation we know for all classes,
 * if they are API or not. Collect base classes on each visit and check in "processErrors". 
 * 
 * Write a list with all checked classes and packages. Must not contain double entries.
 * TODO: Add a central exception-list that suppresses warnings.
 * TODO: Improve code classification: distinguish Call/Inheritance. 
 * TODO: Use PublicParts and ACLs as in NWDI to restrict visibility. (private, package, DC, SC, InternalAPI, CustomerAPI)
 * TODO: Generate a message if ACL is violated.
 * TODO: Make DC, SC, Layer/Build environment available in some Properties.
 * DC/Project name is required to make message 28 more specific.
 * TODO: Make context (Eclipse/nightly-build) available in some property. Eclipse can delete temporary
 * files after each run, nightly build must keep them to the end of the nightly build.
 * TODO: Additional test to enforce one-package one-compile rule plus exception list.
 * TODO: Evaluate "deprecated"
 * 
 */
package com.sap.tc.jtools.jlint.javadiff.tests;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.java.JavaFileArrayTestObject;
import com.sap.tc.jtools.jlint.javadiff.comparator.Comparator;
import com.sap.tc.jtools.jlint.javadiff.util.APIFilter;
import com.sap.tc.jtools.jlint.javadiff.wrapper.Wrapper;
import com.sap.tc.jtools.jlint.javaelements.Class;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * JLin test that compares the APIs of the tested classes with
 * reference values (provided as XML files).
 * 
 * @author d034036
 * @author d034003
 *
 */
public class APIDiff extends JomTestVisitor {

  static public final String TEST_NAME = "API Diff"; //$NON-NLS-1$
  static public final String API_SUFFIX = "_api.xml"; //$NON-NLS-1$

  static public final String MSG_KEY_WRAPPING_ERROR = "apidiff.1"; //$NON-NLS-1$
  static public final String MSG_KEY_REF_CLASS_PARSING_ERROR = "apidiff.3"; //$NON-NLS-1$
  static public final String MSG_KEY_EXTRA_FIELD = "apidiff.4"; //$NON-NLS-1$
  static public final String MSG_KEY_MISSING_FIELD = "apidiff.5"; //$NON-NLS-1$
  static public final String MSG_KEY_EXTRA_METHOD = "apidiff.6"; //$NON-NLS-1$
  static public final String MSG_KEY_MISSING_METHOD = "apidiff.7"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_SIGNATURE = "apidiff.8"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_MODIFIERS = "apidiff.9"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_SUPERCLASS = "apidiff.10"; //$NON-NLS-1$
  static public final String MSG_KEY_EXTRA_SI = "apidiff.11"; //$NON-NLS-1$
  static public final String MSG_KEY_MISSING_SI = "apidiff.12"; //$NON-NLS-1$
  static public final String MSG_KEY_EXTRA_CLASS = "apidiff.13"; //$NON-NLS-1$
  static public final String MSG_KEY_MISSING_CLASS = "apidiff.14"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_RETURN_TYPE = "apidiff.15"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_EXCEPTIONS = "apidiff.16"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_FIELD_TYPE = "apidiff.17"; //$NON-NLS-1$
  static public final String MSG_KEY_CHANGED_FIELD_VALUE = "apidiff.18"; //$NON-NLS-1$

  static public final String PARAMETER_REFERENCE_DIRECTORY = "REFERENCE_DIRS"; //$NON-NLS-1$
  static public final String PARAMETER_EXCEPTIONS_LISTS = "EXCEPTION_LISTS"; //$NON-NLS-1$
 
  private String packageName;
  private String[] referenceDirs;
  
  private static String outputDir;
  private boolean inProcessErrors=false;
//  private static ResultInterface[] allMessages;
  private List messages;
  private Comparator comparator;
  private Listener listener;
  private static boolean messageIsGenerated=false;
  private static APIMetaInfo data=null;
  
  public APIDiff() {
  }

  public String getTestName() {
    return TEST_NAME;
  }

  public boolean visit(ITypeDeclaration node) {
	boolean isAPI = false;
	CodeClassification classif = new CodeClassification(referenceDirs);
	String className = node.getName().getIdentifier();
	String packageAndClass = packageName+"."+className;
	boolean isTest = classif.isTest(this.testObject,packageAndClass);
	data.setTestFlag(packageAndClass,isTest);
	isAPI = classif.isApi(this.testObject,packageAndClass);
	data.setAPIFlag(packageAndClass, isAPI);
	if (!isAPI) {
		return false;
	}
	if (!messageIsGenerated){
		super.addError("apidiff.19",null);
		messageIsGenerated=true;
	}
    listener.setClass(node);
    //so that the listener knows which class we are currently testing
    Class currentClass;
    try {
      currentClass = Wrapper.wrap(node);
    } catch (Exception e) {
      logException(e);
      Properties properties = new Properties();
      properties.setProperty("PACKAGE", packageName);
      properties.setProperty("CLASS", className);
      addError(MSG_KEY_WRAPPING_ERROR, properties, node);
      return false;
    }
	// store meta-info
	String superName = currentClass.getSuperclass();
	if (superName!=null){
		data.addAncestor(packageAndClass,superName);
	}
	String[] interfaces = currentClass.getSuperinterfaces();
	for (int i=0; i<interfaces.length; i++){
		data.addAncestor(packageAndClass,interfaces[i]);
	}
	Class referenceClass;
	try {
      referenceClass = getClass(packageName, className, referenceDirs);
      if (referenceClass == null) {
      	// this means the class was not API in the old version => OK
      	// check only inheritance, but do not compare with old version (that does not exist!)
      	System.out.println("found no reference data for class, is it new? "+packageAndClass);
        return false;
      }
      comparator.compareClasses(currentClass, referenceClass);
    } catch (Exception e) {
      logException(e);
      Properties properties = new Properties();
      properties.setProperty("PACKAGE", packageName);
      properties.setProperty("CLASS", className);
      addError(MSG_KEY_REF_CLASS_PARSING_ERROR, properties, node);
    }
    return false;
  }

  public boolean visit(IPackageDeclaration node) {
    String[] nameComponents = node.getName();
    packageName = nameComponents[0];
    for (int i = 1; i < nameComponents.length; i++) {
      packageName += "." + nameComponents[i];
    }
    return true;
  }

  public boolean visit(ICompilationUnit node) {
    comparator = new Comparator();
    listener = new Listener(packageName,this);
    comparator.attachListener(listener);
    comparator.attachFilter(new APIFilter());
    return true;
  }

  private Class getClass(
	String packageName,
	String className,
	String[] referenceDirs)
	throws IOException, XMLParseException, BadTreeException {
		String oneDir;
		Class result=null;
		if (referenceDirs==null){
			return null;
		}
		for (int i=0; i<referenceDirs.length;i++){
			oneDir=referenceDirs[i];
			result=getClass(packageName, className, oneDir);
			if (result!=null){
				return result; // if we do not exit with an exception, we have found the result.
			}
		}
		return result; // in case the reference-Dirs are an empty array.
	}
	
  private Class getClass(
    String packageName,
    String className,
    String referenceDir)
    throws IOException, XMLParseException, BadTreeException {

    String fullClassName;
    if (packageName != null) {
      fullClassName =
        packageName.replace('.', File.separatorChar)
          + File.separator
          + className;
    } else {
      fullClassName = className;
    }
    String absolutePath =
      referenceDir + File.separator + fullClassName + API_SUFFIX;
    File f = new File(absolutePath);
    if (!f.isFile()) {
      return null;
    }
    Reader fr = null;
    StructureTree st = null;
    try {
     fr = new BufferedReader(new InputStreamReader(new FileInputStream(f), XMLConverter.UTF8_ENCODING));
     st = XMLTool.parseReader(fr);
    } finally {
    	if (fr != null) {
          fr.close();
    	}
    }
    return new Class(st);
  }

	/* (non-Javadoc)
	 * @see com.sap.tc.jtools.jlint.TestComponentInterface#setParameters(com.sap.tc.jtools.jtci.interfaces.ParameterInterface[], com.sap.tc.jtools.jtci.TestObject)
	 */
	public void setParameters(ParameterInterface[] parameters,
			TestObject testObject) {
		super.setParameters(parameters, testObject);
		System.out.println("setParameters is called");
		ParameterInterface tmp = getParameter(XMLConverter.PARAMETER_OUTPUT_DIRECTORY);
		outputDir = (String) tmp.getValue();
	    referenceDirs =
	        (String[]) getParameter(PARAMETER_REFERENCE_DIRECTORY).getValue();
		data = new APIMetaInfo(this,outputDir+"/ClassesInNewApi.ser");
		messageIsGenerated=false;
	}
	
    
	public void addError(String key,Properties params, Position pos){
		if (inProcessErrors){
			ResultInterface tmp = new Result(getName(),JavaFileArrayTestObject.JAVA_FILE_LIST,
                "",null,1,"",params);
            messages.add(tmp);
		}
		else{
			super.addError(key, params, pos);	
		}
	}

	public void addError(String key,Properties params){
		if (inProcessErrors){
			ResultInterface tmp = new Result(getName(),JavaFileArrayTestObject.JAVA_FILE_LIST,
				"",null,1,"",params);
			messages.add(tmp);
		}
		else{
			super.addError(key, params);	
		}
	}

	public ResultInterface[] processErrors(ResultInterface[] params){
		super.processErrors(params);
		System.out.println("processErrors is called");
		ResultInterface[] result=null;
		inProcessErrors=true;
		try{
			if (data!=null){
				messages = new ArrayList();
				for (int i=0; i<params.length;i++){
					messages.add(params[i]);
				}
				//allMessages = params;
				if (outputDir!=null){
					data.saveProcessedClasses(outputDir);
					outputDir=null;
				}
				// create additional messages
				data.processAPIFlags();
				// ToDo: Die Dummy-Message wieder rausfiltern. Besser warten, bis der Bug weg ist.
				Object[] tmp = messages.toArray(new ResultInterface[messages.size()]);
				result = (ResultInterface[])tmp;
			}
			else{
				System.out.println("Ignoring call to processErrors before setParameters");
			}
		}
		finally{
			inProcessErrors = false;
		}
		return result;
	}

	
}
