/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_tests/src/com/sap/tc/jtools/jlint/javadiff/BinaryConverter.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javadiff;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import com.sap.tc.jtools.jlint.javadiff.tests.APIDiffUtil;
import com.sap.tc.jtools.jlint.javaelements.Class;
import com.sap.tc.jtools.jlint.javaelements.Field;
import com.sap.tc.jtools.jlint.javaelements.Method;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * This class converts java bytecode into the abstract API representation. 
 * For each top-level class package.subpackage.Classname a corresponding file
 * outputDirectory/package/subpackage/Classname_api.xml is produced.
 * 
 * Jar-files or folders containing .class files are handled.
 * 
 * @author d034036
 * @author d034003
 * 
 */
public class BinaryConverter {

	/**
	 * API file suffix.
	 */
	static public final String API_SUFFIX = "_api.xml"; //$NON-NLS-1$ 
	static public final String OUTFILENAME = "APIclassesAP.properties"; //$NON-NLS-1$ 

	static private final String CLASS_SUFFIX = ".class"; //$NON-NLS-1$ 
		
	private static Properties _classnames=null;
    // key: package.subpackage.Class in a String
    // value: a String indicating success / failure during loading of the class. 
    
    private static Properties _missingClasses=new Properties();
    
    private URL[] apiJars;  // needed to create ClassLoader were needed. 
    // private ClassLoader loader; The classLoader keeps to many references,
    // resulting in an OutOfMemoryError. Therefore it is a local variable 
    // that is cleared and initialized for each Jarfile.
    // API jars and other jars are both in this array.
	
	private String _outputDir;
     
	/** TODO: Put all error classes in a list and try again in the end.
	 * 
	 * @param rootDirsOrJarFiles  Jars or folders that form the API
	 * @param additionalJarFiles  Jars that are required to compile the api, like ejb.jar from sun.
	 * @param outputDir		 root folder that is used to store XML-files and statistics.
	 */
    public BinaryConverter(File[] rootDirsOrJarFiles,File[] additionalJarFiles,String outputDir){
		_outputDir = outputDir;
		_classnames = APIDiffUtil.loadProcessedClasses(outputDir+File.separator+OUTFILENAME);
		int numberOfDirs = rootDirsOrJarFiles.length;
		int numberOfDirs2 = additionalJarFiles.length;
		apiJars = new URL[numberOfDirs+numberOfDirs2];
		try{
			for (int i=0;i<numberOfDirs;i++){
				apiJars[i] = rootDirsOrJarFiles[i].toURL();
			}
			for (int i=0;i<numberOfDirs2;i++){
				apiJars[numberOfDirs+i] = additionalJarFiles[i].toURL();
			}
		}
		catch(MalformedURLException ex){
			ex.printStackTrace();
			throw new RuntimeException(ex.getMessage());
		}
		File oneDir;
		for (int i=0;i<numberOfDirs;i++){
			oneDir = rootDirsOrJarFiles[i];
			System.out.print("About to process "+(i+1)+" of "+numberOfDirs+" folders/jars ");
			System.out.println(oneDir.getName());
			if (oneDir.isDirectory()){
				convertDirectory("",oneDir);
			}
			else{
				processJar(oneDir);
			}
		}
    }

	/** 
	 * A list of all processed files is stored in "outputDir/APIclasses.properties.
	 * The property value indicates, if the class was loaded successfull.
	 *
	 */
	public void saveStatistics(){
		APIDiffUtil.saveProcessedClasses(_outputDir+File.separator+OUTFILENAME,_classnames);
		APIDiffUtil.saveProcessedClasses(_outputDir+File.separator+"missingClasses.properties",_missingClasses);		
	}
	
    private void processJar(File jarfile){
		JarEntry oneEntry;
		ClassLoader loader = new URLClassLoader(apiJars);
    	try{
			JarFile file = new JarFile(jarfile);
			Enumeration allFiles = file.entries();
			while (allFiles.hasMoreElements()){
				Object tmp = allFiles.nextElement();
				oneEntry = (JarEntry)tmp;
				if (oneEntry.isDirectory()) {
					/* subdirectory - ignore, class files are flat in a jarfile. */
				} 
				else {
					String fileName = oneEntry.getName();
					if (isValidClassFileName(fileName)) {
						handleOneClass(fileName,loader);
					}
				}
			}
    	}
		catch(IOException ex){
			System.err.println(jarfile.getAbsolutePath());
			ex.printStackTrace();
			throw new RuntimeException(ex.getMessage());
		}
    }
    
    private static String getTypeName(java.lang.Class param){
    	String result =null;
    	if (param.isArray()){
    		java.lang.Class tmp = param.getComponentType();
    		String tmp2 = getTypeName(tmp);
    		result = tmp2+"[]";
    	}
    	else{
    		result = param.getName();
    	}
    	return result;
    }
    
    /** 
     * As the name suggestes. 
     * @param relativeFilename: package and Name like "com/sap/MyClass.class"
     * @return  com.sap.MyClass
     */
    private String extractClassNameFromRelativeFilename(String relativeFilename){
    	String result = null;
    	// remove ".class", replace "/" with "."
    	int pos = relativeFilename.indexOf(CLASS_SUFFIX);
		String tmp = relativeFilename.substring(0,pos);
		String tmp2 = tmp.replace(File.separatorChar,'.');
    	result = tmp2.replace('/','.');
    	return result;
    }
    
	/** create the output-folders, if they do not exist. 
	 * 
	 * @param fileName Complete absolute name of a classfile, outputDirectory/Package/ClassName.class
	 */
	private void checkOrCreateFolders(String fileName) {
		// remove classname, create folders.
		int pos = fileName.lastIndexOf(File.separatorChar);
		String outdir;
		if (pos>0){
			outdir = fileName.substring(0,pos);
		}
		else{
			outdir = fileName;
		}
		File outputSubdirectory = new File(outdir);
				
		if (!outputSubdirectory.exists()) {
			outputSubdirectory.mkdirs();
		}
	}

    /** convert a java.lang.reflect.Constructor Object to a jlin.Method object.
     * 
     * @param theMethod
     * @return
     */
	private Method constructorToMethod(Constructor theMethod){
		Method result = null;
		String name = theMethod.getName();
		boolean isConstructor = true;
		int modifiers = theMethod.getModifiers();
		String returnType=null;
		String[] parameterTypes;
		String[] exceptions;
		
		java.lang.Class[] tmp = theMethod.getParameterTypes();
		parameterTypes = new String[tmp.length];
		for (int i=0; i<tmp.length; i++){
			java.lang.Class tmp2 = tmp[i];
			parameterTypes[i] = getTypeName(tmp2);
		}
		tmp = theMethod.getExceptionTypes();
		exceptions = new String[tmp.length];
		for (int i=0; i<tmp.length; i++){
			java.lang.Class tmp2 = tmp[i];
			exceptions[i] = getTypeName(tmp2);
		}
		result = new Method(name,isConstructor,modifiers,returnType,parameterTypes,exceptions);
		return result;
	}
	
	/** convert a java.lang.reflect.Method Object to a jlin.Method object.
	 * 
	 * @param theMethod
	 * @return
	 */
	private Method methodToMethod(java.lang.reflect.Method theMethod){
		Method result = null;
		String name = theMethod.getName();
		boolean isConstructor = false;
		int modifiers = theMethod.getModifiers();
		java.lang.Class returnType = theMethod.getReturnType();
		String returnTypeName = getTypeName(returnType);
		String[] parameterTypes;
		String[] exceptions;
		
		java.lang.Class[] tmp = theMethod.getParameterTypes();
		parameterTypes = new String[tmp.length];
		for (int i=0; i<tmp.length; i++){
			java.lang.Class tmp2 = tmp[i];
			parameterTypes[i] = getTypeName(tmp2);
		}
		tmp = theMethod.getExceptionTypes();
		exceptions = new String[tmp.length];
		for (int i=0; i<tmp.length; i++){
			java.lang.Class tmp2 = tmp[i];
			exceptions[i] = getTypeName(tmp2);
		}
		result = new Method(name,isConstructor,modifiers,returnTypeName,parameterTypes,exceptions);
		return result;
	}
	
	/** convert a java.lang.reflect.Field Object to a jlin.Field object.
	 * 
	 * @param theField
	 * @return
	 */	
	private Field fieldToField(java.lang.reflect.Field theField){
    	Field result =null;
    	String name = theField.getName();
    	int modifier = theField.getModifiers();
    	java.lang.Class type = theField.getType();
    	String typeName = getTypeName(type);
    	result = new Field(name,modifier,typeName);
    	return result;
	}
    
	/** convert a java.lang.Class Object to a jlin.Class object.
	 * 
	 * @param theClass
	 * @return
	 */
    private Class classToClass(java.lang.Class theClass){
    	Class result=null;
		java.lang.Class[] tmp;
		String packageName;
		String className;
		String superClassName=null; // not all classes do have a superclass.
		boolean isInterface;
		String[] superInterfaces;
		int modifiers;
		Method[] methods;
		Field[] fields;
		Class[] innerClasses;
		packageName = theClass.getPackage().getName();
		String tmp1 = theClass.getName(); 
		tmp1 = tmp1.substring(packageName.length()+1);
		className = tmp1.replace('$','.');
		java.lang.Class superClass = theClass.getSuperclass();
		if (superClass!=null){
			superClassName=superClass.getName().replace('$','.');
		}
		isInterface = theClass.isInterface();
		modifiers = theClass.getModifiers();
		
		// Interfaces
		tmp = theClass.getInterfaces();
		superInterfaces = new String[tmp.length];
		for (int i=0;i<tmp.length;i++){
			java.lang.Class tmp2 = tmp[i];	
			superInterfaces[i]=tmp2.getName().replace('$','.');
		}
				
		// Constructors and Methods
		Constructor[] tmp5 = theClass.getDeclaredConstructors();
		java.lang.reflect.Method[] tmp4 = theClass.getDeclaredMethods();
		methods = new Method[tmp5.length+tmp4.length];
		for (int i=0;i<tmp5.length;i++){
			Constructor tmp2 = tmp5[i];	
			methods[i] = constructorToMethod(tmp2);
		}
		// Methods		
		for (int i=0;i<tmp4.length;i++){
			java.lang.reflect.Method tmp2 = tmp4[i];	
			methods[tmp5.length+i] = methodToMethod(tmp2);
		}
		
		// Fields
		java.lang.reflect.Field[] tmp3 = theClass.getDeclaredFields();
		fields = new Field[tmp3.length];
		for (int i=0;i<tmp3.length;i++){
			java.lang.reflect.Field tmp2 = tmp3[i];	
			fields[i] = fieldToField(tmp2);
		}
		
		// nested classes
		tmp = theClass.getDeclaredClasses();
		innerClasses = new Class[tmp.length];
		for (int i=0;i<tmp.length;i++){
    		java.lang.Class tmp2 = tmp[i];	
			innerClasses[i] = classToClass(tmp2);
		}
		result = new Class(packageName,className,superClassName,
		superInterfaces,isInterface,modifiers,methods,fields,innerClasses);
     	
    	return result;
    }
        
    /**  
     * load a class and write it out.
     * 
     * @param fileName  relative filename.
     * @param loader    a ClassLoader that knows all required jars.
     */
    private void handleOneClass(String fileName,ClassLoader loader){ 
		/* class file */
		String classnameWithDollar = extractClassNameFromRelativeFilename(fileName);
		String packageAndClassName1 = classnameWithDollar.replace('$','.');
		if (!_classnames.contains(classnameWithDollar)) {
			String status=null;
			Class theSAPClass = null;
			java.lang.Class theJavaClass = null;
			try {
				theJavaClass = java.lang.Class.forName(classnameWithDollar,false,loader);
				status = "success";
			}
			catch(ClassNotFoundException ex){
				status = "NotFound";
				ex.printStackTrace();
				throw new RuntimeException(ex.getMessage());
			}
			catch(NoClassDefFoundError ex){
				// the class file can not be loaded, maybe some other class is missing.
				String msg = ex.getMessage();
				status = "NotLoaded";
				System.err.println(msg);
				_missingClasses.put(msg,"missing");// the class that causes the error
				_missingClasses.put(classnameWithDollar.replace('$','.'),status); // the class that is not analysed because of the error
				System.err.println("required class is not in Classpath. Ignoring "+classnameWithDollar);
				/** TODO: Write to log or create message. Continue work.  */
			}
			catch(SecurityException ex){
				String msg = ex.getMessage();
				status = "SecurityException";
				System.err.println(msg);
				System.err.println("Security exception when loading "+classnameWithDollar);
				/** TODO: how to handle? why does it happen?  */
			}
			catch(IllegalAccessError err){
				String msg = err.getMessage();
				status = "AccessError";
				System.err.println(msg);
				System.err.println("Class is already loaded in an other version? "+classnameWithDollar);
				/** TODO: how to handle? ignore JLin? guarante compatibility? Log! */
			}
			catch(VerifyError err){
				String msg = err.getMessage();
				status = "VerifyError";
				System.err.println(msg);
				System.err.println("Class seems to be inconsistent? "+classnameWithDollar);
				/** TODO: how to handle?  Log! */
			}
			catch(Exception ex){
				String msg = ex.getMessage();
				status=msg;
				System.err.println(msg);
				System.err.println("What other exceptions do I miss? "+classnameWithDollar);
			}
			if (theJavaClass!=null){
				try{
					theSAPClass = classToClass(theJavaClass);
				}
				catch(NoClassDefFoundError ex){
					// the class file can not be loaded, maybe some other class is missing.
					String msg = ex.getMessage();
					status = "NotLoaded2";
					System.err.println(msg);
					_missingClasses.put(msg,"missing2");
					_missingClasses.put(classnameWithDollar.replace('$','.'),status);
					System.err.println("required class is not in Classpath2. Ignoring "+classnameWithDollar);
					/** TODO: Write to log or create message. Continue work.  */
				}
			}
			String packageAndClassName;
			if (theSAPClass!=null){
				String packageName = theSAPClass.getPckName();
				String className=theSAPClass.getClassName();// with "." instead of "$"
				if (packageName!=null){
					packageAndClassName = packageName+"."+className;
				}
				else{
					packageAndClassName = className;
				}
				writeClass(_outputDir,theSAPClass);
				if (!packageAndClassName.equals(packageAndClassName1)){
					System.err.println("packageAndClassname is inconsistent: "+packageAndClassName+ 
						" "+packageAndClassName1);
				}
			}
			else{
				packageAndClassName = packageAndClassName1;
			}
			_classnames.put(packageAndClassName,status);
		}    	
    }
    
	/**
	 * traverses all class files in a directory (recursivly). 
	 * 
	 * @param relativePath package prefix ending with ".", never null, empty string for top level class
	 * @param dir directory to be parsed
	 */
	private void convertDirectory(String relativePath, File dir) {
		/* examine children */
		File[] children = dir.listFiles();
		ClassLoader loader = new URLClassLoader(apiJars);

		for (int i = 0; i < children.length; i++) {
			if (children[i].isDirectory()) {
				/* subdirectory - start recursion */
				String newRelativePath = relativePath + children[i].getName()+ File.separatorChar;
				convertDirectory(newRelativePath, children[i]);
			} 
			else {
				String name = children[i].getName();
				if (isValidClassFileName(name)) {
					handleOneClass(relativePath+name,loader);
				}
			}
		}
	}
        
    /** Check if the class is a top level class and write it.
     * 
     * @param dir       output directory.
     * @param theClass  the class to write.
     */
    private void writeClass(String dir, Class theClass){
    	String currentClass = theClass.getClassName();
		int lastDotSign = currentClass.lastIndexOf('.');
		if (lastDotSign == -1) { //top-level class
			String packageName = theClass.getPckName();
			currentClass = packageName+"."+currentClass;
			String fileName = dir + File.separator + currentClass.replace('.', File.separatorChar)+ API_SUFFIX;
			checkOrCreateFolders(fileName);
			/* get output file */
			File outputFile = new File( fileName );

			/* write API to file */
			StructureTree tmp2 = theClass.toStructureTree();
			FileWriter tmp = null;
			try{
				tmp = new FileWriter(outputFile);
				XMLTool.writeDocument(tmp2,tmp);
			}
			catch (IOException ex){
				ex.printStackTrace();
				throw new RuntimeException(ex.getMessage());
			}
		}    	
    }

	/**
	 * Returns whether a file name is a valid .class file.
	 * Files not ending in ".class" are invalid.
	 * .class files for anonymous inner classes ($<number>) in filename are invalid.
	 * 
	 * @param filename file name
	 * @return whether the file name is a valid .class file
	 */
	private boolean isValidClassFileName(String filename) {
		if ( !filename.endsWith(CLASS_SUFFIX)) {
			return false;
		}
		String fileNameWithoutExtension = filename.substring(0,filename.length()-CLASS_SUFFIX.length());
		StringTokenizer st = new StringTokenizer(fileNameWithoutExtension,"$");
		if ( st.hasMoreTokens() ) {
			while (st.hasMoreTokens()){
				String token = (String)st.nextToken();
				try{
					Integer.parseInt(token);
					return false; // integer --> anonymous local class --> ignore									
				}
				catch(NumberFormatException ex){
					// ignore
				}
			}
		}
		return true;
	}
	
	/**
	 * Test driver to convert binary files to XML-Files.
	 * 
	 * @param args: args[0]: root of the bytecode file tree; 
	 *              args[1]: output directory
	 */
	public static void main(String[] args) {
		String rootFolder = args[0];
		String outputDir = args[1];
		File folder = new File(rootFolder);
		File[] folders = new File[]{folder};
		File[] others = new File[]{};
		BinaryConverter test = new BinaryConverter(folders,others,outputDir);
		test.saveStatistics();
	}


}
