/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.urlcontentaccess;

import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.util.cache.CacheException;
import com.sapportals.wcm.util.cache.CacheFactory;
import com.sapportals.wcm.util.cache.ICache;
import com.sapportals.wcm.util.cache.ICacheEntry;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.mmparser.AbstractPart;
import com.sapportals.wcm.util.mmparser.FilePart;
import com.sapportals.wcm.util.mmparser.HeaderFields;
import com.sapportals.wcm.util.mmparser.MimeMultipartParser;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.*;

/**
 * Handles the content transfer for external URL resources. Supported protocols:
 * http, https, ftp, file. A proxy and a list of hosts that must not use the
 * proxy can be configured. Using https through a proxy is not supported.
 *
 * @author SAP AG
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/util/api/com/sapportals/wcm/util/urlcontentaccess/URLContentAccess.java#15
 *      $ Copyright (c) SAP AG 2001-2002
 */
public final class URLContentAccess implements IURLContentAccess {

  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.urlcontentaccess.URLContentAccess.class);

  private final static String CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS = "/cm/utilities/content_access";
  private final static String CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS_PLUGINS = "/cm/utilities/content_access/content_access_plugins";
  
  private final static String CAP_CLASS = "ContentAccessPlugin";

  /**
   * Use a proxy
   */
  protected static boolean m_useProxy = false;

  /**
   * Non-proxy-hosts list
   */
  protected static String m_nonProxyHosts = null;

  /**
   * The cache ID from configuration
   */
  private static String m_cacheID = null;

  /**
   * The cache instance
   */
  private static ICache m_cache = null;

  /**
   * The single ContentAccess instance
   */
  private static URLContentAccess m_instance = null;

  /**
   * Table of plug-ins
   */
  private static Hashtable m_protocolPlugins = null;

  /**
   * Private constructor
   */
  private URLContentAccess() {
    readConfig();

    if (m_cacheID != null) {
      try {
        m_cache = CacheFactory.getInstance().getCache(m_cacheID);
      }
      catch (Exception ex) {
        log.errorT("URLContentAccess(101)", "Failed to get cache instance" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      }
    }

    // Configuring the URLConnection
    HttpURLConnection.setFollowRedirects(true);
    enableHttps();// Note: Https through proxy is not possible
  }


  /**
   * Read the content for a URL resource. Note: Content from plug-ins is not
   * cached (parameter <code>useCache</code> has no effect).
   *
   * @param cacheID The key to store/access the content in the cache
   * @param url A full URL
   * @param useCache If the cache must be used
   * @return Reference to a content object
   * @exception URLContentAccessException Exception raised in failure situation
   */
  public IURLContent readContent(String cacheID, String url, boolean useCache)
    throws URLContentAccessException {
    if (url == null) {
      throw new java.lang.NullPointerException("Parameter url was null");
    }
    if (cacheID == null) {
      throw new java.lang.NullPointerException("Parameter cacheID was null");
    }

    if (url.toLowerCase().trim().startsWith("file:")) {
      throw new URLContentAccessException("file: protocol is disabled for security reasons");
    }

    try {
      IURLContent content = null;
      IProtocolPlugin plugIn = null;
      InputStream contentIS = null;
      HeaderFields contentHeader = null;
      URL urlObject = null;

      // check for protocol plugins
      int idx = url.indexOf(":");
      if (idx > 0) {
        String protocol = url.substring(0, idx);
        Enumeration keys = m_protocolPlugins.keys();
        while (plugIn == null && keys.hasMoreElements()) {
          String key = (String)keys.nextElement();
          if (protocol.compareToIgnoreCase(key) == 0) {
            Class ppclass = (Class)m_protocolPlugins.get(key);
            try {
              plugIn = (IProtocolPlugin)ppclass.newInstance();
            }
            catch (Exception e) {
              log.errorT("readContent(154)", "Failed to create plugin: " + e.getMessage() + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
            }// catch
          }// if
        }// while
      }// if

      if (plugIn != null) {

        // Get the content from the Plug-in
        String resourceId = url.substring(idx + 1);
        plugIn.init(resourceId);
        contentIS = plugIn.getInputStream();
        contentHeader = plugIn.getHeaderFields();

        if (contentHeader.getHeaderField("content-type") == null) {
          String contentType = com.sapportals.wcm.util.mimetypes.MimeTable.getMimetypeForName(url);
          contentHeader.setHeaderField("content-type", contentType);
        }
        // create a dummy URL object
        urlObject = new URL("file", "", resourceId);

      }

      else {

        // Try to get content from cache
        if (useCache) {
          if (m_cache != null) {
            ICacheEntry e = m_cache.getEntry(cacheID);
            if (e != null) {
              content = (IURLContent)e.getObject();
            }
          }
        }

        // Send request
        urlObject = new URL(url);
        enableProxy(urlObject);
        URLConnection urlConn = urlObject.openConnection();

        if (urlConn instanceof HttpURLConnection) {
          if (useCache && content != null) {
            // Set If-Mofified-Since Header if there is content in the cache
            urlConn.setIfModifiedSince(((URLContent)content).getFetchTime());
          }
          urlConn.connect();
          HttpURLConnection httpConn = (HttpURLConnection)urlConn;
          int retCode = httpConn.getResponseCode();
          if (retCode >= 400) {
            throw new IOException("HTTP Error: " + retCode + " - " + httpConn.getResponseMessage()
               + ", URL=" + url);
          }
          else if (retCode == 304) {
            // Server answers: "Not Modified", return content from cache
            if (log.beDebug()) {
              log.debugT("readContent(209)", "HTTP 304 Not Modified: " + url + ", since: "
                 + new Date(((URLContent)content).getFetchTime()).toString());
            }
            return content;
          }
          else {
            log.debugT("readContent(215)", "HTTP " + retCode + ", URL=" + url);
          }
        }

        // Check if the server's response is a mime-multipart (e.g. SAP Content Server)
        if (urlConn.getContentType() != null && urlConn.getContentType().toLowerCase().startsWith("multipart/form-data")) {
          // Feed URLConnection into MimeMultipartParser
          MimeMultipartParser mmparser = new MimeMultipartParser((HttpURLConnection)urlConn);
          AbstractPart part = mmparser.nextPart();// We just care about the first component
          if (!part.isFile()) {
            throw new IOException("Error downloading from URL. The multipart response contains no file component. URL=" + url);
          }
          else {
            contentHeader = ((FilePart)part).getHeader();// Headers of the mime-multipart component
            contentIS = ((FilePart)part).getInputStream();
          }
        }
        else {
          // Not a multipart response
          contentIS = urlConn.getInputStream();
          contentHeader = new HeaderFields();
          String name;
          for (int i = 1; (name = urlConn.getHeaderFieldKey(i)) != null; i++) {
            contentHeader.setHeaderField(name, urlConn.getHeaderField(name));// HTTP respose headers
          }
          // No content type, then guess it
          if (contentHeader.getHeaderField("content-type") == null) {
            String contentType = com.sapportals.wcm.util.mimetypes.MimeTable.getMimetypeForName(urlObject.getFile());
            contentHeader.setHeaderField("content-type", contentType);
          }
        }
      }

      content = new URLContent(urlObject, contentHeader, contentIS, useCache);

      // insert / update cache
      if (m_cache != null && useCache) {
        try {
          m_cache.addEntry(cacheID, content);
        }
        catch (CacheException ex) {
          log.errorT("readContent(256)", "Cache exception: " + ex.getMessage() + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
        }
      }

      return content;
    }
    catch (IOException ex) {
      throw new URLContentAccessException("Error downloading from URL: " + url + ", " + ex.getMessage(), ex);
    }
    catch (CacheException ex) {
      throw new URLContentAccessException("Error downloading from URL: " + url + ", Cache exception: " + ex.getMessage(), ex);
    }
  }


  /**
   * Read content
   *
   * @param url TBD: Description of the incoming method parameter
   * @param useCache TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception URLContentAccessException Exception raised in failure situation
   */
  public IURLContent readContent(String url, boolean useCache)
    throws URLContentAccessException {
    return readContent(url, url, useCache);
  }


  /**
   * Store content on a HTTP Server
   *
   * @param url The URL
   * @param data The data to send
   * @param mimeType The content type
   * @param contentLength The content length
   * @param fileName The file name, can be null
   * @param httpHeaders Additional HTTP header, can be null
   * @param fileHeaders File header lines, can be null
   * @param useMimeMultipart true: send a mime multipart POST, false: PUT
   *      request
   * @param user Username for HTTP BASIC authentication, can be null
   * @param pass PAssword for HTTP BASIC authentication, can be null
   * @exception URLContentAccessException Exception raised in failure situation
   */
  public void storeContent(String url, InputStream data, String mimeType, long contentLength,
    String fileName, Properties httpHeaders, Properties fileHeaders, boolean useMimeMultipart,
    String user, String pass)
    throws URLContentAccessException {
    if (url == null || data == null) {
      return;
    }

    try {
      if (useMimeMultipart) {
        postFile(url, data, mimeType, contentLength, fileName, httpHeaders, fileHeaders);
      }
      else {
        putFile(url, data, mimeType, contentLength, httpHeaders, fileHeaders, user, pass);
      }
    }
    catch (IOException ex) {
      throw new URLContentAccessException(ex.getMessage(), ex);
    }
  }


  /**
   * Returns a reference to the single instance.
   *
   * @return The instance
   */
  public static synchronized URLContentAccess getInstance() {
    if (m_instance == null) {
      m_instance = new URLContentAccess();
    }
    return m_instance;
  }


  /**
   * Returns a reference to the cache used.
   *
   * @return cacheInstance
   */
  public static ICache getCacheInstance() {
    return m_cache;
  }


  // --------------------- private / protected -----------------------

  /**
   * Enable HTTPS for the java.net.URL class. Sun's JSSE package must be in the
   * class path.
   */
  private void enableHttps() { }


  /**
   * Enable or disable the use of a proxy for a request
   *
   * @param urlObject The <code>URL</code> object
   */
  private synchronized void enableProxy(URL urlObject) {

    /**
     * @todo ??? Enable authentication at the proxy, like this String authString
     *      = "userid:password"; String auth = "Basic " + new
     *      sun.misc.BASE64Encoder().encode(authString.getBytes()); URL url =
     *      new URL(...); URLConnection conn = url.openConnection();
     *      conn.setRequestProperty("Proxy-Authorization", auth);
     */

    String host = urlObject.getHost();
    if (urlObject.getPort() != -1) {
      host = host + ":" + urlObject.getPort();
    }

    // Check the proxy flag and the nonProxyHosts list and enable or disable proxy
    Properties sysProps = System.getProperties();
    if (m_useProxy && m_nonProxyHosts.indexOf(host) == -1) {
      sysProps.put("proxySet", "true");
      sysProps.put("proxyHost", "proxy");// http. ???
      sysProps.put("proxyPort", "8080");
      sysProps.put("ftpProxySet", "true");
      sysProps.put("ftpProxyHost", "proxy");
      sysProps.put("ftpProxyPort", "8080");

      /**
       * @todo System property nonProxyHosts - funktioninert das ?
       *      http://iwdf8628:1080/twiki/bin/view/Know/JavaProxySetting
       */
      //sysProps.putAll("nonProxyHosts", m_nonProxyHosts);

    }
    else {
      sysProps.put("proxySet", "false");
      sysProps.put("proxyHost", "");
      sysProps.put("proxyPort", "");
      sysProps.put("ftpProxySet", "false");
      sysProps.put("ftpProxyHost", "");
      sysProps.put("ftpProxyPort", "");
    }
  }


  /**
   * Read configuration
   */
  private synchronized void readConfig() {
    Properties props;

    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS);
      IConfigurable[] configurables = plugin.getConfigurables();
      if (configurables.length < 1) {
        throw new Exception("no configurable for plugin " + CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS + " found!");
      }
      props = ConfigCrutch.getConfigurableProperties(configurables[0]);
      completeContentAccessProperties(props, CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS_PLUGINS, CAP_CLASS);
    }
    catch (Exception ex) {
      ex.printStackTrace(System.err);
      return;
    }

    // Proxy
    String useProxy = props.getProperty("useproxy");
    m_useProxy = (useProxy != null && useProxy.equals("true"));

    String nonProxyHosts = props.getProperty("noproxyhosts");
    if (nonProxyHosts != null) {
      m_nonProxyHosts = nonProxyHosts;
    }
    else {
      m_nonProxyHosts = "";
    }

    m_cacheID = props.getProperty("cacheid");

    // read plugins
    m_protocolPlugins = new Hashtable();
    String plugins = props.getProperty("plugins");
    if (plugins != null) {
      StringTokenizer st = new StringTokenizer(plugins, ",");
      ClassLoader cl = CrtClassLoaderRegistry.getClassLoader();
      while (st.hasMoreTokens()) {
        String name = st.nextToken().trim();
        String cn = props.getProperty(name + ".class");
        try {
          Class cp = cl.loadClass(cn);
          m_protocolPlugins.put(name, cp);
        }
        catch (Exception e) {
          log.errorT("readConfig(450)", e.getMessage() + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
        }
      }
    }
  }


  private final static String POST_HEADER_CONTENTTYPE = "Content-Type";

  private final static String POST_HEADER_CONTENTLENGTH = "Content-Length";

  private final static String POST_HEADER_CONTENTDISP = "Content-Disposition";

  private final static String POST_HEADER_MMTYPE = "multipart/form-data; boundary=";

  private final static String POST_HEADER_MMDISP = "form-data; name=\"afile\"; filename=\"";


  /**
   * Issue a HTTP multipart-mime POST request. Only one file is supported in the
   * body at this time.
   *
   * @param url TBD: Description of the incoming method parameter
   * @param data TBD: Description of the incoming method parameter
   * @param mimeType TBD: Description of the incoming method parameter
   * @param contentLength TBD: Description of the incoming method parameter
   * @param fileName TBD: Description of the incoming method parameter
   * @param headers TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @exception IOException Exception raised in failure situation
   */
  private void postFile(String url, InputStream data, String mimeType, long contentLength,
    String fileName, Properties headers, Properties param)
    throws IOException {
    if (url == null || data == null) {
      return;
    }

    if (headers == null) {
      headers = new Properties();
    }
    if (param == null) {
      param = new Properties();
    }
    if (fileName == null || fileName.length() == 0) {
      fileName = "file";
    }

    // A "random" boundary string
    String boundary = "SAPPORTALS_WCM-714236519870341953748";

    headers.setProperty(POST_HEADER_CONTENTTYPE, POST_HEADER_MMTYPE + boundary);

    // Note: SAP Content Server specific parameters (file-headers) must be set by the caller
    // (e.g. X-CompID: filename)
    param.setProperty(POST_HEADER_CONTENTDISP, POST_HEADER_MMDISP + fileName + "\"");
    param.setProperty(POST_HEADER_CONTENTTYPE, mimeType);
    //if (contentLength > 0) {
    //  param.setProperty(POST_HEADER_CONTENTLENGTH, String.valueOf(contentLength));
    //}

    // Encode parameters ("headers" for the part)
    String paramString = null;
    if (param != null && param.size() > 0) {
      StringBuffer buf = new StringBuffer();
      Enumeration names = param.propertyNames();
      while (names.hasMoreElements()) {
        String name = (String)names.nextElement();
        String value = param.getProperty(name);
        buf.append(name);
        buf.append(": ");
        buf.append(value);
        buf.append("\r\n");
      }
      buf.append("\r\n");// Empty line separates headers from content
      paramString = buf.toString();
    }

    // Set up connection
    URL urlObject = new URL(url);
    HttpURLConnection con = (HttpURLConnection)urlObject.openConnection();
    con.setRequestMethod("POST");
    con.setDoInput(true);// Must be set to receive HTTP Response
    con.setDoOutput(true);

    // Set HTTP headers
    if (headers != null) {
      Enumeration en = headers.propertyNames();
      while (en.hasMoreElements()) {
        String name = (String)en.nextElement();
        String value = (String)headers.getProperty(name);
        con.setRequestProperty(name, value);
      }
    }

    // Get output stream
    OutputStream outBody = con.getOutputStream();

    // Boundary
    String boundaryLine = "--" + boundary + "\r\n";
    DataOutputStream outChars = new DataOutputStream(outBody);
    outChars.writeBytes(boundaryLine);

    // Write the parameters
    if (paramString != null) {
      outChars.writeBytes(paramString);
    }

    // Write the data
    byte[] buffer = new byte[8 * 1024];
    int bytesRead = -1;
    while ((bytesRead = data.read(buffer, 0, buffer.length)) != -1) {
      outBody.write(buffer, 0, bytesRead);
    }
    data.close();

    // Boundary
    outChars.writeBytes("\r\n");
    outChars.writeBytes(boundaryLine);
    outChars.close();

    outBody.close();

    // Look at the HTTP response code
    int retCode = con.getResponseCode();
    if (retCode >= 400) {
      throw new IOException("HTTP Error: " + retCode + " - " + ((HttpURLConnection)con).getResponseMessage());
    }
  }


  /**
   * Send a file via HTTP PUT
   *
   * @param url TBD: Description of the incoming method parameter
   * @param data TBD: Description of the incoming method parameter
   * @param mimeType TBD: Description of the incoming method parameter
   * @param contentLength TBD: Description of the incoming method parameter
   * @param headers TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @param user TBD: Description of the incoming method parameter
   * @param pass TBD: Description of the incoming method parameter
   * @exception IOException Exception raised in failure situation
   */
  private void putFile(String url, InputStream data, String mimeType, long contentLength,
    Properties headers, Properties param, String user, String pass)
    throws IOException {
    if (url == null || data == null) {
      return;
    }

    // Encode parameters ("headers" for the part)
    String paramString = null;
    long paramLength = 0;
    if (param != null && param.size() > 0) {
      StringBuffer buf = new StringBuffer("");
      Enumeration names = param.propertyNames();
      while (names.hasMoreElements()) {
        String name = (String)names.nextElement();
        String value = param.getProperty(name);
        buf.append(name);
        buf.append(": ");
        buf.append(value);
        buf.append("\r\n");
      }
      buf.append("\r\n");// Empty line separates headers from content
      paramString = buf.toString();
      paramLength = paramString.length();
    }

    if (mimeType == null) {
      mimeType = "application/octet-stream";
    }

    // Content type and length
    headers.setProperty(POST_HEADER_CONTENTTYPE, mimeType);
    if (contentLength > 0) {
      headers.setProperty(POST_HEADER_CONTENTLENGTH, String.valueOf(contentLength + paramLength));
    }
    // Basic authentication
    if (user != null) {
      if (pass == null) {
        pass = "";
      }
      headers.setProperty("Authorization", "Basic "
         + com.sapportals.wcm.util.base64.Base64Encoder.encode(user + ":" + pass));
    }

    // Set up connection
    URL urlObject = new URL(url);
    HttpURLConnection con = (HttpURLConnection)urlObject.openConnection();
    con.setRequestMethod("PUT");
    con.setDoInput(true);// Must be set to receive HTTP Response
    con.setDoOutput(true);

    // Set HTTP headers
    if (headers != null) {
      Enumeration en = headers.propertyNames();
      while (en.hasMoreElements()) {
        String name = (String)en.nextElement();
        String value = (String)headers.getProperty(name);
        con.setRequestProperty(name, value);
      }
    }

    // Get output stream
    OutputStream outBody = con.getOutputStream();

    // Write the parameters
    if (paramString != null) {
      DataOutputStream outChars = new DataOutputStream(outBody);
      outChars.writeBytes(paramString);
      outChars.close();
    }

    // Write the data
    byte[] buffer = new byte[8 * 1024];
    int bytesRead = -1;
    while ((bytesRead = data.read(buffer, 0, buffer.length)) != -1) {
      outBody.write(buffer, 0, bytesRead);
    }
    data.close();
    outBody.close();

    // Look at the HTTP response code
    int retCode = con.getResponseCode();

    log.infoT("putFile(677)", "http response code for put: url (" + retCode + ")");

    if (retCode >= 400) {
      throw new IOException("HTTP Error: " + retCode + " - " + ((HttpURLConnection)con).getResponseMessage());
    }
  }

  private void completeContentAccessProperties(Properties properties, String pluginName, String theClass)
    throws Exception {
    IConfigClientContext context = IConfigClientContext.createContext(
      ConfigCrutch.getConfigServiceUser());
    IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
    IConfigPlugin plugin = cfgManager.getConfigPlugin(pluginName);
    IConfigurable[] configurables = plugin.getConfigurables();
    for (int i = 0; i < configurables.length; i++) {
      if (configurables[i].getConfigClass().getName().equals(theClass)) {
        addConfigurableProperties(properties, configurables[i].getIdValue(), ConfigCrutch.getConfigurableProperties(configurables[i]));
      }
    }
  }

  private void addConfigurableProperties(Properties properties, String baseKey, Properties additional) {
    Enumeration keys = additional.keys();
    while (keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      properties.setProperty(baseKey + '.' + key, additional.getProperty(key));
    }
  }

}

