/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.urlcontentaccess;
import com.sapportals.wcm.util.content.*;

import com.sapportals.wcm.util.mmparser.HeaderFields;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Collections;
import java.util.Date;

/**
 * Used to hold the data and header information for one component returned by or
 * to be sent to the content server.
 *
 * @author SAP AG
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/util/api/com/sapportals/wcm/util/urlcontentaccess/URLContent.java#6
 *      $ Copyright (c) SAP AG 2001-2002
 */
public class URLContent implements IURLContent {

  /**
   * used to store content if content object is cached
   */
  private byte[] m_content = null;

  /**
   * used if content object is NOT cached
   */
  private InputStream m_in = null;

  /**
   * Http response header or mime-multipart header
   */
  private HeaderFields m_header;

  /**
   * The URL
   */
  private String m_url;

  /**
   * The file name
   */
  private String m_filename;

  /**
   * Date and tim when this conten was fetched fomr the server
   */
  private long m_date;

  /**
   * Content from Http source
   */
  private boolean m_isHttp = false;

  /**
   * Content for local file
   */
  private boolean m_isFile = false;

  /**
   * Constructs a new content object
   *
   * @param urlObject Holds the URL that points to the content source
   * @param header HTTP response or multi-part mime headers
   * @param in An <code>InputStream</code> to read the content from
   * @param cached TBD: Description of the incoming method parameter
   * @exception IOException
   */
  public URLContent(URL urlObject, HeaderFields header, InputStream in, boolean cached)
    throws IOException {
    m_header = header;
    m_url = urlObject.toExternalForm();
    m_date = new Date().getTime();

    m_filename = header.getHeaderField("x-compid");// SAP Content Server specific
    if (m_filename == null) {
      m_filename = urlObject.getFile();
      if (m_filename == null) {
        m_filename = "";
      }
    }

    // Store content
    if (cached) {
      if (urlObject.getProtocol().equals("http") || urlObject.getProtocol().equals("https")) {
        m_isHttp = true;
      }
      else if (urlObject.getProtocol().equals("file")) {
        m_isFile = true;
      }
      int read;
      ByteArrayOutputStream bout = new ByteArrayOutputStream(8 * 1024);
      byte[] buf = new byte[8 * 1024];
      while ((read = in.read(buf)) != -1) {
        bout.write(buf, 0, read);
      }
      m_content = bout.toByteArray();
      bout.close();
      in.close();
    }
    else {
      m_in = in;
    }
  }

  /**
   * Returns the input stream to read the content.
   *
   * @return The input stream reference
   */
  public InputStream getInputStream() {
    if (m_in != null) {
      return m_in;
    }
    else {
      return new ByteArrayInputStream(m_content);
    }
  }

  /**
   * Writes the content to the specified output.
   *
   * @param out An output stream reference
   * @exception URLContentAccessException Exception raised in failure situation
   */
  public void writeTo(OutputStream out)
    throws URLContentAccessException {
    try {
      BufferedInputStream in = new BufferedInputStream(getInputStream());
      int read;
      byte[] buf = new byte[8 * 1024];
      while ((read = in.read(buf)) != -1) {
        out.write(buf, 0, read);
      }
      in.close();
    }
    catch (Exception ex) {
      throw new URLContentAccessException("Error writing content.", ex);
    }
  }


  public HeaderFields getHeader() {
    return m_header;
  }

  public void setHeader(HeaderFields header) {
    m_header = header;
  }

  public String getContentType() {
    return m_header.getHeaderField("content-type");
  }

  public long getContentLength() {
    String len = m_header.getHeaderField("content-length");
    if (len == null) {
      return 0;// default
    }
    else {
      return Integer.parseInt(len);
    }
  }

  public String getURL() {
    return m_url;
  }

  public String getFilename() {
    return m_filename;
  }

  public String getEncoding()
    throws ContentException {
    return null;
  }

  // These methods will be available only if useCache was "true". If useCache was "false"
  // both will be "false" for all files.

  public boolean isFile() {
    return m_isFile;
  }

  public boolean isHttp() {
    return m_isHttp;
  }

  public void close() { }


  // --- protected ---

  protected long getFetchTime() {
    return m_date;
  }
}

