/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.trexconfig;

import com.sapportals.config.fwk.*;

import com.sapportals.portal.security.usermanagement.*;

import com.sapportals.wcm.util.config.*;

//for service user security
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.*;

/**
 * TBD: Description of the class.
 */
public class TrexPropertyStore {
  private final static String CFG_PLUGIN_TREX = "/trex";
  private final static String CFG_PLUGIN_TREX_CLIENT = CFG_PLUGIN_TREX + "/client";
  private final static long CFG_LOCK_TIMEOUT = 2L * 60L * 1000L;// 2 minutes
  private final static long CFG_WAIT_TIMEOUT = 4L * 60L * 1000L;// 4 minute

  private static TrexPropertyStore instance = null;
  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.trexconfig.TrexPropertyStore.class);

  private IConfigClientContext configContext = null;
  private IConfigManager configManager = null;

  /**
   * @return an instance of TrexPropertyStore (singleton)
   * @exception Exception Exception raised in failure situation
   */
  public static synchronized TrexPropertyStore getInstance()
    throws Exception {

    if (TrexPropertyStore.instance == null) {
      TrexPropertyStore.instance = new TrexPropertyStore();
    }
    return TrexPropertyStore.instance;
  }

  /**
   * Construct
   *
   * @exception Exception Exception raised in failure situation
   */
  private TrexPropertyStore()
    throws Exception {

    this.configContext = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    this.configManager = com.sapportals.config.fwk.Configuration.getInstance().getConfigManager(configContext);
  }

  /**
   * Set a property on a configurable
   *
   * @param configClass property to be set
   * @param base property to be set
   * @param key property to be set
   * @param value property to be set
   * @exception Exception Exception raised in failure situation
   */
  public void setProperty(String configClass, String base, String key, String value)
    throws Exception {

    if (configClass == null || base == null || key == null || value == null) {
      throw new java.lang.IllegalArgumentException();
    }

    try {
      IConfigPlugin trexPlugin = this.configManager.getAndLockConfigPlugin(CFG_PLUGIN_TREX_CLIENT, CFG_LOCK_TIMEOUT, CFG_WAIT_TIMEOUT);
      try {
        IMutableConfigurable configurable = trexPlugin.getConfigurable(base);
        if (configurable == null) {
          configurable = createConfigurable(trexPlugin, configClass, base);
        }
        configurable.setPropertyValue(key, value);
        trexPlugin.replaceConfigurable(configurable);
      }
      finally {
        trexPlugin.unlock();
      }
    }
    catch (AlreadyExistLockConfigException e) {
      s_log.warningT("setProperty(105)", "setProperty(): plugin " + CFG_PLUGIN_TREX_CLIENT + " can't be locked");
    }
  }

  /**
   * Set multiple properties on a configurable
   *
   * @param configClass properties to be set
   * @param base properties to be set
   * @param properties properties to be set
   * @exception Exception Exception raised in failure situation
   */
  public void setProperties(String configClass, String base, Properties properties)
    throws Exception {

    if (configClass == null || base == null || properties == null) {
      throw new java.lang.IllegalArgumentException();
    }

    try {
      IConfigPlugin trexPlugin = this.configManager.getAndLockConfigPlugin(CFG_PLUGIN_TREX_CLIENT, CFG_LOCK_TIMEOUT, CFG_WAIT_TIMEOUT);
      try {
        IMutableConfigurable configurable = trexPlugin.getConfigurable(base);
        if (configurable == null) {
          configurable = createConfigurable(trexPlugin, configClass, base);
        }
        Enumeration keys = properties.keys();
        while (keys.hasMoreElements()) {
          String key = (String)keys.nextElement();
          String value = properties.getProperty(key);
          if (key != null && value != null) {
            configurable.setPropertyValue(key, value);
          }
        }
        trexPlugin.replaceConfigurable(configurable);
      }
      finally {
        trexPlugin.unlock();
      }
    }
    catch (AlreadyExistLockConfigException e) {
      s_log.warningT("setProperties(146)", "setProperties(): plugin " + CFG_PLUGIN_TREX_CLIENT + " can't be locked");
    }
  }

  /**
   * @param base TBD: Description of the incoming method parameter
   * @param key TBD: Description of the incoming method parameter
   * @return a configurable property
   * @exception Exception Exception raised in failure situation
   */
  public String getProperty(String base, String key)
    throws Exception {

    if (base == null || key == null) {
      throw new java.lang.IllegalArgumentException();
    }

    Properties properties = getProperties(base);
    return properties == null ? null : properties.getProperty(key);
  }

  /**
   * @param base TBD: Description of the incoming method parameter
   * @return the configurable properties
   * @exception Exception Exception raised in failure situation
   */
  public Properties getProperties(String base)
    throws Exception {

    if (base == null) {
      throw new java.lang.IllegalArgumentException();
    }

    IConfigPlugin trexPlugin = this.configManager.getConfigPlugin(CFG_PLUGIN_TREX_CLIENT);
    IConfigurable configurable = trexPlugin.getConfigurable(base);
    return configurable == null ? null : ConfigCrutch.getConfigurableProperties(configurable);
  }

  /**
   * Remove a configurable
   *
   * @param base TBD: Description of the incoming method parameter
   * @exception Exception Exception raised in failure situation
   */
  public void removeConfigurable(String base)
    throws Exception {

    if (base == null) {
      throw new java.lang.IllegalArgumentException();
    }

    try {
      IConfigPlugin trexPlugin = this.configManager.getAndLockConfigPlugin(CFG_PLUGIN_TREX_CLIENT, CFG_LOCK_TIMEOUT, CFG_WAIT_TIMEOUT);
      try {
        IMutableConfigurable configurable = trexPlugin.getConfigurable(base);
        if (configurable != null) {
          trexPlugin.removeConfigurable(configurable);
        }
      }
      finally {
        trexPlugin.unlock();
      }
    }
    catch (AlreadyExistLockConfigException e) {
      s_log.warningT("removeConfigurable(210)", "removeConfigurable(): plugin " + CFG_PLUGIN_TREX_CLIENT + " can't be locked");
    }
  }

  /**
   * @param configClass TBD: Description of the incoming method parameter
   * @return all configurables of a specific config class
   * @exception Exception Exception raised in failure situation
   */
  public String[] listConfigurables(String configClass)
    throws Exception {

    if (configClass == null) {
      throw new java.lang.IllegalArgumentException();
    }

    IConfigPlugin trexPlugin = this.configManager.getConfigPlugin(CFG_PLUGIN_TREX_CLIENT);
    IConfigurable configurables[] = trexPlugin.getConfigurables(configClass, true);
    String configurableIDs[] = new String[configurables.length];
    for (int i = 0; i < configurables.length; i++) {
      configurableIDs[i] = configurables[i].getIdValue();
    }
    return configurableIDs;
  }

  /**
   * Create a configurable
   *
   * @param plugin TBD: Description of the incoming method parameter
   * @param configClass TBD: Description of the incoming method parameter
   * @param name TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception Exception Exception raised in failure situation
   */
  private IMutableConfigurable createConfigurable(IConfigPlugin plugin, String configClass, String name)
    throws Exception {

    IMutableConfigurable configurable = plugin.createConfigurable(name, null, configClass);
    plugin.addConfigurable(configurable);
    return configurable;
  }

}
