package com.sapportals.wcm.util.opensql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;

/*

DB-table KMC_PATCH

string(80) COMP_NAME
string(80) PATCH_NAME
integer    STATUS (0 = running, 1 = done)

primary key: COMP_NAME, PATCH_NAME 
   
*/

/**
 * This tool maintains a DB table with patch information for components that have to run a data migration during startup.
 * The potentially concurrent startup attemps on different cluster nodes are snychronized. That means the patch will run on
 * a single cluster node only. The patch will run in a single transaction to have a consistent DB even in the case the
 * KM startup is interrupted by a J2EE engine shutdown.
 * The following code template must be used in a KM component's startUpImpl() method:
 * <p>
 * <code>
 * boolean success = false;
 * PatchSynchronizer ps = OpenSQLUtil.getPatchSynchronizer("KMC_XXX", "SP6");
 * if (ps.mustRunPatch(connection)) {
 *   try {
 *     ... change DB content via JDBC ...
 * 
 *     success = true;
 *   }
 *   catch (...) {
 *     ... startup must fail because exception occured during patch ...   
 *   }
 *   finally {
 *     ps.done(success, connection);
 *   }
 * }
 * </code>
 */
public class PatchSynchronizer {

	private static Location LOG = Location.getLocation(PatchSynchronizer.class);
	private final static Category CAT = Category.getCategory(Category.APPLICATIONS, "KMC/RF");

	private static final int STATUS_RUNNING = 0;
	private static final int STATUS_DONE = 1;

	private String componentName;
	private String patchName;
	private long startTS = 0L;

	private PatchSynchronizer(String componentName, String patchName) throws SQLException, IllegalArgumentException {
		this.checkLength(componentName);
		this.checkLength(patchName);
		this.componentName = componentName;
		this.patchName = patchName;
	}

	private void checkLength(String s) {
		final int MAX_LENGTH = 80;
		if (s.length() > MAX_LENGTH) {
			throw new IllegalArgumentException("string length exceeds maximum of " + MAX_LENGTH + " characters: " + s);
		}
	}

	static PatchSynchronizer createInstance(String componentName, String patchName)
		throws SQLException, IllegalArgumentException {
		return new PatchSynchronizer(componentName, patchName);
	}

	/**
	 * Returns "true" if the caller must start with the patch job - "false" otherwise.
	 * The caller must call done() when the job is finished.
	 * In case "false" is returned, the method will block the calling thread if the patch is already running on a different cluster node and it
	 * will block until the patch is finished so that the caller can start using the DB immediately after this method returned.
	 * @param conn JDBC connection. This method will set transaction isolation to "read commited" and auto commit to "false".
	 * @exception SLQException internal error
	 */
	public boolean mustRunPatch(Connection conn) throws SQLException {

		conn.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		conn.setAutoCommit(false);

		int status = 0;

		PreparedStatement ps =
			conn.prepareStatement("select status from kmc_patch where comp_name = ? and patch_name = ? for update");
		ps.setString(1, this.componentName);
		ps.setString(2, this.patchName);
		ResultSet rs = ps.executeQuery();
		if (!rs.next()) {
			rs.close();
			ps.close();
			ps = conn.prepareStatement("insert into kmc_patch (comp_name, patch_name, status) values(?,?,?)");
			ps.setString(1, this.componentName);
			ps.setString(2, this.patchName);
			ps.setInt(3, PatchSynchronizer.STATUS_RUNNING);
			ps.executeUpdate();
			ps.close();
			PatchSynchronizer.CAT.infoT(
				PatchSynchronizer.LOG,
				"Database patch job <" + this.componentName + "/" + this.patchName + "> will be executed");
			this.startTS = System.currentTimeMillis();
			return true;
		}
		else {
			status = rs.getInt(1);
			rs.close();
			ps.close();

			if (status == PatchSynchronizer.STATUS_RUNNING) {
				PatchSynchronizer.CAT.infoT(
					PatchSynchronizer.LOG,
					"Database patch job <"
						+ this.componentName
						+ "/"
						+ this.patchName
						+ "> is being executed on a different cluster node. Waiting for result...");
				this.waitUntilDone(conn);
			}
			return false;
		}
	}

	/**
	 * The caller has finished migrating/patching the DB tables successfully or not.
	 * @param success "true" if the patch run successfully.
	 * @param conn JDBC connection.
	 * @exception SQLException internal error
	 */
	public void done(boolean success, Connection conn) throws SQLException {
		PatchSynchronizer.LOG.debugT("done: success = " + success);
		try {
			if (success) {
				PreparedStatement ps =
					conn.prepareStatement("update kmc_patch set status = ? where comp_name = ? and patch_name = ?");
				ps.setInt(1, PatchSynchronizer.STATUS_DONE);
				ps.setString(2, this.componentName);
				ps.setString(3, this.patchName);
				ps.executeUpdate();
				ps.close();
			}
			else {
				PreparedStatement ps = conn.prepareStatement("delete from kmc_patch where comp_name = ? and patch_name = ?");
				ps.setString(1, this.componentName);
				ps.setString(2, this.patchName);
				ps.executeUpdate();
				ps.close();
			}
			conn.commit();
		}
		finally {
			conn.close();
		}
		if (success) {
			PatchSynchronizer.CAT.infoT(
				PatchSynchronizer.LOG,
				"Database patch job <"
					+ this.componentName
					+ "/"
					+ this.patchName
					+ "> finished successfully. duration="
					+ (System.currentTimeMillis() - this.startTS)
					+ " ms");
		}
		else {
			PatchSynchronizer.CAT.infoT(
				PatchSynchronizer.LOG,
				"Database patch job <"
					+ this.componentName
					+ "/"
					+ this.patchName
					+ "> was cancelled. duration="
					+ (System.currentTimeMillis() - this.startTS)
					+ " ms");
		}
	}

	/**
	 * Poll the table entry until status changed or the entry is gone (patch failed).
	 */
	private void waitUntilDone(Connection conn) throws SQLException {
		try {
			boolean done = false;
			while (!done) {
				PreparedStatement ps =
					conn.prepareStatement("select status from kmc_patch where comp_name = ? and patch_name = ? for update");
				ps.setString(1, this.componentName);
				ps.setString(2, this.patchName);
				ResultSet rs = ps.executeQuery();
				if (rs.next()) {
					int status = rs.getInt(1);
					rs.close();
					if (status == PatchSynchronizer.STATUS_DONE) {
						ps.close();
						conn.commit();
						done = true;
					}
					else {
						this.sleepForNSeconds(10);
					}
				}
				else {
					ps.close();
					conn.commit();
					done = true;
				}
			}
		}
		finally {
			conn.close();
		}
	}

	private void sleepForNSeconds(int sec) {
		try {
			Thread.sleep(1000 * sec);
		}
		catch (InterruptedException ex) {
			PatchSynchronizer.LOG.debugT("catched InterruptedException: " + ex.getMessage());
		}
	}

}
