package com.sapportals.wcm.util.opensql;

import com.sap.sql.NativeSQLAccess;
import com.sap.sql.DatabaseServices;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionWrapper;
import java.sql.*;

/**
 * Tool class with static methods to access special OpenSQL features (which are not available in JDBC API)
 * so that we have build dependencies on OpenSQL only in this single class. This class also features some OpenSQL tools
 * and "add-ons" e.g. generation of unique numbers for primary keys (AutoID).
 */
public final class OpenSQLUtil {
  
  /**
   * Returns a {@see AutoID} instance for the give table name.
   * Use the {@link AutoID#getNextID()} method to retrieve unique values
   * for the table's key column.
   * @return a {@see AutoID} instance for the give table name.
   */
  public static AutoID getAutoID(String tableName) throws SQLException {
    try {
      return AutoID.getAutoID(tableName);
    }
    catch (WcmException ex) {
      // TODO: log stacktrace !
      throw new SQLException(ex.getMessage());
    }
  }
  
  /**
   * @see NativeSQLAccess#isTransientSQLError(Connection, SQLException)
   */
  public static boolean isTransientSQLError(Connection conn, SQLException ex) throws SQLException {
    if (conn instanceof JDBCConnectionWrapper) {
      return NativeSQLAccess.isTransientSQLError(
        ((JDBCConnectionWrapper)conn).getContainedConnection(), ex);
    }
    else {
      return NativeSQLAccess.isTransientSQLError(conn, ex);
    }
  }
  
  /**
   * @see DatabaseServices#getUTCTimestamp(Connection)
   * @throws SQLException
   */
  public static Timestamp getUTCTimestamp(Connection conn) throws SQLException {
    return DatabaseServices.getUTCTimestamp(conn);
  }
  
  /**
   * This is a convenience method for:
   * <p>
   * <code>
   * CachedResultSet crs = new CachedResultSet();
   * crs.populate(jdbcResultSet);
   * jdbcResultSet.close();
   * </code>
   * <p>
   * @see CachedResultSet
   */  
  public static CachedResultSet createCachedResultSet(ResultSet jdbcResultSet) throws SQLException {
    CachedResultSet crs = new CachedResultSet();
    crs.populate(jdbcResultSet);
    jdbcResultSet.close();
    return crs;
  }
  
  /**
   * Returns an instance of a helper class for handling a database migration/patch during the startup of a DB component. 
   * @param componentName A unique name for the DB component, e.g. the table prefix of the component or the name of the database
   * dictionary project.
   * @param patchName A unique name for the update that must be done for the component.
   */
  public static PatchSynchronizer getPatchSynchronizer(String componentName, String patchName) throws Exception {
    return PatchSynchronizer.createInstance(componentName, patchName);
  }
}