package com.sapportals.wcm.util.opensql;

import com.sapportals.wcm.util.regex.*;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.*;
import java.util.Collections;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.Calendar;
import java.util.Comparator;

/**
 * A result set implementation that caches the content of a JDBC result set in memory.<p>
 * Features:<br>
 * - Apply filters (works as an additional WHERE clause) for string columns (JDBC datatypes VARCHAR or CLOB).<br>
 * - Order result set by string columns (like an additional ORDER BY clause).<br>
 * <p>
 * Restrictions:<br>
 * - Content of streams is not cached.<br>
 *   Exception: CLOBs are converted to String (max. length = 2048)<br>
 *   This means that getString() can be called for CLOB columns.<br>
 * - Only some of the getXXX() methods are supported (other methods will throw an UnsupportedOperationException).<br>
 */
public class CachedResultSet implements ResultSet {

  /**
   * case-sensitive match of complete String. See {@link String#compareTo(java.lang.String)}.
   */
  public static final IStringMatcher MATCHER_EQUALS = new EqualsMatcher();

  /**
   * Match if the second string is contained in the first. See {@link String#indexOf(java.lang.String)}.
   */
  public static final IStringMatcher MATCHER_SUBSTRING = new SubstringMatcher();

  /**
   * Uses a regular expression for matching. It is also used internally
   * by the {@link #filterLike(ResultSet, String, String)} method which will convert the LIKE expression
   * into a regular expression.
   */
  public static final IStringMatcher MATCHER_REGEXP = new RegExpMatcher();

  /**
   * Interface for a matcher object which is used by filterXXX methods to
   * match a value against a String or Clob column. The match() method will
   * be called for each column that should be filtered for each row in the
   * ResultSet.
   */
  public interface IStringMatcher {
    /**
     * Must return true if the two string values match, false otherwise.
     * @return true if the two string values match, false otherwise.
     * @param columnValue The value of the column
     * @param filterValue The string to match against as specified in the argument
     * of one of the filterXXX() methods.
     */
    boolean match(String columnValue, String filterValue);
  }

  /**
   * case-sensitive match of complete String.
   */
  private static class EqualsMatcher implements IStringMatcher {
    public boolean match(String s1, String s2) {
      return s1.compareTo(s2) == 0;
    }
  }

  /**
   * match() evaluates to true if the second string is contained in the first.
   */
  private static class SubstringMatcher implements IStringMatcher {
    public boolean match(String s1, String s2) {
      return s1.indexOf(s2) > -1;
    }
  }

  /**
   * Uses regular expressions for matching
   */
  private static class RegExpMatcher implements IStringMatcher {
    private Pattern p;
    private String s2;
    /**
     * match using regular expression contained in the second argument.
     */
    public boolean match(String s1, String s2) {
      boolean newPattern = !(this.p != null && this.s2 != null && s2.equals(this.s2));
      if (newPattern) {
        this.p = new Pattern(s2);
        this.s2 = s2;
      }
      return new Matcher(this.p, s1).matches();
    }
  }

  /**
   * Used by sort() method to compare values of string columns lexicographically.
   */
  private class StringColumnComparator implements Comparator {

    private final int columnIndex;
    private final boolean ascending;

    StringColumnComparator(int columnIndex, boolean ascending) {
      if (columnIndex <= 0) {
        throw new IllegalArgumentException("parameter \"columnIndex\" <= 0");
      }
      this.columnIndex = columnIndex;
      this.ascending = ascending;
    }

    /* (non-Javadoc)
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(Object arg0, Object arg1) {
      Object[] row1 = (Object[])arg0;
      Object[] row2 = (Object[])arg1;
      if (row1[this.columnIndex] == null) {
        return -1;
      }
      if (row2[this.columnIndex] == null) {
        return 1;
      }
      // Column types != String or Clob will cause ClassCaseException !
      String s1 = (String)row1[this.columnIndex];
      String s2 = (String)row2[this.columnIndex];
      if (this.ascending) {
        return s1.compareTo(s2);
      }
      else {
        return s2.compareTo(s1);
      }
    }

  }

  // Contains a list of Object-arrays, which are the "rows" of the result set.
  private List list = null;

  private int rowIndex = 0;

  public static ResultSet filterEquals(ResultSet rs, String columnName, String value) throws SQLException {
    if (columnName == null || value == null)
      throw new IllegalArgumentException();
    CachedResultSet fs = new CachedResultSet();
    fs.populate(rs, columnName, value, CachedResultSet.MATCHER_EQUALS);
    return fs;
  }

  public static ResultSet filterEquals(ResultSet rs, String[] columnNames, String[] values) throws SQLException {
    if (columnNames == null || values == null)
      throw new IllegalArgumentException();
    if (columnNames.length == 0 || values.length == 0)
      throw new IllegalArgumentException();
    if (columnNames.length != values.length)
      throw new IllegalArgumentException();

    CachedResultSet fs = new CachedResultSet();
    IStringMatcher matchers[] = new IStringMatcher[columnNames.length];
    for (int i = 0; i < matchers.length; i++) {
      matchers[i] = CachedResultSet.MATCHER_EQUALS;
    }
    fs.populate(rs, columnNames, values, matchers);
    return fs;
  }

  public static ResultSet filterSubstring(ResultSet rs, String columnName, String value) throws SQLException {
    if (columnName == null || value == null)
      throw new IllegalArgumentException();
    CachedResultSet fs = new CachedResultSet();
    fs.populate(rs, columnName, value, CachedResultSet.MATCHER_SUBSTRING);
    return fs;
  }

  public static ResultSet filterSubstring(ResultSet rs, String[] columnNames, String[] values) throws SQLException {
    if (columnNames == null || values == null)
      throw new IllegalArgumentException();
    if (columnNames.length == 0 || values.length == 0)
      throw new IllegalArgumentException();
    if (columnNames.length != values.length)
      throw new IllegalArgumentException();

    CachedResultSet fs = new CachedResultSet();
    IStringMatcher matchers[] = new IStringMatcher[columnNames.length];
    for (int i = 0; i < matchers.length; i++) {
      matchers[i] = CachedResultSet.MATCHER_SUBSTRING;
    }
    fs.populate(rs, columnNames, values, matchers);
    return fs;
  }

  public static ResultSet filterRegExp(ResultSet rs, String columnName, String regExp) throws SQLException {
    if (columnName == null || regExp == null)
      throw new IllegalArgumentException();
    CachedResultSet fs = new CachedResultSet();
    fs.populate(rs, columnName, regExp, CachedResultSet.MATCHER_REGEXP);
    return fs;
  }

  public static ResultSet filterLike(ResultSet rs, String columnName, String value) throws SQLException {
    if (columnName == null || value == null)
      throw new IllegalArgumentException();
    CachedResultSet fs = new CachedResultSet();
    fs.populate(rs, columnName, CachedResultSet.convertLike2RegExp(value),
                CachedResultSet.MATCHER_REGEXP);
    return fs;
  }


  /**
   * Populates this ResultSet with the data from the specified (jdbc) ResultSet.
   * This method can be called more than once with different ResultSet objects.
   * Each call will append the data.
   * @param rs The jdbc ResultSet to copy the data from - will not be closed (only getMetaData(),
   * next() and getObject() methods are called).
   * @throws SQLException
   */
  public void populate(ResultSet rs) throws SQLException {
    if (this.list == null) {
      this.list = new ArrayList();
    }
    int columnCount = rs.getMetaData().getColumnCount();

    while (rs.next()) {
      Object[] row = new Object[columnCount];
      for (int i = 0; i < columnCount; i++) {
        row[i] = rs.getObject(i + 1);
        if (row[i] != null && row[i] instanceof Clob) {
          row[i] = this.convertToString(row[i]);
        }
      }
      this.list.add(row);
    }
  }

  /**
   * Populate this ResultSet with data from the specified JDBC ResultSet.
   * This method can be called more than once with different ResultSet objects.
   * Each call will append the data.
   * The JDBC ResultSet is not closed. Only next() and getObject() methods are called.
   * The ResultSet rows will be filtered for the specified column, value
   * and matching algorithm.
   * @param rs result set
   * @param columnName MUST be the name of a String column
   * @param value The string value to match against the column value
   * @param matcher An instance of a matcher class which contains the matching algorithm.
   */
  public void populate(ResultSet rs, String columnName, String value, IStringMatcher matcher) throws SQLException {
    if (this.list != null) {
      throw new IllegalStateException();
    }
    this.list = new ArrayList();
    ResultSetMetaData rsMD = rs.getMetaData();
    int columnCount = rsMD.getColumnCount();

    while (rs.next()) {
      Object[] row = new Object[columnCount];
      boolean match = true;
      for (int i = 0; i < columnCount && match; i++) {
        boolean matchColumn = (rsMD.getColumnName(i + 1).equalsIgnoreCase(columnName));
        row[i] = rs.getObject(i + 1);
        if (row[i] != null) {
          if (matchColumn) {
            if (row[i] instanceof Clob) {
              row[i] = this.convertToString(row[i]);
            }
            match = matcher.match((String)row[i], value);
          }
        }
        else {
          if (matchColumn) {
            match = false;
          }
        }
      }
      if (match) {
        ((ArrayList)this.list).add(row);
      }
    }
  }

  /**
  	* Populate this ResultSet with data from the specified JDBC ResultSet.
  	* This method can be called more than once with different ResultSet objects.
  	* Each call will append the data.
  	* The JDBC ResultSet is not closed. Only next() and getObject() methods are called.
  	* The ResultSet rows will be filtered for the specified columns, values
  	* and matching algorithms (AND operator).
  	* @param rs result set
  	* @param columnNames MUST be the names of String columns
  	* @param values The string values to match against the column value
  	* @param matchers Instances of a matcher class which contain the matching algorithms
  	*/
  public void populate(ResultSet rs, String[] columnNames, String[] values, IStringMatcher matchers[])
    throws SQLException {
    if (this.list != null) {
      throw new IllegalStateException();
    }
    this.list = new ArrayList();
    ResultSetMetaData rsMD = rs.getMetaData();
    int columnCount = rsMD.getColumnCount();

    while (rs.next()) {
      Object[] row = new Object[columnCount];
      boolean match = true;
      for (int i = 0; i < columnCount && match; i++) {
        String colName = rsMD.getColumnName(i + 1);
        row[i] = rs.getObject(i + 1);
        if (row[i] != null) {
          if (row[i] instanceof Clob) {
            row[i] = this.convertToString(row[i]);
          }
          for (int m = 0; m < columnNames.length; m++) {
            if (columnNames[m].equalsIgnoreCase(colName)) {
              match &= matchers[m].match((String)row[i], values[m]);
              break;
            }
          }
        }
        else {
          for (int m = 0; m < columnNames.length; m++) {
            if (columnNames[m].equalsIgnoreCase(colName)) {
              match &= false;
              break;
            }
          }
        }
      }
      if (match) {
        ((ArrayList)this.list).add(row);
      }
    }
  }

  /**
   * Sorts the result set lexicographically by the value of a single string column (JDBC datatype String or Clob).
   * @param columnIndex Index of the column (>= 1)
   * @param ascending Whether sort order is ascending or descending.
   * @exception ClassCastException If the specified column is not of type String or Clob.
   */
  public void sort(int columnIndex, boolean ascending) {
    Collections.sort(this.list, new StringColumnComparator(columnIndex, ascending));
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#absolute(int)
   */
  public boolean absolute(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#afterLast()
   */
  public void afterLast() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#beforeFirst()
   */
  public void beforeFirst() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#cancelRowUpdates()
   */
  public void cancelRowUpdates() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#clearWarnings()
   */
  public void clearWarnings() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#close()
   */
  public void close() throws SQLException {
    this.list = null;
    return;
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#deleteRow()
   */
  public void deleteRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#findColumn(java.lang.String)
   */
  public int findColumn(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#first()
   */
  public boolean first() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getArray(int)
   */
  public Array getArray(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getArray(java.lang.String)
   */
  public Array getArray(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getAsciiStream(int)
   */
  public InputStream getAsciiStream(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getAsciiStream(java.lang.String)
   */
  public InputStream getAsciiStream(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBigDecimal(int, int)
   */
  public BigDecimal getBigDecimal(int arg0, int arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBigDecimal(int)
   */
  public BigDecimal getBigDecimal(int arg0) throws SQLException {
    return (BigDecimal) ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBigDecimal(java.lang.String, int)
   */
  public BigDecimal getBigDecimal(String arg0, int arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBigDecimal(java.lang.String)
   */
  public BigDecimal getBigDecimal(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBinaryStream(int)
   */
  public InputStream getBinaryStream(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBinaryStream(java.lang.String)
   */
  public InputStream getBinaryStream(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBlob(int)
   */
  public Blob getBlob(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBlob(java.lang.String)
   */
  public Blob getBlob(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBoolean(int)
   */
  public boolean getBoolean(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return false;
    }
    else {
      return ((Boolean)o).booleanValue();
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBoolean(java.lang.String)
   */
  public boolean getBoolean(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getByte(int)
   */
  public byte getByte(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return 0;
    }
    else {
      return ((Byte)o).byteValue();
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getByte(java.lang.String)
   */
  public byte getByte(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBytes(int)
   */
  public byte[] getBytes(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getBytes(java.lang.String)
   */
  public byte[] getBytes(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getCharacterStream(int)
   */
  public Reader getCharacterStream(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getCharacterStream(java.lang.String)
   */
  public Reader getCharacterStream(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getClob(int)
   */
  public Clob getClob(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getClob(java.lang.String)
   */
  public Clob getClob(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getConcurrency()
   */
  public int getConcurrency() throws SQLException {
    return ResultSet.CONCUR_READ_ONLY;
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getCursorName()
   */
  public String getCursorName() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDate(int, java.util.Calendar)
   */
  public Date getDate(int arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDate(int)
   */
  public Date getDate(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return null;
    }
    else {
      return (Date)o;
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDate(java.lang.String, java.util.Calendar)
   */
  public Date getDate(String arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDate(java.lang.String)
   */
  public Date getDate(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDouble(int)
   */
  public double getDouble(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getDouble(java.lang.String)
   */
  public double getDouble(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getFetchDirection()
   */
  public int getFetchDirection() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getFetchSize()
   */
  public int getFetchSize() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getFloat(int)
   */
  public float getFloat(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getFloat(java.lang.String)
   */
  public float getFloat(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getInt(int)
   */
  public int getInt(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return 0;
    }
    else {
      return ((Integer)o).intValue();
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getInt(java.lang.String)
   */
  public int getInt(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getLong(int)
   */
  public long getLong(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return 0L;
    }
    else {
      return ((Long)o).longValue();
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getLong(java.lang.String)
   */
  public long getLong(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getMetaData()
   */
  public ResultSetMetaData getMetaData() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getObject(int, java.util.Map)
   */
  public Object getObject(int arg0, Map arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getObject(int)
   */
  public Object getObject(int arg0) throws SQLException {
    return ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getObject(java.lang.String, java.util.Map)
   */
  public Object getObject(String arg0, Map arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getObject(java.lang.String)
   */
  public Object getObject(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getRef(int)
   */
  public Ref getRef(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getRef(java.lang.String)
   */
  public Ref getRef(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getRow()
   */
  public int getRow() throws SQLException {
    return this.rowIndex;
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getShort(int)
   */
  public short getShort(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return 0;
    }
    else {
      return ((Short)o).shortValue();
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getShort(java.lang.String)
   */
  public short getShort(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getStatement()
   */
  public Statement getStatement() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getString(int)
   */
  public String getString(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return null;
    }
    else {
      return (String)o;
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getString(java.lang.String)
   */
  public String getString(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTime(int, java.util.Calendar)
   */
  public Time getTime(int arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTime(int)
   */
  public Time getTime(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return null;
    }
    else {
      return (Time)o;
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTime(java.lang.String, java.util.Calendar)
   */
  public Time getTime(String arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTime(java.lang.String)
   */
  public Time getTime(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTimestamp(int, java.util.Calendar)
   */
  public Timestamp getTimestamp(int arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTimestamp(int)
   */
  public Timestamp getTimestamp(int arg0) throws SQLException {
    Object o = ((Object[])this.list.get(this.rowIndex - 1))[arg0 - 1];
    if (o == null) {
      return null;
    }
    else {
      return (Timestamp)o;
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTimestamp(java.lang.String, java.util.Calendar)
   */
  public Timestamp getTimestamp(String arg0, Calendar arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getTimestamp(java.lang.String)
   */
  public Timestamp getTimestamp(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getType()
   */
  public int getType() throws SQLException {
    return ResultSet.TYPE_FORWARD_ONLY;
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getUnicodeStream(int)
   */
  public InputStream getUnicodeStream(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getUnicodeStream(java.lang.String)
   */
  public InputStream getUnicodeStream(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#getWarnings()
   */
  public SQLWarning getWarnings() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#insertRow()
   */
  public void insertRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#isAfterLast()
   */
  public boolean isAfterLast() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#isBeforeFirst()
   */
  public boolean isBeforeFirst() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#isFirst()
   */
  public boolean isFirst() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#isLast()
   */
  public boolean isLast() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#last()
   */
  public boolean last() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#moveToCurrentRow()
   */
  public void moveToCurrentRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#moveToInsertRow()
   */
  public void moveToInsertRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#next()
   */
  public boolean next() throws SQLException {
    if (this.rowIndex >= this.list.size()) {
      return false;
    }
    else {
      this.rowIndex++;
      return true;
    }
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#previous()
   */
  public boolean previous() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#refreshRow()
   */
  public void refreshRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#relative(int)
   */
  public boolean relative(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#rowDeleted()
   */
  public boolean rowDeleted() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#rowInserted()
   */
  public boolean rowInserted() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#rowUpdated()
   */
  public boolean rowUpdated() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#setFetchDirection(int)
   */
  public void setFetchDirection(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#setFetchSize(int)
   */
  public void setFetchSize(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateAsciiStream(int, java.io.InputStream, int)
   */
  public void updateAsciiStream(int arg0, InputStream arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateAsciiStream(java.lang.String, java.io.InputStream, int)
   */
  public void updateAsciiStream(String arg0, InputStream arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBigDecimal(int, java.math.BigDecimal)
   */
  public void updateBigDecimal(int arg0, BigDecimal arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBigDecimal(java.lang.String, java.math.BigDecimal)
   */
  public void updateBigDecimal(String arg0, BigDecimal arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBinaryStream(int, java.io.InputStream, int)
   */
  public void updateBinaryStream(int arg0, InputStream arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBinaryStream(java.lang.String, java.io.InputStream, int)
   */
  public void updateBinaryStream(String arg0, InputStream arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBoolean(int, boolean)
   */
  public void updateBoolean(int arg0, boolean arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBoolean(java.lang.String, boolean)
   */
  public void updateBoolean(String arg0, boolean arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateByte(int, byte)
   */
  public void updateByte(int arg0, byte arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateByte(java.lang.String, byte)
   */
  public void updateByte(String arg0, byte arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBytes(int, byte[])
   */
  public void updateBytes(int arg0, byte[] arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateBytes(java.lang.String, byte[])
   */
  public void updateBytes(String arg0, byte[] arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateCharacterStream(int, java.io.Reader, int)
   */
  public void updateCharacterStream(int arg0, Reader arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateCharacterStream(java.lang.String, java.io.Reader, int)
   */
  public void updateCharacterStream(String arg0, Reader arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateDate(int, java.sql.Date)
   */
  public void updateDate(int arg0, Date arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateDate(java.lang.String, java.sql.Date)
   */
  public void updateDate(String arg0, Date arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateDouble(int, double)
   */
  public void updateDouble(int arg0, double arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateDouble(java.lang.String, double)
   */
  public void updateDouble(String arg0, double arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateFloat(int, float)
   */
  public void updateFloat(int arg0, float arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateFloat(java.lang.String, float)
   */
  public void updateFloat(String arg0, float arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateInt(int, int)
   */
  public void updateInt(int arg0, int arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateInt(java.lang.String, int)
   */
  public void updateInt(String arg0, int arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateLong(int, long)
   */
  public void updateLong(int arg0, long arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateLong(java.lang.String, long)
   */
  public void updateLong(String arg0, long arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateNull(int)
   */
  public void updateNull(int arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateNull(java.lang.String)
   */
  public void updateNull(String arg0) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateObject(int, java.lang.Object, int)
   */
  public void updateObject(int arg0, Object arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateObject(int, java.lang.Object)
   */
  public void updateObject(int arg0, Object arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateObject(java.lang.String, java.lang.Object, int)
   */
  public void updateObject(String arg0, Object arg1, int arg2) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateObject(java.lang.String, java.lang.Object)
   */
  public void updateObject(String arg0, Object arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateRow()
   */
  public void updateRow() throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateShort(int, short)
   */
  public void updateShort(int arg0, short arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateShort(java.lang.String, short)
   */
  public void updateShort(String arg0, short arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateString(int, java.lang.String)
   */
  public void updateString(int arg0, String arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateString(java.lang.String, java.lang.String)
   */
  public void updateString(String arg0, String arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateTime(int, java.sql.Time)
   */
  public void updateTime(int arg0, Time arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateTime(java.lang.String, java.sql.Time)
   */
  public void updateTime(String arg0, Time arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateTimestamp(int, java.sql.Timestamp)
   */
  public void updateTimestamp(int arg0, Timestamp arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#updateTimestamp(java.lang.String, java.sql.Timestamp)
   */
  public void updateTimestamp(String arg0, Timestamp arg1) throws SQLException {
    throw new UnsupportedOperationException();
  }

  /* (non-Javadoc)
   * @see java.sql.ResultSet#wasNull()
   */
  public boolean wasNull() throws SQLException {
    throw new UnsupportedOperationException();
  }

  private static final long MAX_STRING_LENGTH = 2048L;

  private String convertToString(Object o) throws SQLException, IllegalArgumentException {
    if (o instanceof String) {
      return (String)o;
    }
    else if (o instanceof Clob) {
      return ((Clob)o).getSubString(1, (int)Math.max(((Clob)o).length(), CachedResultSet.MAX_STRING_LENGTH));
    }
    else {
      throw new IllegalArgumentException("Column can not be converted to String, must be JDBC datatype VARCHAR or CLOB");
    }
  }

  private static String convertLike2RegExp(String s) {
    StringBuffer b = new StringBuffer(s.length());
    for (int i = 0, l = s.length(); i < l; i++) {
      char c = s.charAt(i);
      switch (c) {
        case '%':
          b.append('.');
          b.append('*');
          break;
        case '?':
          b.append('.');
          break;
        // escape . and *
        case '.':
          b.append('\\');
          b.append('.');
          break;
        case '*':
          b.append('\\');
          b.append('*');
          break;
        default:
          b.append(c);
      }
    }
    return b.toString();
  }
}