/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.opensql;

import java.sql.*;
import javax.sql.DataSource;
import javax.naming.InitialContext;
import com.sapportals.wcm.WcmException;
import com.sap.tc.logging.Location;

public class AutoID {

  private static Location LOG = Location.getLocation(AutoID.class);
  private final static String auto_id_tbl = "KMC_AUTOID";
  private final static int NAME_LENGTH = 80;
  private long autoId;
  private String name, userTable;
  private static String dataSourceName = null;
  private static Object lock = new Object();
  private static boolean isInitialized = false;

  public synchronized long getNextID() {
    this.autoId++;
    /*
    * retrieve new high id from database when wrapping on the lower 32 bits;
    * this needs to be done as there is no guarantee that the next 2^32
    * low-end ids are uniquely used by this vm
    */
    if ((this.autoId & 0x00000000FFFFFFFFL) == 0) {
      resyncToDatabase();
    }
    return this.autoId;
  }

  private AutoID(String name) {
    this.name = name;
    this.autoId = 0;
  }

  public String toString() {
    return this.name;
  }

  private static java.util.Map autoIDMap = new java.util.HashMap();

  public static void setDataSourceName(String dsn) {
    synchronized (lock) {
      if (!isInitialized) {
        dataSourceName = dsn;
        isInitialized = true;
        lock.notifyAll();
      }
    }
  }

  public static AutoID getAutoID(String tableName) throws WcmException {
    try {
      synchronized (lock) {
        while (!isInitialized) {
          lock.wait();
        }
      }
      synchronized (autoIDMap) {
        AutoID rv = (AutoID)autoIDMap.get(tableName);
        if (rv == null) {
          rv = new AutoID(tableName);
          rv.init();
          autoIDMap.put(tableName, rv);
        }
        return rv;
      }
    }
    catch (Exception e) {
      throw new WcmException("Landscape Service: auto id problem", e);
    }
  }

  private void init() throws WcmException {
    if (dataSourceName == null) {
      throw new WcmException();
    }
    userTable = name.trim().toLowerCase();
    if ((userTable.length() > AutoID.NAME_LENGTH) || (userTable.length() <= 0)) {
      throw new WcmException("invalid length (" + userTable.length() + ") of autoID key");
    }
    syncToDatabase(dataSourceName);
  }

  private void resyncToDatabase() {
    /*
    * need to run synchronously on a separate thread as we don't
    * known whether an open OpenSQL connection does exist within the
    * current thread's context
    */
    class LocalThread extends Thread {
      public void run() {
        try {
          syncToDatabase(dataSourceName);
        }
        catch (WcmException we) {
          AutoID.LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(we));
        }
      }
    }

    LocalThread t = new LocalThread();
    t.start();
    boolean joined = false;
    while (!joined) {
      try {
        t.join();
        joined = true;
      }
      catch (InterruptedException ie) {
        AutoID.LOG.debugT(ie.getMessage());
      }
    }
  }

  /*
  * name is primary key column => "select for update" will lock row
  */
  private final static String SQL_SELECT = "select range from " + auto_id_tbl + " where name = ? for update";
  private final static String SQL_UPDATE = "update " + auto_id_tbl + " set range = ? where name = ?";
  private final static String SQL_INSERT = "insert into " + auto_id_tbl + " (name) values(?)";

  private void syncToDatabase(String dsName) throws WcmException {
    InitialContext ctx = null;
    DataSource ds = null;
    Connection con = null;
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      ctx = new InitialContext();
      if (ctx == null) {
        throw new WcmException("context not available");
      }
      ds = (DataSource)ctx.lookup(dsName);
      if (ds == null) {
        throw new WcmException("data source not available");
      }
      con = ds.getConnection();
      con.setAutoCommit(false);
      con.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

      ps = con.prepareStatement(AutoID.SQL_SELECT);
      ps.setString(1, this.userTable);
      rs = ps.executeQuery();
      boolean found = false;
      int high = 1;
      if (rs.next()) {
        found = true;
        high = rs.getInt(1) + 1;
      }
      rs.close();
      rs = null;
      ps.close();
      ps = null;
      if (found) {
        ps = con.prepareStatement(AutoID.SQL_UPDATE);
        ps.setInt(1, high);
        ps.setString(2, userTable);
      }
      else {
        ps = con.prepareStatement(AutoID.SQL_INSERT);
        ps.setString(1, userTable);
      }
      ps.executeUpdate();
      autoId = (long)high << 32L;
      con.commit();
    }
    catch (javax.naming.NamingException ne) {
      throw new WcmException("context not available");
    }
    catch (SQLException se) {
      try {
        if (con != null) {
          con.rollback();
        }
      }
      catch (SQLException se2) {
        AutoID.LOG.debugT(com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(se2));        
      }
      throw new WcmException("sql trouble: ", se);
    }
    finally {
      try {
        if (rs != null) {
          rs.close();
        }
        if (ps != null) {
          ps.close();
        }
        if (con != null) {
          con.close();
        }
      }
      catch (SQLException se) {
        throw new WcmException("sql trouble in final block: ", se);
      }
    }
  }
}
