/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.map;
import java.lang.ref.*;

import java.util.*;

/**
 * A map that uses soft references for the values. Entries will be garbage
 * collected by the VM if the values are no longer referenced from outside the
 * map. But the map keeps a fixed number of internal "hard" references. This way
 * a fixed number of recently used entries are not removed by the VM. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Id:$
 */
public class SoftHashMap extends AbstractMap {

  /**
   * The internal HashMap that holds the SoftReference
   */
  private final Map hash = new HashMap();

  /**
   * The number of internal "hard" references to hold
   */
  private final int hardSize;

  /**
   * The FIFO list of internal "hard" references
   */
  private final LinkedList hardList = new LinkedList();

  /**
   * Reference queue for cleared SoftReference objects.
   */
  private final ReferenceQueue queue = new ReferenceQueue();


  /**
   * Construct a new SoftHashMap
   *
   * @param hardSize The number of hard references to keep
   */
  public SoftHashMap(int hardSize) {
    this.hardSize = hardSize;
  }

  /**
   * Get an object from the map
   *
   * @param key The key
   * @return TBD: Description of the outgoing return value
   */
  public Object get(Object key) {
    Object result = null;
    SoftReference softRef = (SoftReference)this.hash.get(key);
    if (softRef != null) {
      result = softRef.get();
      if (result == null) {
        // If the value has been garbage collected, remove the entry from the HashMap
        this.hash.remove(key);
      }
      else {
        this.hardList.addFirst(result);
        if (this.hardList.size() > this.hardSize) {
          this.hardList.removeLast();
        }
      }
    }
    return result;
  }

  /**
   * Insert a new mapping
   *
   * @param key The key
   * @param value The value
   * @return The inserted value
   */
  public Object put(Object key, Object value) {
    this.processReferenceQueue();
    return this.hash.put(key, new SoftValue(value, key, this.queue));
  }

  /**
   * Renove a mapping
   *
   * @param key The key
   * @return TBD: Description of the outgoing return value
   */
  public Object remove(Object key) {
    this.processReferenceQueue();
    return this.hash.remove(key);
  }

  /**
   * Remove all mappings
   */
  public void clear() {
    this.hardList.clear();
    this.processReferenceQueue();
    this.hash.clear();
  }

  /**
   * The size of the map
   *
   * @return The number of mappings
   */
  public int size() {
    this.processReferenceQueue();
    return this.hash.size();
  }

  /**
   * Not supported!
   *
   * @return TBD: Description of the outgoing return value
   */
  public Set entrySet() {
    throw new UnsupportedOperationException();
  }


  //////////////////////////////////////////////////////////////////////////////

  /**
   * SoftValue extends SoftReference to include the key from the map.
   */
  private static class SoftValue extends SoftReference {

    private final Object key;

    private SoftValue(Object value, Object key, ReferenceQueue q) {
      super(value, q);
      this.key = key;
    }
  }

  /**
   * Remove garbage collected SoftValue objects from the map
   */
  private void processReferenceQueue() {
    SoftValue sv;
    while ((sv = (SoftValue)this.queue.poll()) != null) {
      this.hash.remove(sv.key);
    }
  }
}
