/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.jdbc.escape;

/**
 * Escape strings so that they can be used in JDBC SQL statements - escape
 * backslash, apostrophe, percent, underscore and all multibyte characters - the
 * result is a string which contains no characters beyond ascii 255 Copyright
 * (c) SAP AG 2001-2002
 *
 * @author Dirk Sabiwalsky
 * @version $Id: //javabas/com.sapportals.wcm/dev/kmgmt/bc.util/dev/src/_private/java/api/com/sapportals/wcm/util/jdbc/escape/Escape.java
 *      $
 */

import java.sql.SQLException;

/**
 * TBD: Description of the class.
 */
public class Escape {
  private final static char ESCAPE = '\\';
  private final static char ESCAPE_REP = '1';

  private final static char APOSTROPHE = '\'';
  private final static char APOSTROPHE_REP = '2';

  private final static char PERCENT = '%';
  private final static char PERCENT_REP = '3';

  private final static char UNDERSCORE = '_';
  private final static char UNDERSCORE_REP = '4';

  private final static char UNICODE_PREFIX1 = (char)0xb1;
  private final static char UNICODE_PREFIX_REP1 = '5';

  private final static char UNICODE_PREFIX2 = (char)0xb2;
  private final static char UNICODE_PREFIX_REP2 = '6';

  private final static char UNICODE_PREFIX3 = (char)0xb3;
  private final static char UNICODE_PREFIX_REP3 = '7';

  private final static char UNICODE_PREFIX4 = (char)0xb4;
  private final static char UNICODE_PREFIX_REP4 = '8';

  public static String escapeSqlString(String s, int maxlen)
    throws SQLException {
    if (s == null) {
      return s;
    }
    int length = s.length();
    if (length < 1) {
      return s;
    }

    StringBuffer result = new StringBuffer(length * 2);
    for (int i = 0; i < length; i++) {
      char c = s.charAt(i);
      switch (c) {
        case ESCAPE:
          result.append(ESCAPE);
          result.append(ESCAPE_REP);
          break;
        case APOSTROPHE:
          result.append(ESCAPE);
          result.append(APOSTROPHE_REP);
          break;
        case PERCENT:
          result.append(ESCAPE);
          result.append(PERCENT_REP);
          break;
        case UNDERSCORE:
          result.append(ESCAPE);
          result.append(UNDERSCORE_REP);
          break;
        case UNICODE_PREFIX1:
          result.append(ESCAPE);
          result.append(UNICODE_PREFIX_REP1);
          break;
        case UNICODE_PREFIX2:
          result.append(ESCAPE);
          result.append(UNICODE_PREFIX_REP2);
          break;
        case UNICODE_PREFIX3:
          result.append(ESCAPE);
          result.append(UNICODE_PREFIX_REP3);
          break;
        case UNICODE_PREFIX4:
          result.append(ESCAPE);
          result.append(UNICODE_PREFIX_REP4);
          break;
        default:
          if (isAsciiChar(c)) {
            result.append(c);
          }
          else {
            result.append(encodeUnicode(c));
          }
      }

      if (maxlen > 0 && result.length() > maxlen) {
        throw new SQLException("SQL name too long");
      }
    }

    return result.toString();
  }

  public static String unescapeSqlString(String s) {
    if (s == null) {
      return s;
    }
    int length = s.length();
    if (length < 1) {
      return s;
    }

    StringBuffer result = new StringBuffer(length);
    boolean escaping = false;
    for (int i = 0; i < length; i++) {
      char c = s.charAt(i);
      if (escaping) {
        switch (c) {
          case ESCAPE_REP:
            result.append(ESCAPE);
            break;
          case APOSTROPHE_REP:
            result.append(APOSTROPHE);
            break;
          case PERCENT_REP:
            result.append(PERCENT);
            break;
          case UNDERSCORE_REP:
            result.append(UNDERSCORE);
            break;
          case UNICODE_PREFIX_REP1:
            result.append(UNICODE_PREFIX1);
            break;
          case UNICODE_PREFIX_REP2:
            result.append(UNICODE_PREFIX2);
            break;
          case UNICODE_PREFIX_REP3:
            result.append(UNICODE_PREFIX3);
            break;
          case UNICODE_PREFIX_REP4:
            result.append(UNICODE_PREFIX4);
            break;
        }
        escaping = false;
      }
      else {
        switch (c) {
          case ESCAPE:
            escaping = true;
            break;
          case UNICODE_PREFIX1:
            result.append(decodeUnicode(s.substring(i + 1, i + 2)));
            i += 1;
            break;
          case UNICODE_PREFIX2:
            result.append(decodeUnicode(s.substring(i + 1, i + 3)));
            i += 2;
            break;
          case UNICODE_PREFIX3:
            result.append(decodeUnicode(s.substring(i + 1, i + 4)));
            i += 3;
            break;
          case UNICODE_PREFIX4:
            result.append(decodeUnicode(s.substring(i + 1, i + 5)));
            i += 4;
            break;
          default:
            result.append(c);
        }
      }
    }

    return result.toString();
  }

  private static boolean isAsciiChar(char c) {
    return (int)c <= 255;
  }

  private static String encodeUnicode(char c) {
    String s = Integer.toHexString((int)c);
    int length = s.length();
    switch (length) {
      case 1:
        return UNICODE_PREFIX1 + s;
      case 2:
        return UNICODE_PREFIX2 + s;
      case 3:
        return UNICODE_PREFIX3 + s;
      case 4:
        return UNICODE_PREFIX4 + s;
      default:
        return "";// impossible case, because characters hold only 16 bit
    }
  }

  private static char decodeUnicode(String s) {
    return (char)Integer.parseInt(s, 16);
  }
}
