/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.jdbc.dbinfo;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * This class provides information about the CM database.
 */
public class DBInfo {

  private final static String DBINFO_FILENAME = "DBInfo.properties";
  private final static String PROPNAME_PATCHID_LIST = "patch-list";

  public final static int DB_TYPE_UNKNOWN = 0;
  public final static int DB_TYPE_MSSQL_2000 = 1;
  public final static int DB_TYPE_ORACLE_817 = 2;

  private final static String DRIVER_MSSQL = "SQLServer";
  private final static String DRIVER_ORACLE = "Oracle";// Datadirect
  private final static String DRIVER_ORACLE2 = "Oracle JDBC driver";// Oracle thin

  /**
   * Returns the current patch level of the CM database. This ID should match
   * the latest available patch ID returned by {@link #getLatestPatchID()}. If the
   * two values do not match then the database must be updated.
   *
   * @param conn A JDBC connection instance
   * @return The current patch ID found in the CM database.
   * @exception Exception Exception raised in failure situation
   */
  public static String getPatchID(Connection conn)
    throws Exception {
    return selectCurrentPatchID(conn);
  }

  /**
   * Returns the type of CM database. The method closes the connection.
   *
   * @param conn A JDBC connection instance
   * @return the type of CM database
   * @exception Exception Exception raised in failure situation
   */
  public static int getDBType(Connection conn)
    throws Exception {
    String driver = conn.getMetaData().getDriverName();
    conn.close();
    if (driver.equals(DBInfo.DRIVER_MSSQL)) {
      return DBInfo.DB_TYPE_MSSQL_2000;
    }
    else if (driver.equals(DBInfo.DRIVER_ORACLE)) {
      return DBInfo.DB_TYPE_ORACLE_817;
    }
    else if (driver.equals(DBInfo.DRIVER_ORACLE2)) {
      return DBInfo.DB_TYPE_ORACLE_817;
    }
    else {
      return DBInfo.DB_TYPE_UNKNOWN;
    }
  }

  /**
   * Returns the latest patch ID. This ID should match the current patch level
   * of the database.
   *
   * @return The ID of the latest available DB patch.
   * @exception Exception Exception raised in failure situation
   */
  public static String getLatestPatchID()
    throws Exception {
    Properties props = new Properties();
    props.load(DBInfo.class.getResourceAsStream(DBInfo.DBINFO_FILENAME));
    String patchIDList = props.getProperty(DBInfo.PROPNAME_PATCHID_LIST);
    if (patchIDList == null) {
      throw new Exception("Property not found: " + DBInfo.PROPNAME_PATCHID_LIST);
    }
    List list = DBInfo.parseValueList(patchIDList, ",");
    if (list.size() == 0) {
      throw new Exception(DBInfo.PROPNAME_PATCHID_LIST + " is empty");
    }
    return (String)list.get(list.size() - 1);
  }

  /**
   * Returns the patch ID with the latest date in the wcm_patch table. The
   * method closes the connection.
   *
   * @param conn A JDBC connection instance
   * @return the patch ID with the latest date in the wcm_patch table
   * @exception Exception Exception raised in failure situation
   */
  private static String selectCurrentPatchID(Connection conn)
    throws Exception {

    Timestamp latest = null;
    String patchLevel = null;

    PreparedStatement ps = conn.prepareStatement("select max(DateAndTime) from wcm_patch");
    ResultSet rs = ps.executeQuery();
    if (rs.next()) {
      latest = rs.getTimestamp(1);
    }
    else {
      throw new Exception("Patch history table is empty");
    }
    rs.close();
    ps.close();

    ps = conn.prepareStatement("select PatchID from wcm_patch where DateAndTime = ?");
    ps.setTimestamp(1, latest);
    rs = ps.executeQuery();
    if (rs.next()) {
      patchLevel = rs.getString(1);
    }
    else {
      throw new Exception("Found no patch ID");
    }
    rs.close();
    ps.close();

    conn.close();

    return patchLevel;
  }

  /**
   * Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private static List parseValueList(String values, String separator) {
    List list = new ArrayList();
    if (values == null || values.length() == 0) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }
}
