/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.jdbc.connectionpool;

import com.sapportals.wcm.util.logging.LoggingFormatter;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Map;

/**
 * This is a wrapper for a JDBC Connection.In the close method it will clean up
 * the used connection and put it back into the pool. Also, it will prevent the
 * client from using a connection that it has returned into the pool. <code>
 * CallableStatment</code> instances are cached. They must no longer be used
 * after the connection has been put back into the pool. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/util/api/com/sapportals/wcm/util/jdbc/connectionpool/JDBCConnectionWrapper.java#17$
 */
public class JDBCConnectionWrapper implements Connection {
  
  private final static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionWrapper.class);
  
  private final Connection realConn;
  private final JDBCConnectionPool pool;
  private boolean isClosed = false;
  private boolean lostNetworkConnection = false;
  private boolean isAutoCommitting;
  
  /**
   * Cache for CallableStatement instances
   */
  private HashMap statementCache;
  
  /**
   * debugging finalizer
   */
  private final Exception ex;
  
  private int pendingStmtCnt = 0;
  
  public JDBCConnectionWrapper(Connection realConn, JDBCConnectionPool pool) throws SQLException {
    this.realConn = realConn;
    this.pool = pool;
    this.isAutoCommitting = realConn.getAutoCommit();
    this.ex = new Exception();       
  }
  
  public Connection getContainedConnection() {
    return this.realConn;
  }
  
  public JDBCConnectionPool getPool() {
    return this.pool;
  }
  
  public void close() throws SQLException {
    this.isClosed = true;
    this.realConn.close();        
  }
  
  public boolean isClosed() throws SQLException {
    return this.isClosed;
  }
  
  public void finalize() {    
    if (!this.isClosed) {
      log.errorT("finalize(98)", "finalizing JDBCConnectionWrapper on unclosed connection!" + " - " + LoggingFormatter.extractCallstack(this.ex));      
      try {
        this.realConn.close();
      }
      catch (Exception ex) {
        // $JL-EXC$
      }                                
    }
  }
  
  public boolean isInOpenSQLMode() {
    return true;
  }
  
  // ---------- Connection -------------
  
  public void clearWarnings() throws SQLException {        
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.clearWarnings();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void commit() throws SQLException {                           
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.commit();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public Statement createStatement() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try{
      return this.realConn.createStatement();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public Statement createStatement(int resultsetType, int resultSetConcurrency) throws SQLException {
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try{
      return this.realConn.createStatement(resultsetType, resultSetConcurrency);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public boolean getAutoCommit() throws SQLException {        
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }    
    return this.isAutoCommitting;
  }
  
  public String getCatalog() throws SQLException {                       
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.getCatalog();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public DatabaseMetaData getMetaData() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.getMetaData();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public int getTransactionIsolation() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.getTransactionIsolation();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public SQLWarning getWarnings() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.getWarnings();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public boolean isReadOnly() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.isReadOnly();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public String nativeSQL(String sql) throws SQLException {                                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {      
      return this.realConn.nativeSQL(sql);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public CallableStatement prepareCall(String sql) throws SQLException {                                                       
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }        
    try {
      return this.realConn.prepareCall(sql);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {                                           
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }    
    try {
      return this.realConn.prepareCall(sql, resultSetType, resultSetConcurrency);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public PreparedStatement prepareStatement(String sql) throws SQLException {                                                    
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.prepareStatement(sql);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {                                            
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.prepareStatement(sql, resultSetType, resultSetConcurrency);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void rollback() throws SQLException {                               
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.rollback();      
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void setAutoCommit(boolean autoCommit) throws SQLException {                    
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      if (autoCommit != this.isAutoCommitting) {        
        this.isAutoCommitting = autoCommit;
        this.realConn.setAutoCommit(autoCommit);
      }
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void setCatalog(String catalog) throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.setCatalog(catalog);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void setReadOnly(boolean readOnly) throws SQLException {                                      
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.setReadOnly(readOnly);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void setTransactionIsolation(int level) throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      if (this.realConn.getTransactionIsolation() != level) {
        this.realConn.setTransactionIsolation(level);
      }
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public Map getTypeMap() throws SQLException {                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      return this.realConn.getTypeMap();
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void setTypeMap(Map map) throws SQLException {                                
    if (this.isClosed) {
      throw new SQLException("Pooled connection is closed");
    }
    try {
      this.realConn.setTypeMap(map);
    }
    catch (SQLException e) {
      evalException(e);
      throw e;
    }
  }
  
  public void remove() {
    this.isClosed = true;
  }
  
  // -------------------------- package private -----------------------------
  
  
  void evalException(SQLException e) {
    if ((e != null) && (!lostNetworkConnection)) {
      String state = e.getSQLState();
      if (state != null && (state.startsWith("08S0")
          || state.startsWith("0800") || state.equals("HY000"))) {
        this.lostNetworkConnection = true;
      }
    }
  }
}
