/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cron;
import com.sapportals.wcm.crt.*;

import com.sapportals.wcm.util.trace.*;
import java.io.*;
import java.text.*;

/**
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Frank Khler
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/util/api/com/sapportals/wcm/util/cron/Crontab.java#7
 *      $
 */

import java.util.*;

/**
 * TBD: Description of the class.
 */
class Crontab extends TimerTask {

  Vector itsList = new Vector();
  Tracer itsTracer = null;
  ICrontabData itsData = null;
  long itsDataLastModified = 0;

  HashMap itsActionClasses = new HashMap();
  Vector itsInitialRepeatedActions = new Vector();
  Vector itsRepeatedActions = new Vector();
  Vector itsFinalRepeatedActions = new Vector();


  /**
   * TBD: Description of the class.
   */
  private class RepeatedActionContainer {

    long itsNextRun;
    long itsFrequency;
    IRepeatedAction itsAction;
    Object itsParameter;

    RepeatedActionContainer(IRepeatedAction theAction, long theFirstRun, long theFrequency, Object theParameter) {
      itsAction = theAction;
      itsNextRun = theFirstRun;
      itsFrequency = theFrequency;
      itsParameter = theParameter;
    }

    public boolean equals(Object obj) {
      if (obj.getClass().isInstance(this)) {
        RepeatedActionContainer aContainer = (RepeatedActionContainer)obj;
        return itsAction.equals(aContainer.itsAction);
      }
      else if (obj.getClass().isInstance(itsAction)) {
        return itsAction.equals(obj);
      }
      else {
        return false;
      }
    }

    public int hashCode() {
      return itsAction.hashCode();
    }
  }


  public Crontab(ICrontabData data) {
    itsTracer = Tracer.getInstance();
    itsData = data;
  }


  public void read() {
    if (itsData != null) {
      try {
        itsTracer.write(3, "reading crontab ", itsData.getName());

        InputStreamReader aReader = new InputStreamReader(itsData.getInputStream());
        BufferedReader aBufferedReader = new BufferedReader(aReader);

        itsList.clear();

        while (aBufferedReader.ready()) {
          String aLine = aBufferedReader.readLine();
          if (aLine != null) {
            itsTracer.write(3, "crontab line: ", aLine);
            if (!aLine.startsWith("#") && aLine.length() > 0) {
              itsList.add(aLine);
            }
          }
        }

        aBufferedReader.close();
        aReader.close();

        itsDataLastModified = itsData.lastModified();

        return;
      }
      catch (Exception e) {
        itsTracer.writeException(0, e);
        e.printStackTrace();
      }
    }
    itsTracer.write(2, "generating default crontab");
    String aLine = "*\t*\t*\t*\t*\t*";
    itsList.add(aLine);
  }

  public void run() {
    try {
      Calendar now = Calendar.getInstance();
      if (itsTracer.isLevel(2)) {
        SimpleDateFormat aFormatter = new SimpleDateFormat("HH:mm:ss dd.MM.yyyy");
        itsTracer.write(2, "scheduled cron run at ", aFormatter.format(now.getTime()));
      }

      if (itsDataLastModified < itsData.lastModified()) {
        itsTracer.write(3, "crontab file ", itsData.getName(), " has changed -> reread");
        read();
        Collection aLatestMatchSet = findLatestMatches();
        Iterator it = aLatestMatchSet.iterator();
        while (it.hasNext()) {
          String aParameter = (String)it.next();
          callAction(aParameter);
        }
      }

      Iterator it = itsList.iterator();
      while (it.hasNext()) {
        String aLine = (String)it.next();
        itsTracer.write(3, "checking line ", aLine);
        String aParameter = matchDate(aLine, now);
        if (aParameter != null) {
          itsTracer.write(3, "line matches; parameter ", aParameter);
          callAction(aParameter);
        }
        else {
          itsTracer.write(3, "line doesn't match");
        }
      }

      itsTracer.write(3, "executing registered repeated actions");
      long aCurrentSecond = System.currentTimeMillis();
      executeActions(aCurrentSecond, itsInitialRepeatedActions);
      executeActions(aCurrentSecond, itsRepeatedActions);
      executeActions(aCurrentSecond, itsFinalRepeatedActions);
      itsTracer.write(2, "executed registered repeated actions");
    }
    catch (Exception e) {
      itsTracer.writeException(1, e);
    }
  }

  private void executeActions(long theCurrentSecond, Vector theRepeatedActions) {
    synchronized (theRepeatedActions) {
      Iterator it = theRepeatedActions.iterator();
      while (it.hasNext()) {
        RepeatedActionContainer aContainer = (RepeatedActionContainer)it.next();
        if (theCurrentSecond >= aContainer.itsNextRun) {
          try {
            String aClassName = "";
            if (itsTracer.isLevel(3)) {
              aClassName = aContainer.itsAction.getClass().getName();
              itsTracer.write(3, "executing repeated action class ", aClassName);
            }
            aContainer.itsAction.executeRepeatedAction(aContainer.itsParameter);
            if (itsTracer.isLevel(3)) {
              itsTracer.write(3, "executed repeated action class ", aClassName);
            }
          }
          catch (Exception e) {
            itsTracer.writeException(1, e);
          }
          if (aContainer.itsFrequency == 0) {
            it.remove();
          }
          else {
            aContainer.itsNextRun += aContainer.itsFrequency;
          }
        }
      }
    }
  }

  public void registerRepeatedActionAtBegin(IRepeatedAction theAction, long theFirstRun, long theFrequency, Object theParameter) {
    RepeatedActionContainer aContainer = new RepeatedActionContainer(theAction, theFirstRun, theFrequency, theParameter);
    synchronized (itsInitialRepeatedActions) {
      itsInitialRepeatedActions.add(aContainer);
    }
  }

  public void registerRepeatedAction(IRepeatedAction theAction, long theFirstRun, long theFrequency, Object theParameter) {
    RepeatedActionContainer aContainer = new RepeatedActionContainer(theAction, theFirstRun, theFrequency, theParameter);
    synchronized (itsRepeatedActions) {
      itsRepeatedActions.add(aContainer);
    }
  }

  public void registerRepeatedActionAtEnd(IRepeatedAction theAction, long theFirstRun, long theFrequency, Object theParameter) {
    RepeatedActionContainer aContainer = new RepeatedActionContainer(theAction, theFirstRun, theFrequency, theParameter);
    synchronized (itsFinalRepeatedActions) {
      itsFinalRepeatedActions.add(aContainer);
    }
  }

  public void deregisterRepeatedAction(IRepeatedAction theAction) {
    synchronized (itsInitialRepeatedActions) {
      itsInitialRepeatedActions.remove(theAction);
    }
    synchronized (itsRepeatedActions) {
      itsRepeatedActions.remove(theAction);
    }
    synchronized (itsFinalRepeatedActions) {
      itsFinalRepeatedActions.remove(theAction);
    }
  }

  private long getFileTime(String theFilename) {
    try {
      File aFile = new File(theFilename);
      return aFile.lastModified();
    }
    catch (Exception e) {
      itsTracer.writeException(1, e);
      return 0;
    }
  }

  public void callAction(String theParameter) {
    StringTokenizer aToken = new StringTokenizer(theParameter, "\t");
    if (!aToken.hasMoreTokens()) {
      return;
    }
    String anActionClass = aToken.nextToken();
    String anActionParameter = aToken.nextToken("\r\n");

    try {
      ICronAction anActionObject = (ICronAction)itsActionClasses.get(anActionClass);

      if (anActionObject == null) {
        Class aClass = CrtClassLoaderRegistry.forName(anActionClass);
        Object anObject = aClass.newInstance();
        anActionObject = (ICronAction)anObject;
        itsActionClasses.put(anActionClass, anActionObject);
      }
      anActionObject.execute(anActionParameter);
    }
    catch (Exception e) {
      itsTracer.writeException(1, e);
    }
  }

  public String matchDate(String theLine, Calendar theDate) {
    boolean rc;

    StringTokenizer aToken = new StringTokenizer(theLine, "\t");

    // weekday
    if (aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.DAY_OF_WEEK);
      rc = checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    // day of month
    if (rc && aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.DAY_OF_MONTH);
      rc &= checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    // month
    if (rc && aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.MONTH);
      rc &= checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    // year
    if (rc && aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.YEAR);
      rc &= checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    // hour
    if (rc && aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.HOUR_OF_DAY);
      rc &= checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    // minute
    if (rc && aToken.hasMoreTokens()) {
      int aValue = theDate.get(Calendar.MINUTE);
      rc &= checkNumber(aValue, aToken.nextToken());
    }
    else {
      return null;
    }

    String aResult = null;
    // hmm, we obtained the line via readline. So everything up to the next
    // crlf should fit.
    if (rc && aToken.hasMoreTokens()) {
      aResult = aToken.nextToken("\r\n");
    }

    return aResult;
  }

  private boolean checkNumber(int theNumber, String theRange) {
    boolean rc = false;

    boolean isSingleValue = true;
    boolean isRange = false;

    int aCurrentValue = 0;

    int aLowerBound = 0;

    int i;
    int aLength = theRange.length();
    aCurrentValue = 0;
    for (i = 0; i < aLength; ++i) {
      switch (theRange.charAt(i)) {
        case '*':
          return true;
        case ',':
          if (isRange) {
            rc = (theNumber >= aLowerBound && theNumber <= aCurrentValue);
          }
          else {
            rc = theNumber == aCurrentValue;
          }
          if (rc) {
            return rc;
          }
          aCurrentValue = 0;
          break;
        case '-':
          aLowerBound = aCurrentValue;
          isRange = true;
          aCurrentValue = 0;
          break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
          aCurrentValue = (aCurrentValue * 10) + new Integer(theRange.substring(i, i + 1)).intValue();
          break;
        default:
          break;
      }
    }
    if (isRange) {
      rc = (theNumber >= aLowerBound && theNumber <= aCurrentValue);
    }
    else {
      rc = theNumber == aCurrentValue;
    }
    return rc;
  }

  public Collection findLatestMatches() {
    Iterator it = itsList.iterator();
    Calendar now = Calendar.getInstance();
    Calendar aMaximum = null;
    String aMaximumLine = null;
    SortedMap aSortedEventMap = new TreeMap();
    while (it.hasNext()) {
      String aLine = (String)it.next();
      Calendar aLatestMatch = findLatestEventDate(aLine, now);
      if (aLatestMatch.after(now)) {
        continue;
      }
      aSortedEventMap.put(aLatestMatch.getTime(), getParameter(aLine));
    }
    Date aLatestDate = (Date)aSortedEventMap.lastKey();
    SortedMap aMaxMap = aSortedEventMap.tailMap(aLatestDate);
    return aMaxMap.values();
  }

  private String getParameter(String theLine) {
    String aParameter = null;
    if (theLine != null) {
      StringTokenizer aToken = new StringTokenizer(theLine, "\t");

      int i;
      for (i = 0; i < 6; ++i) {
        if (aToken.hasMoreTokens()) {
          aParameter = aToken.nextToken();
        }
      }
      if (aToken.hasMoreTokens()) {
        aParameter = aToken.nextToken("\r\n");
      }
    }
    return aParameter;
  }

  private Calendar findLatestEventDate(String theLine, Calendar theDate) {
    Calendar aLastDate = (Calendar)theDate.clone();
    aLastDate.set(Calendar.SECOND, 0);
    aLastDate.set(Calendar.MILLISECOND, 0);

    itsTracer.write(3, "searching the latest timestamp matching ", theLine);

    StringTokenizer aToken = new StringTokenizer(theLine, "\t");
    String aWeekday = null;
    String aYear = null;
    String aMonth = null;
    String aDay = null;
    String aHour = null;
    String aMinute = null;

    // weekday
    if (aToken.hasMoreTokens()) {
      aWeekday = aToken.nextToken();
    }

    // day
    if (aToken.hasMoreTokens()) {
      aDay = aToken.nextToken();
    }

    // month
    if (aToken.hasMoreTokens()) {
      aMonth = aToken.nextToken();
    }

    // year
    if (aToken.hasMoreTokens()) {
      aYear = aToken.nextToken();
    }

    // hour
    if (aToken.hasMoreTokens()) {
      aHour = aToken.nextToken();
    }

    // minute
    if (aToken.hasMoreTokens()) {
      aMinute = aToken.nextToken();
    }

    boolean rc;
    Calendar anOldValue = null;
    SimpleDateFormat aFormatter = new SimpleDateFormat("HH:mm:ss dd.MM.yyyy");
    do {
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "adjusting    ", aFormatter.format(aLastDate.getTime()));
      }
      modifyToLastDate(aLastDate, aMinute, Calendar.MINUTE, Calendar.HOUR_OF_DAY);
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "after minute ", aFormatter.format(aLastDate.getTime()));
      }
      modifyToLastDate(aLastDate, aHour, Calendar.HOUR_OF_DAY, Calendar.DAY_OF_MONTH);
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "after hour   ", aFormatter.format(aLastDate.getTime()));
      }
      modifyToLastDate(aLastDate, aDay, Calendar.DAY_OF_MONTH, Calendar.MONTH);
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "after day    ", aFormatter.format(aLastDate.getTime()));
      }
      modifyToLastDate(aLastDate, aMonth, Calendar.MONTH, Calendar.YEAR);
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "after month  ", aFormatter.format(aLastDate.getTime()));
      }
      modifyToLastDate(aLastDate, aYear, Calendar.YEAR, Calendar.SECOND);
      if (itsTracer.isLevel(3)) {
        itsTracer.write(3, "after year   ", aFormatter.format(aLastDate.getTime()));
      }
      int aValue = aLastDate.get(Calendar.DAY_OF_WEEK);
      rc = checkNumber(aValue, aWeekday);
      if (!rc) {
        itsTracer.write(3, "date does not match weekday.");
        aLastDate.add(Calendar.DAY_OF_MONTH, -1);
        aLastDate.set(Calendar.HOUR_OF_DAY, 23);
        aLastDate.set(Calendar.MINUTE, 59);
      }
      if (anOldValue != null && anOldValue.equals(aLastDate)) {
        rc = true;// break
      }
      anOldValue = (Calendar)aLastDate.clone();
    } while (!rc);

    return aLastDate;
  }

  private Vector getValueArray(String theRange) {
    boolean isSingleValue = true;
    boolean isRange = false;
    boolean isValue = false;

    Vector aValueArray = new Vector();

    int aCurrentValue = 0;

    int aLowerBound = 0;

    int i;
    int aLength = theRange.length();
    aCurrentValue = 0;
    for (i = 0; i < aLength; ++i) {
      switch (theRange.charAt(i)) {
        case '*':
          aValueArray.add(new Integer(-1));
          isValue = false;
          break;
        case ',':
          if (isRange) {
            int j;
            for (j = aLowerBound; j <= aCurrentValue; ++j) {
              aValueArray.add(new Integer(j));
            }
          }
          else {
            aValueArray.add(new Integer(aCurrentValue));
          }
          aCurrentValue = 0;
          isValue = false;
          break;
        case '-':
          aLowerBound = aCurrentValue;
          isRange = true;
          aCurrentValue = 0;
          break;
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
          aCurrentValue = (aCurrentValue * 10) + new Integer(theRange.substring(i, i + 1)).intValue();
          isValue = true;
          break;
        default:
          isValue = false;
          break;
      }
    }
    if (isValue) {
      if (isRange) {
        int j;
        for (j = aLowerBound; j <= aCurrentValue; ++j) {
          aValueArray.add(new Integer(j));
        }
      }
      else {
        aValueArray.add(new Integer(aCurrentValue));
      }
    }
    return aValueArray;
  }

  private void modifyToLastDate(Calendar theDate, String theRange, int theDateIdent, int theDependantDateIdent) {
    Integer aMinusOne = new Integer(-1);
    if (theRange != null) {
      TreeSet aValueArray = new TreeSet(getValueArray(theRange));
      if (!aValueArray.contains(aMinusOne)) {
        Iterator it = aValueArray.iterator();
        int aCurrentValue = theDate.get(theDateIdent);
        int aValue = -1;
        while (it.hasNext()) {
          int i = ((Integer)it.next()).intValue();
          if (i <= aCurrentValue) {
            aValue = i;
          }
          else {
            break;
          }
        }
        if (aValue >= 0) {
          theDate.set(theDateIdent, aValue);
        }
        else {
          aValue = ((Integer)aValueArray.last()).intValue();
          theDate.set(theDateIdent, aValue);
          if (theDependantDateIdent != Calendar.SECOND) {
            theDate.add(theDependantDateIdent, -1);
          }
        }
      }
    }
  }

  private void modifyToLastDay(Calendar theDate, String theRange, String theDayOfWeekRange) {
    Integer aMinusOne = new Integer(-1);
    if (theRange != null) {
      boolean rc;
      do {
        TreeSet aValueArray = new TreeSet(getValueArray(theRange));
        if (!aValueArray.contains(aMinusOne)) {
          Iterator it = aValueArray.iterator();
          int aCurrentValue = theDate.get(Calendar.DAY_OF_MONTH);
          int aValue = -1;
          while (it.hasNext()) {
            int i = ((Integer)it.next()).intValue();
            if (i <= aCurrentValue) {
              aValue = i;
            }
            else {
              break;
            }
          }
          if (aValue >= 0) {
            theDate.set(Calendar.DAY_OF_MONTH, aValue);
          }
          else {
            aValue = ((Integer)aValueArray.last()).intValue();
            theDate.set(Calendar.DAY_OF_MONTH, aValue);
            theDate.add(Calendar.MONTH, -1);
          }
        }
        int aDayOfWeek = theDate.get(Calendar.DAY_OF_WEEK);
        rc = checkNumber(aDayOfWeek, theDayOfWeekRange);
        if (!rc) {
          theDate.add(Calendar.DAY_OF_MONTH, -1);
        }
      } while (!rc);
    }
  }

}
