/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.classloader;
import java.io.File;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.URL;

import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * Title: Description: Copyright (c) SAP AG 2001-2002
 *
 * @author roland.preussmann@sap.com
 * @version 1.0
 */
public class WcmClassloader extends ClassLoader {
  public final String SEQUENCE_DEL = ";";

  private ArrayList classpath;
  private AccessControlContext acc;
  private boolean debug = false;

  private Map mappings = new HashMap(3000);

  public WcmClassloader(ClassLoader cl) {
    super(cl);
    this.acc = AccessController.getContext();
    this.classpath = new ArrayList();
  }

  public void setDebug(boolean b) {
    this.debug = b;
  }

  /**
   * @param name TBD: Description of the incoming method parameter
   * @param resolve TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception java.lang.ClassNotFoundException Exception raised in failure
   *      situation
   */
  public synchronized Class loadClass(String name, boolean resolve)
    throws java.lang.ClassNotFoundException {
    // check whether it has been loaded before
    Class c = findLoadedClass(name);
    if (c != null) {
      if (this.debug) {
        System.err.println("Class already loaded: " + name);
      }
      return c;
    }

    //==========================================================================
    // first try to load this class in the local classpath
    // To avoid ClassCastException NEVER load the Servlet class through the WcmClassloader
    if (!name.startsWith("javax.servlet") && !name.startsWith("java.")) {
      try {
        c = findClass(name);
        if (c != null) {
          if (this.debug) {
            System.err.println("WcmClassLoader: " + name);
          }
          if (resolve) {
            resolveClass(c);
          }
        }
      }
      catch (ClassNotFoundException e) {
        // ignore: use parent classloader
        com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());        
      }
    }
    if (c == null) {
      // class not found: just let the default implementation handle it.
      c = super.loadClass(name, resolve);
      if (c != null && this.debug) {
        System.err.println("System ClassLoader: " + name);
      }
    }

    if (c == null) {
      // class not in the local classpath and not found by the parent classloader
      if (this.debug) {
        System.err.println("Class not found: " + name);
      }
      throw new ClassNotFoundException(name);
    }
    return c;
  }

  public URL getResource(String name) {
    URL url = findResource(name);
    if (url == null) {
      url = super.getResource(name);
    }
    return url;
  }

  /**
   * @param folderName jarFolderToClasspath to be added
   */
  public synchronized void addJarFolderToClasspath(String folderName) {
    this.addJarFolderToClasspath(folderName, null);
  }

  /**
   * This method adds all JAR files located in the folder to the classpath
   *
   * @param folderName
   * @param sequence jarFolderToClasspath to be added
   */
  public synchronized void addJarFolderToClasspath(String folderName, String sequence) {
    File folder = new File(folderName);
    String x = folder.getAbsoluteFile().toString();
    if (folder != null && folder.isDirectory()) {
      File[] jars = folder.listFiles(
        new FilenameFilter() {
          public boolean accept(File dir, String name) {
            return ((name != null) && (name.length() > 3) && (name.substring(name.length() - 3).equalsIgnoreCase("JAR")));
          }
        });
      if (jars != null) {
        if (sequence != null) {
          jars = sortJars(jars, sequence);
        }
        // add to Classpath
        if (this.debug) {
          System.err.println("New classpath: ");
        }
        for (int i = 0; i < jars.length; i++) {
          if (this.debug) {
            System.err.println(jars[i].getName());
          }
          addFileToClasspath(jars[i]);
          //This method updates the cache of jar entries
          this.computeJarEntries(jars[i]);
        }
      }
    }
  }

  /**
   * This method updates the cache associated to the class loader. This cache is
   * built as follows: the key is the class name and the value is the name of
   * the jar file that contains this class. When we will attempt to load the
   * class, we will just have to look up its name in the cache and load it from
   * the corresponding jar without scanning all the jar files from the
   * classpath.
   *
   * @param jar is a jar file that is added to the classpath.
   */
  private void computeJarEntries(File jar) {
    try {
      JarFile jarFile = new JarFile(jar);
      Enumeration entries = jarFile.entries();
      Object entry = null;
      String name = null;
      while (entries.hasMoreElements()) {
        entry = entries.nextElement();
        name = entry.toString();
        //The jar file entries also contain the directories, therefore I will
        //only gather the entries that don't end by / (this could be replaced
        //by File.separator). Or we could only store the entries ending by ".class"
        //but this would be slower... By doing this we also store the resources.
        if (!name.endsWith("/")) {
          //We have to check if the mappings map does not already contain an entry
          //with the same name. If it does (and it DOES), it means that a class
          //exists in 2 different jar files. If we override the existing entry wih
          //the new one, we revert the classpath sequence!!! And we don't want that.
          //Note, this code could easily output the names of the classes that are
          //duplicated in the classpath and the conflicting jar files.
          if (!mappings.containsKey(name)) {
            mappings.put(name, jar);
          }
        }
      }
      jarFile.close();
    }
    catch (IOException ioe) {
            //$JL-EXC$      
      ioe.printStackTrace();
    }
  }

  public synchronized void addFileToClasspath(String fileName) {
    try {
      File file = new File(fileName);
      this.classpath.add(file);
    }
    catch (Exception e) {
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
  }

  public synchronized void addFileToClasspath(File file) {
    this.classpath.add(file);
  }

  protected Class findClass(String className)
    throws java.lang.ClassNotFoundException {
    final String name = className;
    try {
      return (Class)AccessController.doPrivileged(
        new PrivilegedExceptionAction() {
          public Object run()
            throws ClassNotFoundException {
            return findClassInternal(name);
          }
        }, acc);
    }
    catch (java.security.PrivilegedActionException pae) {
      if (pae != null) {
        throw (ClassNotFoundException)pae.getException();
      }
      else {
        throw new java.lang.ClassNotFoundException("[AutoClassLoader.findClass] : java.security.PrivilegedActionException");
      }
    }
  }

  private Class findClassInternal(String className)
    throws java.lang.ClassNotFoundException {
    //This is where the optimization takes place.
    //Rather than scanning all the jar files, we look in the mappings map, if the
    //class is there, we load it directly from the resulting jar file.
    String newClassName = className.replace('.', '/') + ".class";
    File jar = (File)mappings.get(newClassName);
    if (jar != null) {
      return loadClassFromJar(jar, className, newClassName);
    }
    else {
      Class c = null;
      Iterator iter = this.classpath.iterator();
      while (c == null && iter.hasNext()) {
        File file = (File)iter.next();
        //We only load from directories as the jar file loading would
        //have already been done.
        if (file.isDirectory()) {
          c = loadClassFromDirectory(file, className);
        }
      }
      return c;
    }

    //This is the former code, uncomment it to check the performance difference.
    /*
     * Class c = null;
     * String newClassName = className.replace('.', '/') + ".class";
     * Iterator iter = this.classpath.iterator();
     * while (c == null && iter.hasNext()) {
     * File file = (File) iter.next();
     * / jar File ??
     * if (file.isDirectory()) {
     * c = loadClassFromDirecory(file, className);
     * }
     * else {
     * c = loadClassFromJar(file, className, newClassName);
     * }
     * }
     * return c;
     */
  }

  private Class loadClassFromDirectory(File file, String className) {
    Class c = null;
    FileInputStream fi = null;
    try {
      File classFile = new File(file, className.replace('.', '/') + ".class");
      fi = new FileInputStream(classFile);
      byte[] classBytes = getClassBytesFromStream(fi, fi.available());
      c = defineClass(className, classBytes, 0, classBytes.length);
      if (this.debug) {
        System.err.println("WcmClassLoader: " + className + " => " + file);
      }
    }
    catch (Exception e) {
      // class not found in direcoty
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
    return c;
  }

  private Class loadClassFromJar(File file, String className, String javaClassName) {
    Class c = null;
    JarFile jar = null;
    InputStream entryStream = null;
    try {
      jar = new JarFile(file);
      if (jar == null) {
        return c;
      }

      JarEntry jarEntry = jar.getJarEntry(javaClassName);
      if (jarEntry != null) {
        int length = (int)jarEntry.getSize();
        entryStream = jar.getInputStream(jarEntry);
        byte[] classBytes = getClassBytesFromStream(entryStream, length);
        c = defineClass(className, classBytes, 0, length);
        if (this.debug) {
          System.err.println("WcmClassLoader: " + className + " => " + file);
        }
      }
    }
    catch (IOException e) {
      //if (this.debug) e.printStackTrace(System.err);
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
    return c;
  }

  /**
   * @param resourceName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected URL findResource(String resourceName) {
    final String name = resourceName;
    return (URL)AccessController.doPrivileged(
      new PrivilegedAction() {
        public Object run() {
          return findResourceInternal(name);
        }
      }, acc);
  }

  // internal implementation
  private URL findResourceInternal(String name) {
    // we follow the order in our classpath
    Iterator iter = this.classpath.iterator();
    if (iter != null) {
      URL resource = null;
      File file = null;
      while (iter.hasNext() && resource == null) {
        file = (File)iter.next();
        if (file.isDirectory()) {
          resource = findResourceInFolder(file, name);
        }
        else {
          resource = findResourceInJar(file, name);
        }
      }
      return resource;
    }
    return null;
  }

  /**
   * find a resource in folder
   *
   * @param folder TBD: Description of the incoming method parameter
   * @param fileName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private URL findResourceInFolder(File folder, String fileName) {
    try {
      File resFile = new File(folder, fileName);
      if ((resFile != null) && (resFile.exists()) && (resFile.isFile())) {
        return resFile.toURL();
      }
    }
    catch (Exception e) {
      // ignore
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
    return null;
  }

  /**
   * find a resource in a Jar file
   *
   * @param jarFile TBD: Description of the incoming method parameter
   * @param fileName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private URL findResourceInJar(File jarFile, String fileName) {
    try {
      JarFile jar = new JarFile(jarFile);
      if (jar != null) {
        try {
          JarEntry jarEntry = jar.getJarEntry(fileName);
          if (jarEntry != null) {
            // it is there! so return a suitable URL
            String urlName = "jar:" + jarFile.toURL().toString() + "!/";
            URL url = new URL(new URL(urlName), fileName);
            return url;
          }
        }
        finally {
          // make sure to close the jar again
          //jar.close();
        }
      }
    }
    catch (IOException ioe) {
      // ignore
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(ioe.getMessage());      
    }
    return null;
  }

  private byte[] getClassBytesFromStream(InputStream stream, int length) {
    byte[] bytes = new byte[length];
    int total = 0;
    int bytesRead = 0;
    try {
      while ((length > 0) && ((bytesRead = stream.read(bytes, total, length)) >= 0)) {
        total += bytesRead;
        length -= bytesRead;
      }
      return bytes;
    }
    catch (Exception e) {
            //$JL-EXC$      
      return null;
    }
  }

  private File[] sortJars(File[] jars, String sequenceFile) {
    File[] newJars = new File[jars.length];
    int index = 0;
    StringTokenizer stok = new StringTokenizer(sequenceFile, SEQUENCE_DEL);
    while (stok.hasMoreTokens()) {
      String name = (String)stok.nextElement();
      for (int i = 0; i < jars.length; i++) {
        if (jars[i] != null) {
          String item = jars[i].getName();
          if (item.endsWith(name)) {
            newJars[index++] = jars[i];
            jars[i] = null;
          }
        }
      }
    }
    // add the remaining items to the new classpath
    for (int i = 0; i < jars.length; i++) {
      if (jars[i] != null) {
        newJars[index++] = jars[i];
      }
    }
    return newJars;
  }
}

