/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels.wcm;

import com.sapportals.portal.security.usermanagement.IUser;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.channels.AbstractChannel;
import com.sapportals.wcm.util.channels.ChannelFactory;
import com.sapportals.wcm.util.channels.IRecipient;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.lang.String;
import java.util.Date;
import java.util.Properties;
import javax.mail.*;
import javax.mail.internet.*;

/**
 * An <code>IChannel</code> -implementation for SMS. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author paul.goetz@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/channels/wcm/SmsChannel.java#4
 *      $
 */
public class SmsChannel
   extends AbstractChannel {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * server name configuration parameter.
   */
  private final static String SERVER_TAG = "server";

  /**
   * user id configuration parameter.
   */
  private final static String USER_TAG = "user";

  /**
   * user's password configuration parameter.
   */
  private final static String PASSWD_TAG = "passwd";

  /**
   * prefix configuration parameter.
   */
  private final static String GATEWAY_TAG = "gateway";

  /**
   * sms format configuration parameter.
   */
  private final static String FORMAT_TAG = "format";

  /**
   * prefix configuration parameter.
   */
  private final static String PREFIX_TAG = "prefix";

  /**
   * addresstype configuration parameter.
   */
  private final static String FIELD_TAG = "field";

  /**
   * addresstype 'to' configuration constant .
   */
  private final static String FIELD_TO = "to";

  /**
   * addresstype 'subject' configuration constant .
   */
  private final static String FIELD_SUBJECT = "subject";


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  /**
   * Create a SMS channel with a specific id.
   *
   * @param id TBD: Description of the incoming method parameter
   * @param properties TBD: Description of the incoming method parameter
   * @throws WcmException if the channel can't be created.
   */
  public SmsChannel(String id,
    Properties properties)
    throws WcmException {

    super(id, properties);

  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  /**
   * create a mail session.
   *
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private Session newSession()
    throws WcmException {

    String step = null;
    try {
      step = "getting properites";
      Properties properties = System.getProperties();
      step = "getting session instance";
      return Session.getInstance(properties, null);
    }
    catch (Exception e) {
      String message = "SmsChannel: session error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      _log.warningT("newSession(139)", message + " - " + LoggingFormatter.extractCallstack(e));
      throw new WcmException(message, e);
    }

  }


  /**
   * create a sms message.
   *
   * @param session TBD: Description of the incoming method parameter
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param encoding TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private MimeMessage newMessage(Session session,
    String address,
    String from,
    String title,
    String message,
    String type,
    String encoding)
    throws WcmException {

    String step = null;
    try {
      step = "creating mime text message";
      MimeMessage mimemessage = new MimeMessage(session);
      step = "setting from address";
      mimemessage.setFrom(new InternetAddress(from));
      step = "setting recipient";
      mimemessage.setRecipient(Message.RecipientType.TO, new InternetAddress(address));
      step = "setting subject";
      mimemessage.setSubject(title, encoding);
      step = "setting sent date";
      mimemessage.setSentDate(new Date());
      step = "setting content";
      mimemessage.setContent(message, type);
      return mimemessage;
    }
    catch (Exception e) {
      String exceptionmessage = "SmsChannel: message error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      _log.warningT("newMessage(187)", exceptionmessage + " - " + LoggingFormatter.extractCallstack(e));
      throw new WcmException(exceptionmessage, e);
    }

  }


  /**
   * send a SMS email via SMTP.
   *
   * @param session TBD: Description of the incoming method parameter
   * @param server TBD: Description of the incoming method parameter
   * @param user TBD: Description of the incoming method parameter
   * @param passwd TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  private void sendSmtp(Session session,
    String server,
    String user,
    String passwd,
    Message message)
    throws WcmException {

    String step = null;
    try {
      step = "getting transport";
      Transport transport = session.getTransport("smtp");
      step = "connecting to server";
      transport.connect(server, user, passwd);
      step = "sending message";
      transport.sendMessage(message, message.getAllRecipients());
      step = "closing connection";
      transport.close();
    }
    catch (Exception e) {
      String emsg = "SmsChannel: SMTP error while " + step
         + " for " + m_ID + e.getMessage();
      _log.warningT("sendSmtp(225)", emsg + " - " + LoggingFormatter.extractCallstack(e));
      throw new WcmException(emsg, e);
    }

  }


  // --------
  // IChannel -----------------------------------------------------------------
  // --------

  /**
   * Get the type for this channel.
   *
   * @return <code>String</code> is the channel's type (see ChannelFactory).
   */
  public String getType() {

    return ChannelFactory.SMS;
  }


  /**
   * Get the address for the specified recipient.
   *
   * @param recipient TBD: Description of the incoming method parameter
   * @return <code>String</code> is the address of the given recipient for this
   *      channel.
   * @throws WcmException if the recipient's address for this
   *      channel can't be resolved.
   */
  public String getRecipientAddress(IRecipient recipient)
    throws WcmException {

    String[] values = null;

    try {
      IUser user = WPUMFactory.getUserFactory().getUser(recipient.getID());
      values = user.getMobile();
    }
    catch (Exception e) {
      _log.debugT(LoggingFormatter.extractCallstack(e));
    }
    if ((values == null) || (values[0] == null)) {
      return null;
    }

    return values[0];
  }


  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    String title,
    String message,
    String type)
    throws WcmException {
    this.sendTo(address, from, title, message, type, AbstractChannel.ENCODING);
  }

  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param encoding TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    String title,
    String message,
    String type,
    String encoding)
    throws WcmException {

    String server = m_Properties.getProperty(SERVER_TAG);
    String user = m_Properties.getProperty(USER_TAG, "");
    String passwd = m_Properties.getProperty(PASSWD_TAG, "");
    String format = m_Properties.getProperty(FORMAT_TAG, type);
    String gateway = m_Properties.getProperty(GATEWAY_TAG, "");
    String prefix = m_Properties.getProperty(PREFIX_TAG, "");
    String field = m_Properties.getProperty(FIELD_TAG, FIELD_SUBJECT);

    if (title == null) {
      title = "";
    }

    if (field.equalsIgnoreCase(FIELD_TO)) {
      address = prefix + address + gateway;// to=prefix...@gateway
    }
    else {
      title = prefix + address;// subject=prefix...
      address = gateway;// to=gateway
    }

    if ((from != null) && (server != null)) {
      Session session = newSession();
      MimeMessage mimemessage = newMessage(session, address, from, title, message, format, encoding);
      sendSmtp(session, server, user, passwd, mimemessage);
    }
    else {
      _log.infoT("sendTo(340)", "channel sms disabled");
    }

  }


  /**
   * Send a mime message via this channel.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param mimemessage TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    MimeMessage mimemessage)
    throws WcmException {

    String server = m_Properties.getProperty(SERVER_TAG);
    String user = m_Properties.getProperty(USER_TAG, "");
    String passwd = m_Properties.getProperty(PASSWD_TAG, "");
    String gateway = m_Properties.getProperty(GATEWAY_TAG, "");
    String prefix = m_Properties.getProperty(PREFIX_TAG, "");
    String field = m_Properties.getProperty(FIELD_TAG, FIELD_SUBJECT);

    if (server == null) {
      _log.infoT("sendTo(367)", "channel sms disabled");
      return;
    }

    try {

      MimeMessage smsmessage = new MimeMessage(mimemessage);

      smsmessage.addFrom(mimemessage.getFrom());
      smsmessage.setReplyTo(mimemessage.getReplyTo());
      smsmessage.setSentDate(mimemessage.getSentDate());
      smsmessage.setContent(mimemessage.getContent(), mimemessage.getContentType());

      /*
       * Address addresses[] = mimemessage.getRecipients(Message.RecipientType.TO);
       * for( int i = 0; i < addresses.length; i++ ) {
       * if( addresses[i] instanceof InternetAddress ) {
       * InternetAddress internetaddress = (InternetAddress)addresses[i];
       * InternetAddress smsaddress;
       * if( field.equalsIgnoreCase(FIELD_TO) ) {
       * smsmessage.setSubject(mimemessage.getSubject());
       * smsaddress = new InternetAddress(prefix + internetaddress.getAddress() + gateway); // to=prefix...@gateway
       * } else {
       * smsmessage.setSubject(prefix + internetaddress); // subject=prefix...
       * smsaddress = new InternetAddress(gateway); // to=gateway
       * }
       * smsmessage.addRecipient(Message.RecipientType.TO, smsaddress);
       * Session session = newSession();
       * sendSmtp(session, server, user, passwd, smsmessage);
       * } else {
       * _log.infoT("only internet addresses for SMS channel");
       * }
       * }
       */
      InternetAddress smsaddress;
      if (field.equalsIgnoreCase(FIELD_TO)) {
        smsmessage.setSubject(mimemessage.getSubject());
        smsaddress = new InternetAddress(prefix + address + gateway);// to=prefix...@gateway
      }
      else {
        smsmessage.setSubject(prefix + address);// subject=prefix...
        smsaddress = new InternetAddress(gateway);// to=gateway
      }
      smsmessage.addRecipient(Message.RecipientType.TO, smsaddress);
      Session session = newSession();
      sendSmtp(session, server, user, passwd, smsmessage);

    }
    catch (Exception e) {
      String emsg = "caught exception " + e.getMessage() + " while sending SMS message.";
      _log.warningT("sendTo(417)", emsg + " - " + LoggingFormatter.extractCallstack(e));
      throw new WcmException(emsg, e);
    }

  }

}
