/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels.wcm;

import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IGroupFactory;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IRoleFactory;

import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.portal.security.usermanagement.IUserFactory;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.channels.ChannelFactory;
import com.sapportals.wcm.util.channels.IChannel;
import com.sapportals.wcm.util.channels.IChannelList;
import com.sapportals.wcm.util.channels.IChannelListIterator;
import com.sapportals.wcm.util.channels.IRecipient;
import com.sapportals.wcm.util.channels.IRecipientList;
import com.sapportals.wcm.util.channels.RecipientFactory;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.resource.ResourceBundles;

import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.lang.String;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Locale;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

/**
 * A basic IRecipient implementation. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author paul.goetz@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/channels/wcm/Recipient.java#7
 *      $
 * @see IRecipient
 * @see RecipientFactory
 */
public class Recipient
   implements IRecipient {

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  protected static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.channels.wcm.Recipient.class);

  /**
   * resource bundles.
   */
  protected static ResourceBundles g_Res = ResourceBundles.getBundle("com.sapportals.wcm.util.channels.wcm.Res");

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  /**
   * the principal type.
   */
  private int m_Type = IUMPrincipal.IUSER;

  /**
   * unique system id.
   */
  private String m_ID = null;

  /**
   * a displayable name.
   */
  private String m_Description = null;

  /**
   * the associated user object (if it's a user).
   */
  private IUser m_User = null;

  /**
   * the associated group object (if it's a group).
   */
  private IGroup m_Group = null;

  /**
   * the associated role object (if it's a role).
   */
  private IRole m_Role = null;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  /**
   * Prevent the creation of a recipient without id.
   */
  private Recipient() { }


  /**
   * Create a recipient from a given id and user.<br>
   * <i>note</i> : This constructor should only be used by the <code>
   * RecipientFactory</code> !
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      user-id).
   * @param user an <code>IUser</code> with the user object for the recipient.
   */
  public Recipient(String id,
    IUser user) {

    m_ID = id;
    m_Type = IUMPrincipal.IUSER;
    m_User = user;
    m_Group = null;
    m_Role = null;

  }


  /**
   * Create a recipient from a given id and group.<br>
   * <i>note</i> : This constructor should only be used by the <code>
   * RecipientFactory</code> !
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      group-id).
   * @param group an <code>IGroup</code> with the group object for the
   *      recipient.
   */
  public Recipient(String id,
    IGroup group) {

    m_ID = id;
    m_Type = IUMPrincipal.IGROUP;
    m_User = null;
    m_Group = group;
    m_Role = null;

  }


  /**
   * Create a recipient from a given id and role.<br>
   * <i>note</i> : This constructor should only be used by the <code>
   * RecipientFactory</code> !
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      role-id).
   * @param role an <code>IRole</code> with the role object for the recipient.
   */
  public Recipient(String id, IRole role) {

    m_ID = id;
    m_Type = IUMPrincipal.IROLE;
    m_User = null;
    m_Group = null;
    m_Role = role;

  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  private boolean containsForRole(IRecipient recipient)
    throws Exception {

    String[] roles = null;

    if (recipient.getType() == IUMPrincipal.IGROUP) {
      IGroup group = WPUMFactory.getGroupFactory().getGroup(recipient.getID());
      roles = group.getRoles();
    }
    else {// recipient to check is a user
      IUser user = WPUMFactory.getUserFactory().getUser(recipient.getID());
      roles = user.getRoles();
    }

    if (roles == null) {
      return false;
    }

    for (int i = 0; i < roles.length; i++) {
      if (m_ID.equals(roles[i])) {
        return true;
      }
    }

    return false;
  }


  private boolean containsForGroup(IRecipient recipient)
    throws Exception {

    if (recipient.getType() == IUMPrincipal.IGROUP) {
      return (m_Group.containsGroup(recipient.getID()));
    }
    else {// recipient to check is a user
      return (m_Group.containsUser(recipient.getID()));
    }

  }


  // ------
  // Object -------------------------------------------------------------------
  // ------

  public boolean equals(Object object) {

    if (object instanceof Recipient) {

      Recipient other = (Recipient)object;

      return ((m_ID.equals(other.m_ID))
         && (m_Type == other.m_Type)
        );
    }

    if (object instanceof IRecipient) {

      IRecipient other = (IRecipient)object;

      try {
        return ((m_ID.equals(other.getID()))
           && (m_Type == other.getType())
          );
      }
      catch (Exception e) {
        g_Log.warningT("equals(253)", "equals: unable to compare " + m_ID + " with other IRecipient" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      }

    }

    return false;
  }


  // ----------
  // IRecipient ---------------------------------------------------------------
  // ----------

  public int getType() {

    return m_Type;
  }


  public String getID() {

    return m_ID;
  }


  public String getDescription() {

    if (m_Description != null) {
      return m_Description;
    }

    if (m_User != null) {
      m_Description = m_User.getDisplayName();
    }
    else if (m_Group != null) {
      m_Description = m_Group.getDescription();
    }
    else if (m_Role != null) {
      m_Description = m_Role.getDescription();
    }

    if (m_Description == null) {
      m_Description = m_ID;
    }

    return m_Description;
  }


  public String getDescription(Locale locale) {

    return getDescription();
  }


  public Locale getLocale()
    throws WcmException {

    Locale locale = null;

    if (m_Role != null) {
      // throw new WcmException("Recipient: no getLocale() available for roles");
      return Locale.getDefault();
    }
    if (m_Group != null) {
      // throw new WcmException("Recipient: no getLocale() available for groups");
      return Locale.getDefault();
    }

    // it's a user
    try {
      locale = m_User.getLocale();
    }
    catch (Exception e) {
      g_Log.warningT("getLocale(327)", "getLocale: unable to retrieve locale for " + m_ID + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    if (locale == null) {
      locale = Locale.getDefault();
    }

    return locale;
  }


  public IUMPrincipal getUserManagementObject()
    throws WcmException {

    if (m_User != null) {
      return m_User;
    }

    if (m_Group != null) {
      return m_Group;
    }

    if (m_Role != null) {
      return m_Role;
    }

    return null;
  }


  public boolean isGroup()
    throws WcmException {

    return ((m_Group != null)
       || (m_Role != null)
      );
  }


  public IRecipientList getGroupMembers()
    throws WcmException {

    if (!isGroup()) {
      g_Log.infoT("getGroupMembers(370)", "getGroupMembers: " + m_ID + " is not a group");
      return null;
    }

    IRecipientList members = new RecipientList();

    try {
      String[] users = null;
      if (m_Role != null) {
        users = m_Role.getAllUsers();
      }
      else {// it's a group
        users = m_Group.getAllChildUsers();
      }
      IUserFactory userFactory = WPUMFactory.getUserFactory();
      if (users != null) {
        for (int i = 0; i < users.length; i++) {
          IUser user = userFactory.getUser(users[i]);
          Recipient member = new Recipient(users[i], user);
          members.addLast(member);
        }
      }
    }
    catch (Exception e) {
      throw new WcmException("Recipient: caught exception while getting members for "
         + m_ID + ": " + e.getMessage(), e);
    }

    return members;
  }


  public boolean contains(IRecipient recipient)
    throws WcmException {

    if (!isGroup()) {
      g_Log.infoT("contains(406)", "contains: " + m_ID + " is not a group");
      return false;
    }

    if (recipient.getType() == IUMPrincipal.IROLE) {
      g_Log.infoT("contains(411)", "contains: " + recipient.getID() + " is a role");
      return false;
    }

    try {
      if (getType() == IUMPrincipal.IROLE) {
        return containsForRole(recipient);
      }
      else {// it's a group
        return containsForGroup(recipient);
      }
    }
    catch (Exception e) {
      throw new WcmException("Recipient: caught exception checking " + m_ID
         + " for contains: " + e.getMessage(), e);
    }

  }


  public IChannel getChannel(String id)
    throws WcmException {

    IChannel channel = ChannelFactory.getInstance().getChannel(id);

    if (channel == null) {
      g_Log.infoT("getChannel(437)", "getChannel: channel " + id + " not available for " + m_ID);
    }

    try {// only necessary for groups
      if ((channel.getRecipientAddress(this) == null)
         && (!isGroup())
        ) {
        channel = null;
      }
    }
    catch (Exception e) {
      throw new WcmException("Recipient: exception while getting channel " + channel.getID()
         + "'s address for " + m_ID + ": " + e.getMessage(), e);
    }

    return channel;
  }


  public IChannelList getChannels()
    throws WcmException {

    IChannelList channels = new ChannelList();

    IChannelList allChannels = ChannelFactory.getInstance().getChannels();

    if (allChannels == null) {
      g_Log.infoT("getChannels(464)", "getChannels: no channels available for " + m_ID);
      return channels;
    }

    IChannelListIterator iterator = allChannels.listIterator();
    while (iterator.hasNext()) {
      IChannel channel = iterator.next();
      try {
        if (channel.getRecipientAddress(this) != null) {
          channels.add(channel);
        }
      }
      catch (Exception e) {
        g_Log.warningT("getChannels(477)", "getChannels: caught exception while getting channel " + channel.getID()
           + "'s address for " + m_ID + ": " + e.getMessage() + " - " + LoggingFormatter.extractCallstack(e));
      }
    }

    return channels;
  }


  public IChannel getDefaultChannel()
    throws WcmException {

    IChannel channel = null;

    IChannelList emailChannels = ChannelFactory.getInstance().getChannelForType(ChannelFactory.EMAIL);
    IChannelListIterator iterator = emailChannels.listIterator();
    if (iterator.hasNext()) {
      channel = iterator.next();
    }
    else {
      g_Log.warningT("getDefaultChannel(497)", "getDefaultChannel: no available default channel for " + m_ID);
    }

    return channel;
  }


  public void sendTo(IChannel channel,
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException {

    if (channel == null) {
      throw new WcmException("Recipient: invalid channel <null>");
    }

    String address = null;

    try {
      address = channel.getRecipientAddress(this);
    }
    catch (Exception e) {
      throw new WcmException("Recipient: caught exception getting channel " + channel.getID()
         + "'s address for " + m_ID + ": " + e.getMessage(), e);
    }

    if (address != null) {
      channel.sendTo(address, from, title, messagetext, type);
    }
    else {
      g_Log.warningT("sendTo(529)", "sendTo: invalid channel " + channel.getID() + "'s address for " + m_ID);
    }

  }


  public void sendTo(IChannel channel,
    String from,
    MimeMessage mimemessage)
    throws WcmException {

    if (channel == null) {
      throw new WcmException("Recipient: invalid channel <null>");
    }

    String address = null;

    try {
      address = channel.getRecipientAddress(this);
    }
    catch (Exception e) {
      throw new WcmException("Recipient: caught exception getting channel " + channel.getID()
         + "'s address for " + m_ID + ": " + e.getMessage(), e);
    }

    if (address != null) {
      channel.sendTo(address, from, mimemessage);
    }
    else {
      g_Log.warningT("sendTo(558)", "sendTo: invalid channel " + channel.getID() + "'s address for " + m_ID);
    }

  }

}
