/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels.wcm;
import com.sap.tc.logging.Location;

import com.sapportals.portal.security.usermanagement.*;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.channels.PartialSendException;
import com.sapportals.wcm.util.channels.AbstractChannel;
import com.sapportals.wcm.util.channels.ChannelFactory;
import com.sapportals.wcm.util.channels.IRecipient;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.lang.String;
import java.net.URL;
import java.util.Date;
import java.util.Properties;
import javax.activation.*;
import javax.mail.*;
import javax.mail.internet.*;

/**
 * An <code>IChannel</code> -implementation for emails. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author paul.goetz@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/channels/wcm/EmailChannel.java#6 $
 */
public class EmailChannel
   extends AbstractChannel {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * server name configuration parameter.
   */
  private final static String SERVER_TAG = "server";

  /**
   * user id configuration parameter.
   */
  private final static String USER_TAG = "user";

  /**
   * user's password configuration parameter.
   */
  private final static String PASSWD_TAG = "passwd";

  /**
   * email format configuration parameter.
   */
  private final static String FORMAT_TAG = "format";

  /**
   * email type configuration parameter.
   */
  private final static String MAILTYPE_TAG = "type";

  /**
   * email type SMTP configuration constant.
   */
  private final static String MAILTYPE_SMTP = "smtp";

  /**
   * email type IMAP configuration constant.
   */
  private final static String MAILTYPE_IMAP = "imap";

  /**
   * email send partial flag.
   */
  private final static String SENDPARTIAL = "sendpartial";
  

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  private Session m_Session = null;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  /**
   * Create an email channel with a specific id.
   *
   * @param id TBD: Description of the incoming method parameter
   * @param properties TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   * @throws <code> WcmException</code> if the channel can't be created.
   */
  public EmailChannel(String id,
    Properties properties)
    throws WcmException {

    super(id, properties);

    /*
     * String step = null;
     * try {
     * step = "getting properites";
     * Properties sysProperties = System.getProperties();
     * step = "getting session instance";
     * m_Session = Session.getInstance(sysProperties, null);
     * }
     * catch( Exception e ) {
     * String message = "EmailChannel: session error while " + step
     * + " for " + m_ID + ": " + e.getMessage();
     * _log.warningT(message, e);
     * throw new WcmException(message, e);
     * }
     */
  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  /**
   * get a mail session.
   *
   * @return session
   * @exception WcmException Exception raised in failure situation
   */
  private Session getSession()
    throws WcmException {

    String step = null;
    try {
      step = "getting properites";
      Properties sysProperties = System.getProperties();
      String partial = m_Properties.getProperty(SENDPARTIAL);
      if( partial != null ) {
        sysProperties.setProperty("mail.smtp.sendpartial", partial);
      }
      step = "getting session instance";
      m_Session = Session.getInstance(sysProperties, null);
    }
    catch (Exception e) {
      String message = "EmailChannel: session error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      if( _log.beInfo() ) _log.infoT(message);
      throw new WcmException(message, e);
    }

    return m_Session;
  }


  /**
   * create an email message, attachments aren't still final, class Attachment
   * necessary, problem for Resource management: JavaMail need absolute
   * OS-dependent path or fully specified URL with host, port and path
   *
   * @param session TBD: Description of the incoming method parameter
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param subject TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param encoding TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private MimeMessage newMessage(Session session,
    String address,
    String from,
    String subject,
    String messagetext,
    String type,
    String encoding)
    throws WcmException {

    String step = null;
    try {
      step = "creating mime message";
      MimeMessage mimemessage = new MimeMessage(session);
      step = "setting from address";
      mimemessage.setFrom(new InternetAddress(from));
      step = "setting recipient";
      mimemessage.setRecipient(Message.RecipientType.TO, new InternetAddress(address));
      step = "setting subject";
      mimemessage.setSubject(subject, encoding);
      step = "setting sent date";
      mimemessage.setSentDate(new Date());
      step = "setting content";
      mimemessage.setText(messagetext, encoding); // mimemessage.setContent(messagetext, type);      
      step = "setting header";
      mimemessage.setHeader("Content-Type", type + "; charset=" + encoding);
      return mimemessage;
    }
    catch (Exception e) {
      String exceptionmessage = "EmailChannel: message error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      if( _log.beInfo() ) _log.infoT(exceptionmessage);
      throw new WcmException(exceptionmessage, e);
    }

  }


  /**
   * send an email via SMTP.
   *
   * @param session TBD: Description of the incoming method parameter
   * @param server TBD: Description of the incoming method parameter
   * @param user TBD: Description of the incoming method parameter
   * @param passwd TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  private void sendSmtp(Session session,
    String server,
    String user,
    String passwd,
    Message message)
    throws WcmException {

    // Make several try-catchs because all transport-methods
    // throw the same exception. The Exception from the
    // transport.sendMessage(...) should go to AppLog and log
    // the others only log to file
    String step = null;
    Transport transport = null;
    try {
      step = "getting transport";
      transport = session.getTransport("smtp");
      step = "connecting to server";
      transport.connect(server, user, passwd);
    }
    catch (Exception e) {
      String emsg = "EmailChannel: SMTP error while " + step
         + " for " + m_ID + e.getMessage();
      if( _log.beInfo() ) _log.infoT(emsg);
      throw new WcmException(emsg, e);
    }
    try {
      step = "sending message";
      transport.sendMessage(message, message.getAllRecipients());
    }
    catch( SendFailedException e ) {
      if( _log.beInfo() ) {
        int i;
        Address[] a;
        StringBuffer emsg = new StringBuffer(2000);
        emsg.append("unable to send message with subject ");
        try {
          emsg.append("'" + message.getSubject() + "'");
        }
        catch( MessagingException me ) {
          emsg.append("<n/a>");
        }
        emsg.append(" from ");
        try {
          if( message.getFrom() == null ) {
            emsg.append("<null>");
          } else {
            emsg.append("'" + message.getFrom()[0] + "'");
          }
        }
        catch( MessagingException me ) {
          emsg.append("<n/a>");
        }
        emsg.append(" to ");
        a = e.getValidUnsentAddresses();
        if( a != null ) {
          if( a.length > 0 ) {
            emsg.append(" unsent addresses: ");
            for( i = 0; i < a.length; i++ ) {
              emsg.append(a[i].toString());
              emsg.append(',');
            }
          }
        }
        a = e.getInvalidAddresses();
        if( a != null ) {
          if( a.length > 0 ) {
            emsg.append(" invalid addresses: ");
            for( i = 0; i < a.length; i++ ) {
              emsg.append(a[i].toString());
              emsg.append(',');
            }
          }
        }
        emsg.append(" " + e.getMessage());
        _log.infoT(emsg.toString());
      }
      throw new PartialSendException(e);
    }
    catch (Exception e) {
      String emsg = "EmailChannel: SMTP error while " + step + " for " + m_ID + e.getMessage();
      if (_log.beInfo()) {
        _log.infoT("sendSmtp(254)", emsg);
      }
      throw new WcmException(emsg, e);
    }
    try {
      step = "closing connection";
      transport.close();
    }
    catch (Exception e) {
      String emsg = "EmailChannel: SMTP error while " + step
         + " for " + m_ID + e.getMessage();
      if( _log.beInfo() ) _log.infoT(emsg);
      throw new WcmException(emsg, e);
    }

  }


  /**
   * store an email in recipients INBOX via IMAP.
   *
   * @param session TBD: Description of the incoming method parameter
   * @param server TBD: Description of the incoming method parameter
   * @param user TBD: Description of the incoming method parameter
   * @param passwd TBD: Description of the incoming method parameter
   * @param message TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  private void storeImap(Session session,
    String server,
    String user,
    String passwd,
    Message message)
    throws WcmException {

    String step = null;
    try {
      step = "getting store";
      Store store = session.getStore("imap");
      step = "connecting to server";
      store.connect(server, user, passwd);
      step = "getting inbox folder";
      Folder folder = store.getFolder("INBOX");
      step = "creating message array";
      Message[] array = new Message[1];
      array[0] = message;
      step = "appending message";
      folder.appendMessages(array);
    }
    catch (Exception e) {
      String emsg = "EmailChannel: IMAP error while " + step
         + " for " + m_ID + e.getMessage();
      if( _log.beInfo() ) _log.infoT(emsg);
      throw new WcmException(emsg, e);
    }

  }


  // --------
  // IChannel -----------------------------------------------------------------
  // --------

  /**
   * Get the type for this channel.
   *
   * @return <code>String</code> is the channel's type (see ChannelFactory).
   */
  public String getType() {
    return ChannelFactory.EMAIL;
  }


  /**
   * Get an empty mime message for this channel.
   *
   * @return an empty <code>MimeMessage</code> for this channel.
   * @exception WcmException Exception raised in failure situation
   * @throws <code> WcmException</code> if there was an error while creating the
   *      mime message.
   */
  public MimeMessage getEmptyMessage()
    throws WcmException {

    String step = null;
    try {
      step = "creating mime message";
      return new MimeMessage(getSession());
    }
    catch (Exception e) {
      String message = "error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      if( _log.beInfo() ) _log.infoT(message);
      throw new WcmException(message, e);
    }

  }


  /**
   * Get the address for the specified recipient.
   *
   * @param recipient TBD: Description of the incoming method parameter
   * @return <code>String</code> is the address of the given recipient for this
   *      channel.
   * @exception WcmException Exception raised in failure situation
   * @throws <code> WcmException</code> if the recipient's address for this
   *      channel can't be resolved.
   */
  public String getRecipientAddress(IRecipient recipient)
    throws WcmException {

    String[] values = null;

    try {
      IUser user = WPUMFactory.getUserFactory().getUser(recipient.getID());
      if (user != null) {// check if it's a user
        values = user.getEmail();
      }
    }
    catch (Exception e) {
      if( _log.beDebug() ) _log.debugT(LoggingFormatter.extractCallstack(e));
    }

    if ((values == null)
       || (values[0] == null)
      ) {
      return null;
    }

    return values[0];
  }

  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param subject TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    String subject,
    String messagetext,
    String type)
    throws WcmException {
    this.sendTo(address, from, subject, messagetext, type, AbstractChannel.ENCODING);
  }


  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param subject TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param encoding TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    String subject,
    String messagetext,
    String type,
    String encoding)
    throws WcmException {

    String mailtype = m_Properties.getProperty(MAILTYPE_TAG, MAILTYPE_SMTP);
    String server = m_Properties.getProperty(SERVER_TAG);
    String user = m_Properties.getProperty(USER_TAG, "");
    String passwd = m_Properties.getProperty(PASSWD_TAG, "");
    String contenttype = m_Properties.getProperty(FORMAT_TAG, (type != null) ? type : "text/plain");

    if (subject == null) {
      subject = "";
    }

    if ((from != null) && (server != null)) {
      Session session = getSession();
      MimeMessage mimemessage = newMessage(session, address, from, subject, messagetext, contenttype, encoding);
      try {
        mimemessage.setSubject(subject, "UTF-8");
      }
      catch (Exception e) {
        // do nothing, as subject (non UTF-8) is already set
        if( _log.beDebug() ) _log.debugT(LoggingFormatter.extractCallstack(e));
      }
      if (mailtype.equalsIgnoreCase(MAILTYPE_IMAP)) {
        storeImap(session, server, user, passwd, mimemessage);
      }
      else {
        sendSmtp(session, server, user, passwd, mimemessage);
      }
    }
    else {
      if( _log.beInfo() ) _log.infoT("channel email disabled");
    }

  }


  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param mimemessage TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String address,
    String from,
    MimeMessage mimemessage)
    throws WcmException {

    String mailtype = m_Properties.getProperty(MAILTYPE_TAG, MAILTYPE_SMTP);
    String server = m_Properties.getProperty(SERVER_TAG);
    String user = m_Properties.getProperty(USER_TAG, "");
    String passwd = m_Properties.getProperty(PASSWD_TAG, "");

    if (server == null) {
      if( _log.beInfo() ) _log.infoT("channel email disabled");
      return;
    }

    try {
      InternetAddress fromaddress = new InternetAddress(from);
      mimemessage.setFrom(fromaddress);
      InternetAddress internetaddress = new InternetAddress(address);
      mimemessage.addRecipient(Message.RecipientType.TO, internetaddress);
      mimemessage.setSubject(mimemessage.getSubject(), "UTF-8");
    }
    catch (Exception e) {
      if( _log.beDebug() ) _log.debugT(LoggingFormatter.extractCallstack(e));
    }

    Session session = getSession();
    if (mailtype.equalsIgnoreCase(MAILTYPE_IMAP)) {
      storeImap(session, server, user, passwd, mimemessage);
    }
    else {
      sendSmtp(session, server, user, passwd, mimemessage);
    }

  }


  /**
   * Send a message via this channel to the given addresses.
   *
   * @param addresses TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param mimemessage TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String addresses[],
    String from,
    MimeMessage mimemessage)
    throws WcmException {

    String mailtype = m_Properties.getProperty(MAILTYPE_TAG, MAILTYPE_SMTP);
    String server = m_Properties.getProperty(SERVER_TAG);
    String user = m_Properties.getProperty(USER_TAG, "");
    String passwd = m_Properties.getProperty(PASSWD_TAG, "");

    if (server == null) {
      if( _log.beInfo() ) _log.infoT("sendTo(521)", "channel email disabled");
      return;
    }

    try {
      InternetAddress fromaddress = new InternetAddress(from);
      mimemessage.setFrom(fromaddress);
      mimemessage.setSubject(mimemessage.getSubject(), "UTF-8");
      for (int i = 0; i < addresses.length; i++) {
        InternetAddress internetaddress = new InternetAddress(addresses[i]);
        mimemessage.addRecipient(Message.RecipientType.TO, internetaddress);
      }
    }
    catch (Exception e) {
      if( _log.beDebug() ) _log.debugT(LoggingFormatter.extractCallstack(e));
    }

    Session session = getSession();
    if (mailtype.equalsIgnoreCase(MAILTYPE_IMAP)) {
      storeImap(session, server, user, passwd, mimemessage);
    }
    else {
      sendSmtp(session, server, user, passwd, mimemessage);
    }

  }

}
