/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache.persistent.filesystem;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.util.logging.LoggingFormatter;

import com.sapportals.wcm.util.uuid.UUID;

//import com.sun.crypto.provider.SunJCE;
import iaik.security.provider.IAIK;

import java.io.*;
import java.security.*;
import java.util.*;

import javax.crypto.*;
import javax.crypto.spec.*;

/**
 * TBD: Description of the class.
 */
public class FilesystemMap implements Map {// not synchronized

  private final static char PATH_SEPARATOR = '/';
  private final static String KEY_FILE_ASSIGNMENT_FILE = "kf.dta";

  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.cache.persistent.filesystem.FilesystemMap.class);

  private String m_folder;
  private String m_filePrefix;
  private boolean m_secure;

  private Hashtable m_keyFileAssignments;


  // -- construct --

  public FilesystemMap(String folder, String filePrefix, Boolean clearOnInit, Boolean secure)
    throws Exception {
    m_folder = folder;
    if (m_folder.length() > 0 && m_folder.charAt(m_folder.length() - 1) != PATH_SEPARATOR) {
      m_folder += PATH_SEPARATOR;
    }

    m_filePrefix = filePrefix;
    m_secure = secure.booleanValue();

    m_keyFileAssignments = new Hashtable();

    if (m_secure) {
      Security.addProvider(new iaik.security.provider.IAIK());
    }

    if (clearOnInit.booleanValue()) {
      removeMapFiles();
    }
    else {
      readKeyFileAssignments();
    }
  }

  // -- interface --

  public int size() {
    return m_keyFileAssignments.size();
  }

  public boolean isEmpty() {
    return size() == 0;
  }

  public boolean containsKey(Object key) {
    return m_keyFileAssignments.containsKey(key);
  }

  public boolean containsValue(Object value) {
    throw new UnsupportedOperationException();
  }

  public Object get(Object key) {
    String filename = (String)m_keyFileAssignments.get(key);
    if (filename == null) {
      return null;
    }

    return readEntryFromFile(filename);
  }

  public Object put(Object key, Object value) {
    if (value == null) {
      throw new NullPointerException();
    }

    String filename = (String)m_keyFileAssignments.get(key);
    if (filename == null) {
      filename = getNewFilename();
      m_keyFileAssignments.put(key, filename);
      writeKeyFileAssignments();
    }

    return writeEntryToFile(filename, value);// return size
  }

  public Object remove(Object key) {
    String filename = (String)m_keyFileAssignments.get(key);
    if (filename == null) {
      return null;
    }

    Object result = readEntryFromFile(filename);

    m_keyFileAssignments.remove(key);
    writeKeyFileAssignments();
    removeFile(filename);

    return result;
  }

  public void putAll(Map t) {
    throw new UnsupportedOperationException();
  }

  public void clear() {
    Collection filenames = m_keyFileAssignments.values();
    if (filenames != null) {
      Iterator iterator = filenames.iterator();
      while (iterator != null && iterator.hasNext()) {
        String filename = (String)iterator.next();
        if (filename != null) {
          removeFile(filename);
        }
      }
    }

    m_keyFileAssignments.clear();
    writeKeyFileAssignments();
  }

  public Set keySet() {
    return m_keyFileAssignments.keySet();
  }

  public Collection values() {
    throw new UnsupportedOperationException();
  }

  public Set entrySet() {
    throw new UnsupportedOperationException();
  }

  // -- private --

  private void readKeyFileAssignments()
    throws Exception {
    String filePathName = makeFilePathName(KEY_FILE_ASSIGNMENT_FILE);

    File file = new File(filePathName);
    if (!file.exists()) {
      return;
    }

    ObjectInputStream in;
    try {
      if (m_secure) {
        in = new ObjectInputStream(new CipherInputStream(new FileInputStream(file), getDecryptCipher()));
      }
      else {
        in = new ObjectInputStream(new FileInputStream(file));
      }
    }
    catch (FileNotFoundException e) {
                  //$JL-EXC$
      return;
    }
    catch (Exception e) {
      s_log.errorT("readKeyFileAssignments(189)", "can't open (existing) key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      throw e;
    }
    if (in == null) {
      return;
    }

    boolean readFailed = false;
    do {
      try {
        String key = (String)in.readObject();
        String filename = (String)in.readObject();

        m_keyFileAssignments.put(key, filename);
      }
      catch (IOException e) {
        // probably eof
        readFailed = true;
      }
      catch (Exception e) {
        s_log.errorT("readKeyFileAssignments(209)", "can't read key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
        throw e;
      }
    } while (!readFailed);

    try {
      in.close();
    }
    catch (IOException e) {
      s_log.errorT("readKeyFileAssignments(218)", "can't close key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      throw e;
    }
  }

  private void writeKeyFileAssignments() {
    String filePathName = makeFilePathName(KEY_FILE_ASSIGNMENT_FILE);

    File file = new File(filePathName);
    if (file.exists()) {
      if (!file.delete()) {
        s_log.errorT("writeKeyFileAssignments(229)", "can't delete old key-filename assignment file (" + filePathName + ")");
        return;
      }
    }

    ObjectOutputStream out;
    try {
      if (m_secure) {
        out = new ObjectOutputStream(new CipherOutputStream(new FileOutputStream(file), getEncryptCipher()));
      }
      else {
        out = new ObjectOutputStream(new FileOutputStream(file));
      }
    }
    catch (Exception e) {
      s_log.errorT("writeKeyFileAssignments(244)", "can't create key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      return;
    }
    if (out == null) {
      return;
    }

    boolean writeFailed = false;
    Enumeration keys = m_keyFileAssignments.keys();
    while (!writeFailed && keys != null && keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      if (key != null) {
        try {
          out.writeObject(key);
          out.writeObject(m_keyFileAssignments.get(key));
        }
        catch (Exception e) {
          s_log.errorT("writeKeyFileAssignments(261)", "can't write key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
          writeFailed = true;
        }
      }
    }

    try {
      out.close();
    }
    catch (IOException e) {
      s_log.errorT("writeKeyFileAssignments(271)", "can't close key-filename assignment file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
  }

  private String makeFilePathName(String filename) {
    return m_folder + m_filePrefix + filename;
  }

  private Object readEntryFromFile(String filename) {
    String filePathName = makeFilePathName(filename);

    File file = new File(filePathName);
    if (!file.exists()) {
      return null;
    }

    ObjectInputStream in;
    try {
      if (m_secure) {
        in = new ObjectInputStream(new CipherInputStream(new FileInputStream(file), getDecryptCipher()));
      }
      else {
        in = new ObjectInputStream(new FileInputStream(file));
      }
    }
    catch (Exception e) {
      s_log.errorT("readEntryFromFile(297)", "can't open entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      return null;
    }
    if (in == null) {
      return null;
    }

    Object result = null;

    try {
      result = in.readObject();
    }
    catch (Exception e) {
      s_log.errorT("readEntryFromFile(310)", "can't read entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    try {
      in.close();
    }
    catch (IOException e) {
      s_log.warningT("readEntryFromFile(317)", "can't close entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    return result;
  }

  private Long writeEntryToFile(String filename, Object entry) {
    String filePathName = makeFilePathName(filename);

    File file = new File(filePathName);
    if (file.exists()) {
      if (!file.delete()) {
        s_log.errorT("writeEntryToFile(329)", "can't delete old entry file (" + filePathName + ")");
        return null;
      }
    }

    ObjectOutputStream out;
    try {
      if (m_secure) {
        out = new ObjectOutputStream(new CipherOutputStream(new FileOutputStream(file), getEncryptCipher()));
      }
      else {
        out = new ObjectOutputStream(new FileOutputStream(file));
      }
    }
    catch (Exception e) {
      s_log.errorT("writeEntryToFile(344)", "can't create entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      return null;
    }
    if (out == null) {
      return null;
    }

    try {
      out.writeObject(entry);
    }
    catch (Exception e) {
      s_log.errorT("writeEntryToFile(355)", "can't write entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    try {
      out.close();
    }
    catch (IOException e) {
      s_log.warningT("writeEntryToFile(362)", "can't close entry file (" + filePathName + ")" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    return new Long(file.length());// return size
  }

  private void removeFile(String filename) {
    String filePathName = makeFilePathName(filename);

    File file = new File(filePathName);
    if (!file.exists()) {
      return;
    }

    if (!file.delete()) {
      s_log.errorT("removeFile(377)", "can't delete file (" + filePathName + ")");
    }
  }

  private String getNewFilename() {
    return new UUID().toString();
  }

  private void removeMapFiles() {
    try {
      File cacheDirectory = new File(m_folder);
      File[] cacheFiles = cacheDirectory.listFiles();
      for (int i = 0; i < cacheFiles.length; i++) {
        if (!cacheFiles[i].isDirectory() && cacheFiles[i].getName().startsWith(m_filePrefix)) {
          if (!cacheFiles[i].delete()) {
            s_log.errorT("removeMapFiles(392)", "can't delete file (" + cacheFiles[i].getName() + ")");
          }
        }
      }
    }
    catch (Exception e) {
      s_log.errorT("removeMapFiles(398)", "can't delete cache files" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
  }

  private final String SCRAMBLED_KEY = "sc2nbsd0fb9qb3kds5vbhkds";

  private byte[] unscrambleKey(String key) {
    byte[] bytes = key.getBytes();
    for (int i = 0; i < bytes.length; i++) {
      bytes[i] ^= '#';
    }
    return bytes;
  }

  private SecretKey getSecretKey()
    throws Exception {
    return new SecretKeySpec(unscrambleKey(SCRAMBLED_KEY), "DESede");
  }

  private Cipher getEncryptCipher()
    throws Exception {
    Cipher encryptCypher = Cipher.getInstance("DESede/ECB/PKCS5Padding");
    encryptCypher.init(Cipher.ENCRYPT_MODE, getSecretKey());
    return encryptCypher;
  }

  private Cipher getDecryptCipher()
    throws Exception {
    Cipher decryptCypher = Cipher.getInstance("DESede/ECB/PKCS5Padding");
    decryptCypher.init(Cipher.DECRYPT_MODE, getSecretKey());
    return decryptCypher;
  }
}
