/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache.persistent;

/*
 * import com.sapportals.wcm.util.regex.re.RE;
 * import com.sapportals.wcm.util.regex.re.RESyntaxException;
 */
import com.sap.tc.logging.Location;

import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.cache.persistent.*;
import com.sapportals.wcm.util.cache.persistent.filesystem.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;

import java.io.*;

import java.lang.reflect.Constructor;
import java.util.*;

/**
 * @todo: Description of the class.
 */
public class PersistentLRUCache implements ICache, ICacheStatistics, IExtendedCacheStatistics {
  private int m_capacity = ICache.DEFAULT_CAPACITY;// max. no. of entries in the cache
  private long m_maxCacheSize = ICache.DEFAULT_MAX_CACHE_SIZE;// max. cache size (in bytes)
  private long m_maxEntrySize = ICache.DEFAULT_MAX_ENTRY_SIZE;// max. entry size (in bytes)
  private int m_defaultTimeToLive = ICache.DEFAULT_TIME_TO_LIVE;// (in seconds)
  private String m_storageClass = ICache.DEFAULT_STORAGE_CLASS;// class for persisting maps
  private String m_folder = ICache.DEFAULT_FOLDER;// folder
  private String m_filePrefix = ICache.DEFAULT_FILE_PREFIX;// filename prefix
  private boolean m_secure = ICache.DEFAULT_SECURE;// encryption?
  private boolean m_clearCacheOnInit = ICache.DEFAULT_CLEAR_CACHE_ON_INIT;// clear cache on init?
  private boolean m_autoDelayExpiration = ICache.DEFAULT_AUTO_DELAY_EXPIRATION;// auto delay expiration?

  private final String m_id;

  private long m_size;// cache size (in bytes)
  private int m_entryCount;

  private Map m_cache;
  private Hashtable m_strippedCache;
  private StrippedCacheEntry[] m_strippedEntries;

  private int m_head, m_tail;
  private int[] m_next, m_prev;
  private LinkedList m_free;

  private long m_maxEntryCount;
  private long m_addCount;
  private long m_insertCount;
  private long m_removeCount;
  private long m_getCount;
  private long m_hitCount;

  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.cache.persistent.PersistentLRUCache.class);

  // -- construct --

  /**
   * Construct object of class PersistentLRUCache.
   *
   * @param cacheID
   * @param properties
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   */
  public PersistentLRUCache(String cacheID, Properties properties)
    throws CacheException {
    if (properties == null) {
      throw new CacheException("properties for cache " + cacheID + " missing in WCM configuration");
    }

    try {
      m_capacity = new Integer(properties.get(CacheFactory.CFG_CAPACITY_KEY).toString()).intValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(92)", "property " + CacheFactory.CFG_CAPACITY_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_maxCacheSize = new Long(properties.get(CacheFactory.CFG_MAX_CACHE_SIZE_KEY).toString()).longValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(98)", "property " + CacheFactory.CFG_MAX_CACHE_SIZE_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_maxEntrySize = new Long(properties.get(CacheFactory.CFG_MAX_ENTRY_SIZE_KEY).toString()).longValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(104)", "property " + CacheFactory.CFG_MAX_ENTRY_SIZE_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_defaultTimeToLive = new Integer(properties.get(CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY).toString()).intValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(110)", "property " + CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_storageClass = properties.get(CacheFactory.CFG_STORAGE_CLASS_KEY).toString().trim();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(116)", "property " + CacheFactory.CFG_STORAGE_CLASS_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_folder = properties.get(CacheFactory.CFG_FOLDER_KEY).toString().trim();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(122)", "property " + CacheFactory.CFG_FOLDER_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_filePrefix = properties.get(CacheFactory.CFG_FILE_PREFIX_KEY).toString().trim();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(128)", "property " + CacheFactory.CFG_FILE_PREFIX_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_secure = new Boolean(properties.get(CacheFactory.CFG_SECURE_KEY).toString()).booleanValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(134)", "property " + CacheFactory.CFG_SECURE_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_clearCacheOnInit = new Boolean(properties.get(CacheFactory.CFG_CLEAR_CACHE_ON_INIT_KEY).toString()).booleanValue();
    }
    catch (Exception e) {
      s_log.warningT("PersistentLRUCache(140)", "property " + CacheFactory.CFG_CLEAR_CACHE_ON_INIT_KEY + " for cache " + cacheID + " missing in WCM configuration");
    }
    try {
      m_autoDelayExpiration = new Boolean(properties.get(CacheFactory.CFG_AUTO_DELAY_EXPIRATION_KEY).toString()).booleanValue();
    }
    catch (Exception e) {
      s_log.debugT(LoggingFormatter.extractCallstack(e));
    }

    m_id = cacheID;

    m_cache = getPersistentMap(properties);
    if (m_cache == null) {
      throw new CacheException("can't load persistent cache class - check WCM configuration");
    }

    m_strippedCache = new Hashtable(m_capacity);
    m_strippedEntries = new StrippedCacheEntry[m_capacity];

    m_next = new int[m_capacity];
    m_prev = new int[m_capacity];
    m_free = new LinkedList();

    m_size = 0;
    m_entryCount = 0;

    m_head = -1;
    m_tail = -1;

    m_free = new LinkedList();

    for (int i = 0; i < m_capacity; i++) {
      m_strippedEntries[i] = new StrippedCacheEntry(i);
      m_free.add(new Integer(i));
    }

    resetCounters();

    if (!m_clearCacheOnInit) {
      initStripped();
    }
  }

  // -- interfaces --

  /**
   * Get the ID attribute of the PersistentLRUCache object.
   *
   * @return The ID value
   */
  public String getID() {
    return m_id;
  }

  /**
   * Get the Entry attribute of the PersistentLRUCache object.
   *
   * @param key
   * @return The Entry value
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the incoming method parameter
   */
  public ICacheEntry getEntry(String key)
    throws CacheException {
    synchronized (this) {
      m_getCount++;

      Object cachedStrippedObject = m_strippedCache.get(key);
      if (cachedStrippedObject == null) {
        return null;
      }

      StrippedCacheEntry strippedEntry = (StrippedCacheEntry)cachedStrippedObject;
      if (strippedEntry.isExpired()) {
        removeEntry(strippedEntry);
        return null;
      }

      Object cachedObject = m_cache.get(key);
      if (cachedObject == null) {
        removeStrippedEntry(strippedEntry);
        return null;
      }
      SerializablePersistentCacheEntry entry = (SerializablePersistentCacheEntry)cachedObject;

      if (entry.m_autoDelay) {
        entry.m_expirationTime += entry.m_timeToLive * 1000;
        cachePut(key, entry);

        strippedEntry.m_expirationTime += strippedEntry.m_timeToLive * 1000;
      }

      moveToFront(strippedEntry.m_index);

      m_hitCount++;

      return new PersistentCacheEntry(key, entry.m_object, strippedEntry.m_timeToLive, strippedEntry.m_expirationTime, strippedEntry.m_modificationTime, strippedEntry.m_autoDelay, strippedEntry.m_size);
    }
  }

  /**
   * Get the Capacity attribute of the PersistentLRUCache object.
   *
   * @return The Capacity value
   */
  public int getCapacity() {
    return m_capacity;
  }

  /**
   * Get the MaxEntrySize attribute of the PersistentLRUCache object.
   *
   * @return The MaxEntrySize value
   */
  public long getMaxEntrySize() {
    return m_maxEntrySize;
  }

  /**
   * Get the EntryCount attribute of the PersistentLRUCache object.
   *
   * @return The EntryCount value
   */
  public long getEntryCount() {
    return m_entryCount;
  }

  /**
   * Get the MaximumEntryCount attribute of the PersistentLRUCache object.
   *
   * @return The MaximumEntryCount value
   */
  public long getMaximumEntryCount() {
    return m_maxEntryCount;
  }

  /**
   * Get the AddCount attribute of the PersistentLRUCache object.
   *
   * @return The AddCount value
   */
  public long getAddCount() {
    return m_addCount;
  }

  /**
   * Get the InsertCount attribute of the PersistentLRUCache object.
   *
   * @return   The InsertCount value
   */
  public long getInsertCount() {
    return m_insertCount;
  }

  /**
   * Get the RemoveCount attribute of the PersistentLRUCache object.
   *
   * @return The RemoveCount value
   */
  public long getRemoveCount() {
    return m_removeCount;
  }

  /**
   * Get the GetCount attribute of the PersistentLRUCache object.
   *
   * @return The GetCount value
   */
  public long getGetCount() {
    return m_getCount;
  }

  /**
   * Get the HitCount attribute of the PersistentLRUCache object.
   *
   * @return The HitCount value
   */
  public long getHitCount() {
    return m_hitCount;
  }

  /**
   * Get the Size attribute of the PersistentLRUCache object.
   *
   * @return The Size value
   */
  public long getSize() {
    return m_size;
  }

  /**
   * Get the SizeFullyDetermined attribute of the PersistentLRUCache object.
   *
   * @return The SizeFullyDetermined value
   */
  public boolean isSizeFullyDetermined() {
    return true;
  }

  /**
   * Add Entry to the PersistentLRUCache object.
   *
   * @param entry Entry to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(ICacheEntry entry)
    throws CacheException {
    if (entry == null) {
      throw new NullPointerException();
    }

    localAddEntry(entry.getKey(), entry.getObject(), entry.getTimeToLive(), entry.getExpirationTime(), entry.isAutoDelaying());
  }

  /**
   * Add Entry to the PersistentLRUCache object.
   *
   * @param key Entry to be added
   * @param object Entry to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object)
    throws CacheException {
    addEntry(key, object, m_defaultTimeToLive);
  }

  /**
   * Add Entry to the PersistentLRUCache object.
   *
   * @param key Entry to be added
   * @param object Entry to be added
   * @param timeToLive Entry to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object, int timeToLive)
    throws CacheException {
    localAddEntry(key, object, timeToLive, timeToLive == 0 ? 0 : new Date().getTime() + (long)timeToLive * 1000, m_autoDelayExpiration);
  }

  /**
   * Add EntryAutoDelay to the PersistentLRUCache object.
   *
   * @param key EntryAutoDelay to be added
   * @param object EntryAutoDelay to be added
   * @param timeToLive EntryAutoDelay to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive)
    throws CacheException {
    localAddEntry(key, object, timeToLive, timeToLive == 0 ? 0 : new Date().getTime() + (long)timeToLive * 1000, true);
  }

  /**
   * Add Entry to the PersistentLRUCache object.
   *
   * @param key Entry to be added
   * @param object Entry to be added
   * @param timeToLive Entry to be added
   * @param size Entry to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object, int timeToLive, long size)
    throws CacheException {
    // ignore size
    addEntry(key, object, timeToLive);
  }

  /**
   * Add EntryAutoDelay to the PersistentLRUCache object.
   *
   * @param key EntryAutoDelay to be added
   * @param object EntryAutoDelay to be added
   * @param timeToLive EntryAutoDelay to be added
   * @param size EntryAutoDelay to be added
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive, long size)
    throws CacheException {
    // ignore size
    addEntryAutoDelay(key, object, timeToLive);
  }

  /**
   * @param entry
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  public boolean removeEntry(ICacheEntry entry)
    throws CacheException {
    if (entry == null) {
      throw new NullPointerException();
    }

    return removeEntry(entry.getKey());
  }

  /**
   * @param key
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  public boolean removeEntry(String key)
    throws CacheException {
    synchronized (this) {
      Object cachedStrippedObject = m_strippedCache.get(key);
      if (cachedStrippedObject == null) {
        return false;
      }
      removeEntry((StrippedCacheEntry)cachedStrippedObject);
      return true;
    }
  }

  /*
   * public boolean removeEntries(String keyPattern) throws CacheException
   * {
   * boolean result=false;
   * LinkedList keysToRemove=new LinkedList();
   * try
   * {
   * RE regExp=new RE(keyPattern);
   * synchronized(this)
   * {
   * Enumeration keys=m_strippedCache.keys();
   * while(keys!=null && keys.hasMoreElements())
   * {
   * String key=(String)keys.nextElement();
   * if(key!=null && regExp.match(key)) keysToRemove.add(key);
   * }
   * }
   * }
   * catch(RESyntaxException e)
   * {
   * throw new CacheException("PersistentLRUCache.removeEntries(): invalid regular expression");
   * }
   * Iterator iterator=keysToRemove.iterator();
   * while(iterator!=null && iterator.hasNext())
   * {
   * String key=(String)iterator.next();
   * if(key!=null)
   * {
   * removeEntry(key);
   * result|=true;
   * }
   * }
   * return result;
   * }
   */
  /**
   * @param prefix
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  public boolean removeEntriesStartingWith(String prefix)
    throws CacheException {
    boolean result = false;

    LinkedList keysToRemove = new LinkedList();

    synchronized (this) {
      Enumeration keys = m_strippedCache.keys();
      while (keys != null && keys.hasMoreElements()) {
        String key = (String)keys.nextElement();
        if (key != null && key.startsWith(prefix)) {
          keysToRemove.add(key);
        }
      }
    }

    Iterator iterator = keysToRemove.iterator();
    while (iterator != null && iterator.hasNext()) {
      String key = (String)iterator.next();
      if (key != null) {
        removeEntry(key);
        result |= true;
      }
    }

    return result;
  }

  /**
   * @param timestamp
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  public boolean removeEntriesOlderThan(long timestamp)
                                 throws CacheException {

    boolean result = false;
    LinkedList keysToRemove = new LinkedList();
    StrippedCacheEntry entry;

    synchronized(this) {
      Collection entries = m_strippedCache.values();
      Iterator i = entries.iterator();
      while( i.hasNext() ) {
        entry = (StrippedCacheEntry)i.next();
        if(   ( entry != null )
           && ( entry.m_key != null )
           && ( entry.m_modificationTime <= timestamp )
          ) {
          keysToRemove.add(entry.m_key);
        }
      }
    }

    Iterator iterator = keysToRemove.iterator();
    while( iterator != null && iterator.hasNext() ) {
      String key = (String)iterator.next();
      if( key != null ) {
        removeEntry(key);
        result = true;
      }
    }

    return result;

  }

  /**
   * @param key
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  public boolean containsEntry(String key)
    throws CacheException {
    synchronized (this) {
      return m_strippedCache.containsKey(key);
    }
  }

  /**
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the outgoing return value
   * @deprecated as of NW04.
   */
  public Enumeration keys()
    throws CacheException {
    synchronized (this) {
      return m_strippedCache.keys();
    }
  }

  /**
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the outgoing return value
   */
  public Set keySet()
    throws CacheException {
    synchronized (this) {
      return m_strippedCache.keySet();
    }
  }

  /**
   * @return
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the outgoing return value
   */
  public CacheEntryList elements()
    throws CacheException {
    synchronized (this) {
      CacheEntryList result = new CacheEntryList();

      Enumeration elements = m_strippedCache.elements();
      while (elements != null && elements.hasMoreElements()) {
        StrippedCacheEntry strippedEntry = (StrippedCacheEntry)elements.nextElement();
        if (strippedEntry != null) {
          Object cachedObject = m_cache.get(strippedEntry.m_key);
          if (cachedObject != null) {
            SerializablePersistentCacheEntry entry = (SerializablePersistentCacheEntry)cachedObject;
            result.add(new PersistentCacheEntry(strippedEntry.m_key, entry.m_object, strippedEntry.m_timeToLive, strippedEntry.m_expirationTime, strippedEntry.m_modificationTime, strippedEntry.m_autoDelay, strippedEntry.m_size));
          }
        }
      }

      return result;
    }
  }

  /**
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   */
  public void clearCache()
    throws CacheException {
    synchronized (this) {
      m_cache.clear();
      m_strippedCache.clear();

      m_size = 0;
      m_entryCount = 0;

      m_head = -1;
      m_tail = -1;

      m_free.clear();
      for (int i = 0; i < m_capacity; i++) {
        m_free.add(new Integer(i));
      }
    }
  }

  /**
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   */
  public void refresh()
    throws CacheException {
    LinkedList entriesToRemove = new LinkedList();

    synchronized (this) {
      Enumeration keys = m_strippedCache.keys();
      while (keys != null && keys.hasMoreElements()) {
        String key = (String)keys.nextElement();
        if (key != null) {
          StrippedCacheEntry strippedEntry = (StrippedCacheEntry)m_strippedCache.get(key);
          if (strippedEntry != null && strippedEntry.isExpired()) {
            entriesToRemove.add(strippedEntry);
          }
        }
      }
    }

    Iterator iterator = entriesToRemove.iterator();
    while (iterator != null && iterator.hasNext()) {
      StrippedCacheEntry strippedEntry = (StrippedCacheEntry)iterator.next();
      if (strippedEntry != null) {
        removeEntry(strippedEntry);
      }
    }
  }

  /**
   * @todo: Description of the Method.
   */
  public void resetCounters() {
    m_maxEntryCount = 0;
    m_addCount = 0;
    m_insertCount = 0;
    m_removeCount = 0;
    m_getCount = 0;
    m_hitCount = 0;
  }

  /**
   * Get the PersistentMap attribute of the PersistentLRUCache object.
   *
   * @param properties
   * @return The PersistentMap value
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the incoming method parameter
   */
  private Map getPersistentMap(Properties properties)
    throws CacheException {
    Map result = null;

    try {
      Class cacheClass = CrtClassLoaderRegistry.forName(m_storageClass);
      Class[] paramTypes = {String.class, String.class, Boolean.class, Boolean.class};
      Constructor cacheConstructor = cacheClass.getConstructor(paramTypes);
      Object[] initArgs = {m_folder, m_filePrefix, new Boolean(m_clearCacheOnInit), new Boolean(m_secure)};
      result = (Map)cacheConstructor.newInstance(initArgs);
    }
    catch (ClassNotFoundException e) {
      s_log.errorT("getPersistentMap(673)", "persistent cache class not found - check WCM configuration" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
    catch (Exception e) {
      s_log.errorT("getPersistentMap(676)", "can't load persistent cache class - check WCM configuration" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }

    return result;
  }

  // -- private --

  /**
   * @param strippedEntry
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   */
  private void removeEntry(StrippedCacheEntry strippedEntry) {
    if (strippedEntry == null) {
      throw new NullPointerException();
    }

    m_cache.remove(strippedEntry.m_key);
    m_size -= strippedEntry.m_size;
    m_removeCount++;
    removeStrippedEntry(strippedEntry);
  }

  /**
   * @param strippedEntry
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   */
  private void removeStrippedEntry(StrippedCacheEntry strippedEntry) {
    if (strippedEntry == null) {
      throw new NullPointerException();
    }

    m_strippedCache.remove(strippedEntry.m_key);
    remove(strippedEntry.m_index);
  }

  /**
   * @param key
   * @param object
   * @param timeToLive
   * @param expirationTime
   * @param autoDelay
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   */
  private void localAddEntry(String key, Object object, long timeToLive, long expirationTime, boolean autoDelay)
    throws CacheException {
    synchronized (this) {
      if (key == null) {
        throw new NullPointerException();
      }

      Object cachedStrippedObject = m_strippedCache.get(key);
      if (cachedStrippedObject != null) {
        // replace
        StrippedCacheEntry strippedEntry = (StrippedCacheEntry)cachedStrippedObject;

        m_size -= strippedEntry.m_size;
        strippedEntry.m_size = cachePut(key, new SerializablePersistentCacheEntry(object, timeToLive == 0 ? 0 : new Date().getTime() + (long)timeToLive * 1000, timeToLive, autoDelay));
        m_size += strippedEntry.m_size;

        strippedEntry.m_timeToLive = timeToLive;
        strippedEntry.m_expirationTime = expirationTime;
        strippedEntry.updateModificationTime();
        strippedEntry.m_autoDelay = autoDelay;

        moveToFront(strippedEntry.m_index);
      }
      else {
        Object freeObject = m_free.isEmpty() ? null : m_free.removeFirst();
        if (freeObject != null) {
          // append
          int index = ((Integer)freeObject).intValue();
          append(index);
          moveToFront(index);
        }
        else {
          // replace oldest
          m_size -= m_strippedEntries[m_tail].m_size;
          m_cache.remove(m_strippedEntries[m_tail].m_key);
          m_strippedCache.remove(m_strippedEntries[m_tail].m_key);
          moveToFront(m_tail);
        }

        m_strippedEntries[m_head].m_key = key;

        m_strippedEntries[m_head].m_size = cachePut(key, new SerializablePersistentCacheEntry(object, timeToLive, timeToLive == 0 ? 0 : new Date().getTime() + (long)timeToLive * 1000, autoDelay));
        m_size += m_strippedEntries[m_head].m_size;

        m_strippedEntries[m_head].m_timeToLive = timeToLive;
        m_strippedEntries[m_head].m_expirationTime = expirationTime;
        m_strippedEntries[m_head].updateModificationTime();
        m_strippedEntries[m_head].m_autoDelay = autoDelay;

        m_strippedCache.put(key, m_strippedEntries[m_head]);
        m_insertCount++;
      }

      // drop entries to stay under max size
      while (m_maxCacheSize > 0 && m_size > m_maxCacheSize) {
        removeEntry(m_strippedEntries[m_tail]);
      }

      m_addCount++;
    }
  }

  /**
   * @param key
   * @param entry
   * @return
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  private long cachePut(String key, SerializablePersistentCacheEntry entry) {
    // returns size

    Object length = m_cache.put(key, entry);
    if (length == null) {
      return 0;
    }
    return new Long(length.toString()).longValue();
  }

  /**
   * @param index
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   */
  private void moveToFront(int index) {
    if (m_head == index) {
      return;
    }

    if (m_tail == index && m_prev[index] >= 0) {
      m_tail = m_prev[index];
    }

    if (m_prev[index] >= 0) {
      m_next[m_prev[index]] = m_next[index];
    }
    if (m_next[index] >= 0) {
      m_prev[m_next[index]] = m_prev[index];
    }

    m_prev[m_head] = index;

    m_prev[index] = -1;
    m_next[index] = m_head;

    m_head = index;
  }

  /**
   * @param index
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   */
  private void append(int index) {
    if (m_head < 0 || m_tail < 0) {
      m_head = m_tail = index;
      m_prev[index] = m_next[index] = -1;
      m_entryCount = 1;
      if (m_entryCount > m_maxEntryCount) {
        m_maxEntryCount = m_entryCount;
      }
      return;
    }

    m_next[m_tail] = index;
    m_next[index] = -1;
    m_prev[index] = m_tail;
    m_tail = index;

    m_entryCount++;
    if (m_entryCount > m_maxEntryCount) {
      m_maxEntryCount = m_entryCount;
    }
  }

  /**
   * @param index
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   */
  private void remove(int index) {
    if (m_prev[index] >= 0) {
      m_next[m_prev[index]] = m_next[index];
    }
    if (m_next[index] >= 0) {
      m_prev[m_next[index]] = m_prev[index];
    }

    if (m_head == index) {
      m_head = m_next[index];
    }
    if (m_tail == index) {
      m_tail = m_prev[index];
    }

    m_free.add(new Integer(index));

    m_entryCount--;
  }

  /**
   * @exception CacheException Exception raised in failure situation
   * @todo: Description of the Method.
   */
  private void initStripped()
    throws CacheException {
    Set keys = m_cache.keySet();
    if (keys == null) {
      return;
    }

    Iterator iterator = keys.iterator();
    while (iterator != null && iterator.hasNext()) {
      String key = (String)iterator.next();
      if (key != null) {
        Object cachedObject = m_cache.get(key);
        if (cachedObject != null) {
          SerializablePersistentCacheEntry entry = (SerializablePersistentCacheEntry)cachedObject;

          if (entry.isExpired()) {
            m_cache.remove(key);
          }
          else {
            localAddEntry(key, entry.m_object, entry.m_timeToLive, entry.m_expirationTime, entry.m_autoDelay);
          }
        }
      }
    }
  }

  /**
   * @todo: Description of the class.
   */
  class StrippedCacheEntry {
    // does not hold the object

    public String m_key = null;
    public long m_timeToLive;
    public long m_expirationTime = 0;// 0: never expires
    public long m_modificationTime;
    public long m_size = 0;
    public boolean m_autoDelay = false;
    public int m_index = 0;

    /**
     * Construct object of class StrippedCacheEntry.
     *
     * @param index
     * @todo: Description of the incoming method parameter
     */
    public StrippedCacheEntry(int index) {
      m_index = index;
    }

    /**
     * Get the Expired attribute of the StrippedCacheEntry object.
     *
     * @return The Expired value
     */
    public boolean isExpired() {
      return m_expirationTime == 0 ? false : m_expirationTime < new Date().getTime();
    }

    /**
     * @todo: Description of the Method.
     */
    public void updateModificationTime() {
      m_modificationTime = new Date().getTime();
    }
  }
}
