/*
 * Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 * Created on 09.03.2004
 */

package com.sapportals.wcm.util.cache.memory.optimized;

import java.util.*;

/**
 * Multithreading-safe implementation of a least recently used (LRU) cache,
 * i.e. cache entries are displaced in reverse order of usage. Usage is
 * defined as put() operation with a new key (initial), as put() operation
 * with a used key (update) and as all get() operations on a cache entry.
 * 
 * Note: This cache trades memory usage for computational speed, i.e. the
 * data structures have been designed to implement all operation in O(1).
 * This is achieved by holding a hash map for fast key lookup and by holding
 * a set of arrays for maintaining the LRU order. This cache isn't implemented
 * with the help of a maintenance thread, but replaces cache entries on demand,
 * i.e. the cache is also suitable for interrupt-free realtime applications.
 * 
 * Note: All data on a cache entry is stored in separate arrays. No uniting
 * cache entry is used for that purpose, because of memory and CPU usage reasons.
 * 
 * Note: Cache entries are equiped with an expiration timestamp. This timestamp
 * is checked on access only. If the cache entry is expired on access, it will
 * be removed. This might lead to displacment of valid, but long ago used cache
 * entries, while expired cache entries are still hold. Trashing would be the
 * consequence. This might become a problem with short-lived cache entries and
 * longer-timed cache access on longer-lived cache entries. Statistical information
 * should help to identify that problem and to take steps against that situation.
 */
public class StringKeyMemoryCache
{
  // Configurated members of this cache instance
  private final int capacity;
  private final int defaultTimetolive;
  private final boolean defaultRefreshing;

  // Data structure used for fast lookup of key to index of internal data structures in:
  // Best case in direct lookup: O(1); worst case if collisions arise: O(n)
  private final int hashCapacity;
  private final int hashSqrt;
  private final String[] hashKeys;
  private final int[] hashValues;

  // Data structures used for cached entry data accessed by index in: O(1)
  private final String[] keyArray;
  private final Object[] valueArray;
  private final long[] modificationArray;
  private final long[] expirationArray;
  private final int[] timetoliveArray;
  private final boolean[] refreshingArray;

  // Data structures used for LRU algorithm accessed by index in: O(1)
  private final int[] prevArray;
  private final int[] nextArray;

  // Indizes of head/tail of used/free entries
  private int headOfUsed;
  private int tailOfUsed;
  private int headOfFree;

  // Counter for statistical purposes
  private int actEntryCount;
  private int maxEntryCount;
  private int putCount;
  private int putNewCount;
  private int putNewDisplacingCount;
  private int removeCount;
  private int removeHitCount;
  private int getCount;
  private int getHitCount;
  private int getHitExpiredCount;
  private int hashPutCount;
  private int hashCollisionPutCount;
  private int hashRemoveCount;
  private int hashCollisionRemoveCount;
  private int hashGetCount;
  private int hashCollisionGetCount;
  private int hashRehashCount;

  public StringKeyMemoryCache(int capacity)
  {
    this(capacity, 0, false);
  }

  public StringKeyMemoryCache(int capacity, int defaultTimetolive, boolean defaultRefreshing)
  {
    // Store capacity if valid
    if (capacity < 2)
    {
      // Throw an exception if the capacity is less than 2, because the
      // cache builds on at least 2 cache entries (minimum constraint).
      throw new RuntimeException("Capacity may not be less than 2!");
    }
    this.capacity = capacity;
    this.defaultTimetolive = defaultTimetolive;
    this.defaultRefreshing = defaultRefreshing;

    // Allocate memory for all data structures
    hashCapacity = PrimeFinder.findNextPrime((this.capacity << 1) + 1);
    hashSqrt = (int)Math.sqrt(hashCapacity);
    hashKeys = new String[hashCapacity];
    hashValues = new int[hashCapacity];
    keyArray = new String[this.capacity + 1];
    valueArray = new Object[this.capacity + 1];
    modificationArray = new long[this.capacity + 1];
    expirationArray = new long[this.capacity + 1];
    timetoliveArray = new int[this.capacity + 1];
    refreshingArray = new boolean[this.capacity + 1];
    prevArray = new int[this.capacity + 1];
    nextArray = new int[this.capacity + 1];
    headOfUsed = 0;
    tailOfUsed = 0;
    headOfFree = 1;
    for (int i = 1; i < this.capacity;)
    {
      prevArray[i] = ++i;
    }
    putCount = 0;
    putNewCount = 0;
    putNewDisplacingCount = 0;
    removeCount = 0;
    removeHitCount = 0;
    getCount = 0;
    getHitCount = 0;
    getHitExpiredCount = 0;
    hashPutCount = 0;
    hashCollisionPutCount = 0;
    hashRemoveCount = 0;
    hashCollisionRemoveCount = 0;
    hashGetCount = 0;
    hashCollisionGetCount = 0;
    hashRehashCount = 0;
  }

  public synchronized void clear()
  {
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        actEntryCount--;

        // Remove entry from map
        internalRemoveIndexFromHash(keyArray[index]);

        // Remove expired entry
        internalRemoveEntry(index);
      }
    }
  }

  public synchronized void refresh()
  {
    long expiration;
    long timestamp = System.currentTimeMillis();
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        expiration = expirationArray[index];
        if ((expiration != 0) && (expiration < timestamp))
        {
          actEntryCount--;

          // Remove entry from map
          internalRemoveIndexFromHash(keyArray[index]);

          // Remove expired entry
          internalRemoveEntry(index);
        }
      }
    }
  }

  public void putEntry(StringKeyMemoryCacheEntry entry)
  {
    putEntry(
      entry.getKey(),
      entry.getValue(),
      entry.getExpirationTime(),
      (int)entry.getTimeToLive(),
      entry.isRefreshing());
  }

  public void putEntry(String key, Object value)
  {
    putEntry(key, value, System.currentTimeMillis() + defaultTimetolive * 1000, defaultTimetolive, defaultRefreshing);
  }

  public void putEntry(String key, Object value, long expiration)
  {
    putEntry(key, value, expiration, 0, false);
  }

  public void putEntry(String key, Object value, int timetolive, boolean refreshing)
  {
    putEntry(key, value, System.currentTimeMillis() + timetolive * 1000, timetolive, refreshing);
  }

  public synchronized void putEntry(String key, Object value, long expiration, int timetolive, boolean refreshing)
  {
    putCount++;
    int index = internalGetIndexFromHash(key);
    if (index != 0)
    {
      // Update entry
      valueArray[index] = value;
      modificationArray[index] = System.currentTimeMillis();
      expirationArray[index] = expiration;
      timetoliveArray[index] = timetolive;
      refreshingArray[index] = refreshing;

      // Refresh entry by moving it up to head of used entries
      internalMoveToHeadOfUsed(index);
    }
    else
    {
      putNewCount++;

      // Check for full cache
      if (headOfFree == 0)
      {
        putNewDisplacingCount++;

        // Fix head of free indizes
        headOfFree = tailOfUsed;
        prevArray[headOfFree] = 0;

        // Fix tail of used indizes
        tailOfUsed = nextArray[tailOfUsed];
        prevArray[tailOfUsed] = 0;

        // Remove entry from map
        internalRemoveIndexFromHash(keyArray[headOfFree]);

        // Drop hold references in data structures, so that the GC can collect the garbage
        keyArray[index] = null;
        valueArray[index] = null;
      }
      else
      {
        if (actEntryCount++ > maxEntryCount)
        {
          maxEntryCount++;
        }
      }

      // Get free index
      index = headOfFree;
      headOfFree = prevArray[headOfFree];

      // Point the last head of used indizes to this index forwards
      if (headOfUsed != 0)
      {
        nextArray[headOfUsed] = index;
      }
      else
      {
        tailOfUsed = index;
      }

      // Fix head of used indizes
      prevArray[index] = headOfUsed;
      nextArray[index] = 0;
      headOfUsed = index;

      // Add entry to map
      internalPutIndexToHash(key, index);

      // Fill entry
      keyArray[index] = key;
      valueArray[index] = value;
      modificationArray[index] = System.currentTimeMillis();
      expirationArray[index] = expiration;
      timetoliveArray[index] = timetolive;
      refreshingArray[index] = refreshing;
    }
    return;
  }

  public boolean removeEntry(StringKeyMemoryCacheEntry entry)
  {
    return removeEntry(entry.getKey());
  }

  public synchronized boolean removeEntry(String key)
  {
    removeCount++;
    int index = internalRemoveIndexFromHash(key);
    if (index != 0)
    {
      removeHitCount++;
      actEntryCount--;

      // Remove entry
      internalRemoveEntry(index);
      return true;
    }
    else
    {
      return false;
    }
  }

  public synchronized boolean removeEntriesStartingWith(String keyPrefix)
  {
    int lastEntryCount = actEntryCount;
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        String key = keyArray[index];
        if ((key != null) && (key.startsWith(keyPrefix)))
        {
          actEntryCount--;

          // Remove entry from map
          internalRemoveIndexFromHash(key);

          // Remove expired entry
          internalRemoveEntry(index);
        }
      }
    }
    return lastEntryCount != actEntryCount;
  }

  public synchronized boolean removeEntriesOlderThan(long timestamp)
  {
    int lastEntryCount = actEntryCount;
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        if (modificationArray[index] < timestamp)
        {
          actEntryCount--;

          // Remove entry from map
          internalRemoveIndexFromHash(keyArray[index]);

          // Remove expired entry
          internalRemoveEntry(index);
        }
      }
    }
    return lastEntryCount != actEntryCount;
  }

  public synchronized void getEntry(StringKeyMutableMemoryCacheEntry entry)
  {
    long expiration;
    String key = entry.getKey();
    getCount++;
    int index = internalGetIndexFromHash(key);
    if (index != 0)
    {
      getHitCount++;
      expiration = expirationArray[index];
      if ((expiration != 0) && (expiration < System.currentTimeMillis()))
      {
        getHitExpiredCount++;
        actEntryCount--;

        // Clear supplied entry
        entry.clearEntry();

        // Remove entry from map
        internalRemoveIndexFromHash(keyArray[index]);

        // Remove expired entry
        internalRemoveEntry(index);
      }
      else
      {
        // Complete supplied entry
        entry.completeEntry(
          valueArray[index],
          modificationArray[index],
          expiration,
          timetoliveArray[index],
          refreshingArray[index]);

        // Refresh entry by moving it up to head of used entries
        internalMoveToHeadOfUsed(index);
      }
    }
    else
    {
      // Clear supplied entry
      entry.clearEntry();
    }
    return;
  }

  public synchronized Object getEntryValue(String key)
  {
    long expiration;
    Object value = null;
    getCount++;
    int index = internalGetIndexFromHash(key);
    if (index != 0)
    {
      getHitCount++;
      expiration = expirationArray[index];
      if ((expiration != 0) && (expiration < System.currentTimeMillis()))
      {
        getHitExpiredCount++;
        actEntryCount--;

        // Remove entry from map
        internalRemoveIndexFromHash(keyArray[index]);

        // Remove expired entry
        internalRemoveEntry(index);
      }
      else
      {
        // Extract value of entry
        value = valueArray[index];

        // Refresh entry by moving it up to head of used entries
        internalMoveToHeadOfUsed(index);
      }
    }
    return value;
  }

  public synchronized long getEntryModification(String key)
  {
    long expiration;
    long modification = 0;
    getCount++;
    int index = internalGetIndexFromHash(key);
    if (index != 0)
    {
      getHitCount++;
      expiration = expirationArray[index];
      if ((expiration != 0) && (expiration < System.currentTimeMillis()))
      {
        getHitExpiredCount++;
        actEntryCount--;

        // Remove entry from map
        internalRemoveIndexFromHash(keyArray[index]);

        // Remove expired entry
        internalRemoveEntry(index);
      }
      else
      {
        // Extract modification of entry
        modification = modificationArray[index];

        // Refresh entry by moving it up to head of used entries
        internalMoveToHeadOfUsed(index);
      }
    }
    return modification;
  }

  public synchronized Set getKeys()
  {
    Set result = new HashSet();
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        // Add key to result
        result.add(keyArray[index]);
      }
    }
    return result;
  }

  public synchronized List getEntries()
  {
    List result = new ArrayList();
    for (int index = 1; index < valueArray.length; index++)
    {
      if (valueArray[index] != null)
      {
        // Add entry to result
        result.add(
          new StringKeyMemoryCacheEntry(
            keyArray[index],
            valueArray[index],
            modificationArray[index],
            expirationArray[index],
            timetoliveArray[index],
            refreshingArray[index]));
      }
    }
    return result;
  }

  public int getCapacity()
  {
    return capacity;
  }

  public int getActEntryCount()
  {
    return actEntryCount;
  }

  public int getMaxEntryCount()
  {
    return maxEntryCount;
  }

  public int getPutCount()
  {
    return putCount;
  }

  public int getPutUpdateCount()
  {
    return putCount - putNewCount;
  }

  public int getPutNewCount()
  {
    return putNewCount;
  }

  public int getPutNewDisplacingCount()
  {
    return putNewDisplacingCount;
  }

  public int getPutNewAddingCount()
  {
    return putNewCount - putNewDisplacingCount;
  }

  public int getRemoveCount()
  {
    return removeCount;
  }

  public int getRemoveHitCount()
  {
    return removeHitCount;
  }

  public int getRemoveMissCount()
  {
    return removeCount - removeHitCount;
  }

  public int getGetCount()
  {
    return getCount;
  }

  public int getGetHitCount()
  {
    return getHitCount;
  }

  public int getGetHitExpiredCount()
  {
    return getHitExpiredCount;
  }

  public int getGetHitValidCount()
  {
    return getHitCount - getHitExpiredCount;
  }

  public int getGetMissCount()
  {
    return getCount - getHitCount;
  }

  public int getHashCapacity()
  {
    return hashCapacity;
  }

  public int getHashPutCount()
  {
    return hashPutCount;
  }

  public int getHashCollisionPutCount()
  {
    return hashCollisionPutCount;
  }

  public int getHashRemoveCount()
  {
    return hashRemoveCount;
  }

  public int getHashCollisionRemoveCount()
  {
    return hashCollisionRemoveCount;
  }

  public int getHashGetCount()
  {
    return hashGetCount;
  }

  public int getHashCollisionGetCount()
  {
    return hashCollisionGetCount;
  }

  public int getHashRehashCount()
  {
    return hashRehashCount;
  }

  public synchronized void getStatistic(CacheStatistic statistic)
  {
    // Store all statistic data
    statistic.setCache(this);
    statistic.setCapacity(getCapacity());
    statistic.setActEntryCount(getActEntryCount());
    statistic.setMaxEntryCount(getMaxEntryCount());
    statistic.setPutCount(getPutCount());
    statistic.setPutUpdateCount(getPutUpdateCount());
    statistic.setPutNewCount(getPutNewCount());
    statistic.setPutNewDisplacingCount(getPutNewDisplacingCount());
    statistic.setPutNewAddingCount(getPutNewAddingCount());
    statistic.setRemoveCount(getRemoveCount());
    statistic.setRemoveHitCount(getRemoveHitCount());
    statistic.setRemoveMissCount(getRemoveMissCount());
    statistic.setGetCount(getGetCount());
    statistic.setGetHitCount(getGetHitCount());
    statistic.setGetHitValidCount(getGetHitValidCount());
    statistic.setGetHitExpiredCount(getGetHitExpiredCount());
    statistic.setGetMissCount(getGetMissCount());
    statistic.setHashCapacity(getHashCapacity());
    statistic.setHashPutCount(getHashPutCount());
    statistic.setHashCollisionPutCount(getHashCollisionPutCount());
    statistic.setHashRemoveCount(getHashRemoveCount());
    statistic.setHashCollisionRemoveCount(getHashCollisionRemoveCount());
    statistic.setHashGetCount(getHashGetCount());
    statistic.setHashCollisionGetCount(getHashCollisionGetCount());
    statistic.setHashRehashCount(getHashRehashCount());
  }

  public synchronized void resetStatistic()
  {
    // Store all statistic data
    maxEntryCount = actEntryCount;
    putCount = 0;
    putNewCount = 0;
    putNewDisplacingCount = 0;
    removeCount = 0;
    removeHitCount = 0;
    getCount = 0;
    getHitCount = 0;
    getHitExpiredCount = 0;
    hashPutCount = 0;
    hashCollisionPutCount = 0;
    hashRemoveCount = 0;
    hashCollisionRemoveCount = 0;
    hashGetCount = 0;
    hashCollisionGetCount = 0;
    hashRehashCount = 0;
  }

  /**
   * Internal method called to refresh an entry by moving it up to the head of used
   * entries.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private void internalMoveToHeadOfUsed(int index)
  {
    if (index != headOfUsed)
    {
      // Fix previous and next indizes
      int prevIndex = prevArray[index];
      int nextIndex = nextArray[index];
      if (prevIndex != 0)
      {
        // Point the previous index to the next index forwards
        nextArray[prevIndex] = nextIndex;
      }
      else
      {
        // Fix tail of used indizes
        tailOfUsed = nextIndex;
      }
      prevArray[nextIndex] = prevIndex;

      // Point the last head of used indizes to this index forwards
      nextArray[headOfUsed] = index;

      // Fix head of used indizes
      prevArray[index] = headOfUsed;
      nextArray[index] = 0;
      headOfUsed = index;
    }

    // Refresh expiration if feature is activated
    if (refreshingArray[index])
    {
      if (expirationArray[index] != 0)
      {
        expirationArray[index] = System.currentTimeMillis() + timetoliveArray[index] * 1000;
      }
    }
  }

  /**
   * Internal method called to remove an entry indexed by the given index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private void internalRemoveEntry(int index)
  {
    // Fix previous and next indizes
    int prevIndex = prevArray[index];
    int nextIndex = nextArray[index];
    if (prevIndex != 0)
    {
      // Point the previous index to the next index forwards
      nextArray[prevIndex] = nextIndex;
    }
    else
    {
      // Fix tail of used indizes
      tailOfUsed = nextIndex;
    }
    if (nextIndex != 0)
    {
      // Point the next index to the previous index backwards
      prevArray[nextIndex] = prevIndex;
    }
    else
    {
      // Fix head of used indizes
      headOfUsed = prevIndex;
    }

    // Fix head of free indizes
    prevArray[index] = headOfFree;
    headOfFree = index;

    // Drop hold references in data structures, so that the GC can collect the garbage
    keyArray[index] = null;
    valueArray[index] = null;
  }

  /**
   * Internal method to compute a hash value from a key.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private int internalComputeHash(String key)
  {
    // Compute a hash value from a key in the range of 0 to hashCapacity-1
    return (key.hashCode() & 0x7fffffff) % hashCapacity;
  }

  /**
   * Internal method to rehash the hash if necessary (used in removal).
   * 
   * Note: Rehashing is done, when an entry in a collision chain
   * has been removed. Without rehashing the entries behind the removed
   * entry could no longer found, if their original hash would lead to
   * an entry before the removed entry. Rehashing is the process of
   * removing all entries behind the removed entry and adding them
   * again to the map. Thereby they could be assigned to different slots
   * or the same. The point is, that if their hash would lead to an
   * entry before the removed entry, the first of them will fall into
   * the slot of the removed entry and the others will follow. 
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private void internalRehashHash(int hash)
  {
    String key;
    int value;
    hash = (hash + hashSqrt) % hashCapacity;

    // While there is still an entry to be rehashed
    while (hashValues[hash] != 0)
    {
      // Get next value, empty slot and add entry again
      hashRehashCount++;
      key = hashKeys[hash];
      value = hashValues[hash];
      hashKeys[hash] = null;
      hashValues[hash] = 0;
      internalPutIndexToHash(key, value);
      hash = (hash + hashSqrt) % hashCapacity;
    }
  }

  /**
   * Internal method to put an index to the fast lookup hash data structure.
   * 
   * Note: The index may not be null.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private int internalPutIndexToHash(String key, int index)
  {
    hashPutCount++;
    int value;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (!key.equals(hashKeys[hash]))
    {
      // While we are in a collision
      while (hashValues[hash] != 0)
      {
        // Compute next hash
        hashCollisionPutCount++;
        hash = (hash + hashSqrt) % hashCapacity;

        // Check if we have found, what we were looking for
        if (key.equals(hashKeys[hash]))
        {
          value = hashValues[hash];
          hashValues[hash] = index;
          return value;
        }
      }
      hashKeys[hash] = key;
      hashValues[hash] = index;
      return 0;
    }
    value = hashValues[hash];
    hashValues[hash] = index;
    return value;
  }

  /**
   * Internal method to remove an index from the fast lookup hash data structure.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private int internalRemoveIndexFromHash(String key)
  {
    hashRemoveCount++;
    int value;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (!key.equals(hashKeys[hash]))
    {
      // While we are in a collision
      while (hashValues[hash] != 0)
      {
        // Compute next hash
        hashCollisionRemoveCount++;
        hash = (hash + hashSqrt) % hashCapacity;

        // Check if we have found, what we were looking for
        if (key.equals(hashKeys[hash]))
        {
          value = hashValues[hash];
          hashKeys[hash] = null;
          hashValues[hash] = 0;
          internalRehashHash(hash);
          return value;
        }
      }
      return 0;
    }
    value = hashValues[hash];
    hashKeys[hash] = null;
    hashValues[hash] = 0;
    internalRehashHash(hash);
    return value;
  }

  /**
   * Internal method to get an index from the fast lookup hash data structure.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private int internalGetIndexFromHash(String key)
  {
    hashGetCount++;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (!key.equals(hashKeys[hash]))
    {
      // While we are in a collision
      while (hashValues[hash] != 0)
      {
        // Compute next hash
        hashCollisionGetCount++;
        hash = (hash + hashSqrt) % hashCapacity;

        // Check if we have found, what we were looking for
        if (key.equals(hashKeys[hash]))
        {
          return hashValues[hash];
        }
      }
      return 0;
    }
    return hashValues[hash];
  }

  public static void mainFunction()
  {
    final StringKeyMemoryCache cache = new StringKeyMemoryCache(3);

    cache.putEntry("1", "1", 0, 0, false);
    cache.putEntry("2", "2", 0, 0, false);
    cache.removeEntry("3");
    cache.removeEntry("2");
    cache.putEntry("3", "3", 0, 0, false);
    cache.putEntry("4", "4", 0, 0, false);

    System.out.println(cache.getEntryValue("12"));
    System.out.println(cache.getEntryValue("7"));
    System.out.println(cache.getEntryValue("4"));
    System.out.println(cache.getEntryValue("3"));
    System.out.println(cache.getEntryValue("2"));
    System.out.println(cache.getEntryValue("1"));

    cache.putEntry("4", "4", 0, 0, false);
    cache.putEntry("5", "5", 0, 0, false);

    System.out.println(cache.getEntryValue("5"));
    System.out.println(cache.getEntryValue("4"));
    System.out.println(cache.getEntryValue("2"));
    System.out.println(cache.getEntryValue("3"));
    System.out.println(cache.getEntryValue("1"));

    CacheStatistic statistic = new CacheStatistic();
    cache.getStatistic(statistic);
    System.out.println(statistic.getStatisticAsString());
    System.out.println(statistic.getRecommendationsAsString());
  }

  public static void mainPerformance()
  {
    long startTime;
    long durationTime;
    long startMem1;
    long startMem2;
    long usedMem1;
    long usedMem2;

    try
    {
      // Create cache and raw data
      System.gc();
      Thread.yield();
      System.gc();
      startMem1 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      final StringKeyMemoryCache cache = new StringKeyMemoryCache(CacheStatistic.SIZE);
      startMem2 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      Object[] objs = new Object[CacheStatistic.SIZE];
      for (int i = 0; i < objs.length; i++)
      {
        objs[i] = new Integer(i).toString();
      }

      startTime = new Date().getTime();
      for (int j = 0; j < objs.length; j++)
      {
        cache.putEntry((String)objs[j], objs[j], 0, 0, false);
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          cache.putEntry((String)objs[j], objs[j], 0, 0, false);
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for re-adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntryValue((String)objs[j]) != objs[j])
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering existent cache entries: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntryValue(objs[j] + "x") != null)
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering non-existent cache entries: " + durationTime);

      // Dump used memory
      usedMem1 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);
      startTime = new Date().getTime();
      System.gc();
      Thread.yield();
      System.gc();
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for GC: " + durationTime);
      usedMem1 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);

      CacheStatistic statistic = new CacheStatistic();
      cache.getStatistic(statistic);
      System.out.println(statistic.getStatisticAsString());
      System.out.println(statistic.getRecommendationsAsString());

      Thread.sleep(10000);
    }
    catch (Throwable throwable)
    {
            //$JL-EXC$      
      System.err.println("Error!");
      throwable.printStackTrace();
    }
  }

  public static void mainThreading()
  {
    final StringKeyMemoryCache cache = new StringKeyMemoryCache(CacheStatistic.SIZE);
    final Random random = new Random(System.currentTimeMillis());

    // Populate
    for (int i = 0; i < CacheStatistic.SIZE; i++)
    {
      // Add cache entry
      Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
      cache.putEntry((String)obj, obj, 0, 0, false);
    }

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Add cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.putEntry((String)obj, obj, 0, 0, false);
            // System.currentTimeMillis() + random.nextInt() % (60 * 1000));

            // Go to sleep
            Thread.sleep(10);
          }
          catch (Exception exception)
          {
            //$JL-EXC$            
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Remove cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.removeEntry((String)obj);

            // Go to sleep
            Thread.sleep(100);
          }
          catch (Exception exception)
          {
            //$JL-EXC$            
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.getEntryValue((String)obj);

            // Go to next
            Thread.yield();
          }
          catch (Exception exception)
          {
            //$JL-EXC$            
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            CacheStatistic statistic = new CacheStatistic();
            cache.getStatistic(statistic);
            System.out.println(statistic.getStatisticAsString());
            System.out.println(statistic.getRecommendationsAsString());

            // Go to sleep
            Thread.sleep(1000);
          }
          catch (Exception exception)
          {
            //$JL-EXC$            
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();
  }

  public static void mainLocking()
  {
    final StringKeyMemoryCache cache = new StringKeyMemoryCache(CacheStatistic.SIZE);
    final Random random = new Random(System.currentTimeMillis());

    // Populate
    for (int i = 0; i < CacheStatistic.SIZE; i++)
    {
      // Add cache entry
      Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
      cache.putEntry((String)obj, obj, 0, 0, false);
    }

    // Prepare data
    Thread[] readers = new Thread[CacheStatistic.READERS];
    Thread[] writers = new Thread[CacheStatistic.WRITERS];
    cache.resetStatistic();
    System.gc();
    Thread.yield();
    System.gc();

    // Create readers
    for (int i = 0; i < CacheStatistic.READERS; i++)
    {
      readers[i] = new Thread(new Runnable()
      {
        public void run()
        {
          long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
          while (System.currentTimeMillis() < exitTime)
          {
            // Safeguard operations
            try
            {
              // Get cache entry
              Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
              cache.getEntryValue((String)obj);
            }
            catch (Exception exception)
            {
            //$JL-EXC$              
              // Handle caught exception
              System.err.println("Error!");
              exception.printStackTrace();
            }
          }
        }
      });
    }

    // Create writers
    for (int i = 0; i < CacheStatistic.WRITERS; i++)
    {
      writers[i] = new Thread(new Runnable()
      {
        public void run()
        {
          long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
          while (System.currentTimeMillis() < exitTime)
          {
            // Safeguard operations
            try
            {
              // Put cache entry
              Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
              cache.putEntry((String)obj, obj, 0, 0, false);
            }
            catch (Exception exception)
            {
            //$JL-EXC$              
              // Handle caught exception
              System.err.println("Error!");
              exception.printStackTrace();
            }
          }
        }
      });
    }

    // Start readers
    for (int i = 0; i < CacheStatistic.READERS; i++)
    {
      readers[i].start();
    }

    // Start writers
    for (int i = 0; i < CacheStatistic.WRITERS; i++)
    {
      writers[i].start();
    }

    // Join readers
    try
    {
      Thread.sleep(CacheStatistic.REP);
      for (int i = 0; i < CacheStatistic.READERS; i++)
      {
        readers[i].join();
      }

      // Join writers
      for (int i = 0; i < CacheStatistic.WRITERS; i++)
      {
        writers[i].join();
      }
    }
    catch (Exception exception)
    {
            //$JL-EXC$      
      // Handle caught exception
      System.err.println("Error!");
      exception.printStackTrace();
    }

    // Output throughput
    System.out.println(
      "Locking Test Throughput: Gets: "
        + (cache.getGetCount() / 1000)
        + " K   Puts: "
        + (cache.getPutCount() / 1000)
        + " K");
  }

  public static void main(String[] args)
  {
    try
    {
      if (CacheStatistic.TEST_FUNCTION)
      {
        mainFunction();
      }
      if (CacheStatistic.TEST_PERFORMANCE)
      {
        mainPerformance();
      }
      if (CacheStatistic.TEST_THREADING)
      {
        mainThreading();
      }
      if (CacheStatistic.TEST_LOCKING)
      {
        mainLocking();
      }
    }
    catch (Exception exception)
    {
                  //$JL-EXC$
      exception.printStackTrace();
    }
  }
}
