/*
 * Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 * Created on 09.03.2004
 */

package com.sapportals.wcm.util.cache.memory.optimized;

import java.util.*;

/**
 * Multithreading-safe implementation of a least recently used (LRU) cache,
 * i.e. cache entries are displaced in reverse order of usage. Usage is
 * defined as put() operation with a new key (initial), as put() operation
 * with a used key (update) and as all get() operations on a cache entry.
 * 
 * Note: This cache trades memory usage for computational speed, i.e. the
 * data structures have been designed to implement all operation in O(1).
 * This is achieved by holding a hash map for fast key lookup and by holding
 * a set of arrays for maintaining the LRU order. This cache isn't implemented
 * with the help of a maintenance thread, but replaces cache entries on demand,
 * i.e. the cache is also suitable for interrupt-free realtime applications.
 * 
 * Note: All data on a cache entry is stored in separate arrays. No uniting
 * cache entry is used for that purpose, because of memory and CPU usage reasons.
 * 
 * Note: Cache entries are equiped with an expiration timestamp. This timestamp
 * is checked on access only. If the cache entry is expired on access, it will
 * be removed. This might lead to displacment of valid, but long ago used cache
 * entries, while expired cache entries are still hold. Trashing would be the
 * consequence. This might become a problem with short-lived cache entries and
 * longer-timed cache access on longer-lived cache entries. Statistical information
 * should help to identify that problem and to take steps against that situation.
 */
public class ObjectObjectFastHashMap
{
  // Data structure used for fast lookup of key to index of internal data structures in:
  // Best case in direct lookup: O(1); worst case if collisions arise: O(n)
  private final int _hashCapacity;
  private final int _hashSqrt;
  private final Object[] _hashKeys;
  private final Object[] _hashValues;

  public ObjectObjectFastHashMap(int capacity)
  {
    // Allocate memory for all data structures
    _hashCapacity = PrimeFinder.findNextPrime((capacity << 1) + 1);
    _hashSqrt = (int)Math.sqrt(_hashCapacity);
    _hashKeys = new Object[_hashCapacity];
    _hashValues = new Object[_hashCapacity];
  }

  /**
   * Internal method to compute a hash value from a key.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private int internalComputeHash(Object key)
  {
    // Compute a hash value from a key in the range of 0 to _hashCapacity-1
    return (key.hashCode() & 0x7fffffff) % _hashCapacity;
  }

  /**
   * Internal method to rehash the hash if necessary (used in removal).
   * 
   * Note: Rehashing is done, when an entry in a collision chain
   * has been removed. Without rehashing the entries behind the removed
   * entry could no longer found, if their original hash would lead to
   * an entry before the removed entry. Rehashing is the process of
   * removing all entries behind the removed entry and adding them
   * again to the map. Thereby they could be assigned to different slots
   * or the same. The point is, that if their hash would lead to an
   * entry before the removed entry, the first of them will fall into
   * the slot of the removed entry and the others will follow. 
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  private final void internalRehashHash(int hash, Object[] hashKeys, Object[] hashValues)
  {
    Object key;
    Object value;
    hash = (hash + _hashSqrt) % _hashCapacity;

    // While there is still an entry to be rehashed
    while (hashValues[hash] != null)
    {
      // Get next value, empty slot and add entry again
      key = hashKeys[hash];
      value = hashValues[hash];
      hashKeys[hash] = null;
      hashValues[hash] = null;
      put(key, value);
      hash = (hash + _hashSqrt) % _hashCapacity;
    }
  }

  /**
   * Internal method to put an index to the fast lookup hash data structure.
   * 
   * Note: The index may not be null.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  public final Object put(Object key, Object newValue)
  {
    Object[] hashKeys = _hashKeys;
    Object[] hashValues = _hashValues;
    Object value;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (hashKeys[hash] != key)
    {
      // While we are in a collision
      while (hashValues[hash] != null)
      {
        // Compute next hash
        hash = (hash + _hashSqrt) % _hashCapacity;

        // Check if we have found, what we were looking for
        if (hashKeys[hash] == key)
        {
          value = hashValues[hash];
          hashValues[hash] = newValue;
          return value;
        }
      }
      hashKeys[hash] = key;
      hashValues[hash] = newValue;
      return null;
    }
    value = hashValues[hash];
    hashValues[hash] = newValue;
    return value;
  }

  /**
   * Internal method to remove an index from the fast lookup hash data structure.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  public final Object remove(Object key)
  {
    Object[] hashKeys = _hashKeys;
    Object[] hashValues = _hashValues;
    Object value;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (hashKeys[hash] != key)
    {
      // While we are in a collision
      while (hashValues[hash] != null)
      {
        // Compute next hash
        hash = (hash + _hashSqrt) % _hashCapacity;

        // Check if we have found, what we were looking for
        if (hashKeys[hash] == key)
        {
          value = hashValues[hash];
          hashKeys[hash] = null;
          hashValues[hash] = null;
          internalRehashHash(hash, hashKeys, hashValues);
          return value;
        }
      }
      return null;
    }
    value = hashValues[hash];
    hashKeys[hash] = null;
    hashValues[hash] = null;
    internalRehashHash(hash, hashKeys, hashValues);
    return value;
  }

  /**
   * Internal method to get an index from the fast lookup hash data structure.
   * 
   * Note: This method returns null, if the key was not found, otherwise the old index.
   * 
   * Note: The thread must own the master synchronization monitor,
   * because this method does no synchronization.  
   */
  public final Object get(Object key)
  {
    Object[] hashKeys = _hashKeys;
    Object[] hashValues = _hashValues;
    int hash = internalComputeHash(key);

    // Check the most likely case first
    if (hashKeys[hash] != key)
    {
      // While we are in a collision
      while (hashValues[hash] != null)
      {
        // Compute next hash
        hash = (hash + _hashSqrt) % _hashCapacity;

        // Check if we have found, what we were looking for
        if (hashKeys[hash] == key)
        {
          return hashValues[hash];
        }
      }
      return null;
    }
    return hashValues[hash];
  }

  /**
   * @deprecated as of NW04.
   */
  public final boolean containsKey(Object obj)
  {
    return get(obj) != null;
  }

  /**
   * @deprecated as of NW04.
   */
  public final Set keySet()
  {
    Object[] hashKeys = _hashKeys;
    Object[] hashValues = _hashValues;
    Set set = new HashSet();
    for (int i = 0; i < hashValues.length; i++)
    {
      if (hashValues[i] != null)
      {
        set.add(hashKeys[i]);
      }
    }
    return set;
  }

  /**
   * @deprecated as of NW04.
   */
  public final void clear()
  {
    Object[] hashKeys = _hashKeys;
    Object[] hashValues = _hashValues;
    for (int i = 0; i < hashValues.length; i++)
    {
      hashKeys[i] = null;
      hashValues[i] = null;
    }
  }
}
