/*
 * Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 * Created on 09.03.2004
 */

package com.sapportals.wcm.util.cache.memory.optimized;

/**
 * Not multithreading-safe implementation (doesn't need to be) of a data structure
 * used to exchange cache entry data with the cache atomically.
 * 
 * Motivation: When accessing data of a cache entry through the cache, this data can
 * only be queried sequentially from the cache, but since the cache lacks any kind
 * of read consistency (due to performance reasons), you may end up in an inconsistent
 * view of the cache entry. Therefore you could use instances of this class (newly
 * created or reused - it's your responsibility to create and get rid of these instances
 * and not to work on them while the cache works on them) and hand them to the cache and
 * let the cache fill them up in a consistent and synchronized manner.
 */
public class LongKeyMemoryCacheEntry
{
  protected long key = 0;
  protected Object value = null;
  protected long modification = 0;
  protected long expiration = 0;
  protected int timetolive = 0;
  protected boolean refreshing = false;

  protected LongKeyMemoryCacheEntry()
  {
  }

  protected LongKeyMemoryCacheEntry(long key)
  {
    this.key = key;
  }

  public LongKeyMemoryCacheEntry(
    long key,
    Object value,
    long modification,
    long expiration,
    int timetolive,
    boolean refreshing)
  {
    this.key = key;
    this.value = value;
    this.modification = modification;
    this.expiration = expiration;
    this.timetolive = timetolive;
    this.refreshing = refreshing;
  }

  public long getKey()
  {
    return this.key;
  }

  public Object getValue()
  {
    return this.value;
  }

  public long getModificationTime()
  {
    return this.modification;
  }

  public long getExpirationTime()
  {
    return this.expiration;
  }

  public long getTimeToLive()
  {
    return this.timetolive;
  }

  public boolean isExpired()
  {
    return this.expiration == 0 ? false : this.expiration < System.currentTimeMillis();
  }

  public boolean isRefreshing()
  {
    return this.refreshing;
  }
}
