/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache.memory;

import java.util.*;

import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.cache.memory.optimized.*;

public class MemoryLRUCacheIndexedByString implements ICache, ICacheStatistics, IExtendedCacheStatistics
{
  private final static com.sap.tc.logging.Location s_log =
    com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.cache.memory.MemoryLRUCacheIndexedByString.class);

  private final String id;
  private final StringKeyMemoryCache cache;
  private long maxEntrySize;
  private long defaultEntrySize;
  private int defaultTimetolive;
  private boolean defaultRefresh;

  public MemoryLRUCacheIndexedByString(String cacheID, Properties properties) throws CacheException
  {
    id = cacheID;
    maxEntrySize = ICache.DEFAULT_MAX_ENTRY_SIZE;
    defaultEntrySize = ICache.DEFAULT_AVG_ENTRY_SIZE;
    defaultTimetolive = ICache.DEFAULT_TIME_TO_LIVE;
    defaultRefresh = ICache.DEFAULT_AUTO_DELAY_EXPIRATION;
    int capacity = ICache.DEFAULT_CAPACITY;

    // Parse properties
    if (properties == null)
    {
      s_log.warningT("initCache(84)", "properties for cache " + this.id + " missing in WCM configuration");
    }
    else
    {
      try
      {
        capacity = Integer.parseInt(properties.getProperty(CacheFactory.CFG_CAPACITY_KEY));
      }
      catch (Exception e)
      {
        s_log.warningT(
          "initCache(91)",
          "property " + CacheFactory.CFG_CAPACITY_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try
      {
        maxEntrySize = Long.parseLong(properties.getProperty(CacheFactory.CFG_MAX_ENTRY_SIZE_KEY));
      }
      catch (Exception e)
      {
        s_log.warningT(
          "initCache(103)",
          "property "
            + CacheFactory.CFG_MAX_ENTRY_SIZE_KEY
            + " for cache "
            + this.id
            + " missing in WCM configuration");
      }
      try
      {
        defaultEntrySize = Long.parseLong(properties.getProperty(CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY));
      }
      catch (Exception e)
      {
        s_log.warningT(
          "initCache(109)",
          "property "
            + CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY
            + " for cache "
            + this.id
            + " missing in WCM configuration");
      }
      try
      {
        defaultTimetolive = Integer.parseInt(properties.getProperty(CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY));
      }
      catch (Exception e)
      {
        s_log.warningT(
          "initCache(115)",
          "property "
            + CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY
            + " for cache "
            + this.id
            + " missing in WCM configuration");
      }
      try
      {
        defaultRefresh =
          new Boolean(properties.get(CacheFactory.CFG_AUTO_DELAY_EXPIRATION_KEY).toString()).booleanValue();
      }
      catch (Exception e)
      {
        // ignore
      }
    }

    // Create cache
    cache = new StringKeyMemoryCache(capacity, defaultTimetolive, defaultRefresh);
  }

  public StringKeyMemoryCache getStringKeyMemoryCache()
  {
    return cache;
  }

  /**
   * @deprecated as of NW04.
   */
  public void initCache(Properties properties) throws CacheException
  {
    // Question: What was the reason for making this method public?
    // Answer: To lose time in repeated configuration reading and in synchronization or to cause race conditions!
    if (properties != null)
    {
      ;
    }
  }

  public String getID()
  {
    return id;
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntry(ICacheEntry entry) throws CacheException
  {
    if ((maxEntrySize == 0) || (entry.getSize() <= maxEntrySize))
    {
      cache.putEntry(
        entry.getKey(),
        entry.getObject(),
        entry.getExpirationTime(),
        (int)entry.getTimeToLive(),
        entry.isAutoDelaying());
    }
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntry(String key, Object object) throws CacheException
  {
    addEntry(key, object, defaultTimetolive, defaultEntrySize, defaultRefresh);
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntry(String key, Object object, int timeToLive) throws CacheException
  {
    addEntry(key, object, timeToLive, defaultEntrySize, defaultRefresh);
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntry(String key, Object object, int timeToLive, long size) throws CacheException
  {
    addEntry(key, object, timeToLive, size, defaultRefresh);
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive) throws CacheException
  {
    addEntry(key, object, timeToLive, defaultEntrySize, true);
  }

  /**
   * @deprecated as of NW04.
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive, long size) throws CacheException
  {
    addEntry(key, object, timeToLive, size, true);
  }

  /**
   * @deprecated as of NW04.
   */
  private void addEntry(String key, Object object, int timeToLive, long size, boolean autoDelaying)
    throws CacheException
  {
    if ((maxEntrySize == 0) || (size <= maxEntrySize))
    {
      cache.putEntry(
        key,
        object,
        (timeToLive == 0) ? 0 : System.currentTimeMillis() + (long)timeToLive * 1000,
        defaultTimetolive,
        autoDelaying);
    }
  }

  public void putEntry(String key, Object value)
  {
    cache.putEntry(key, value);
  }

  public void putEntry(String key, Object value, long expiration)
  {
    cache.putEntry(key, value, expiration);
  }

  public void putEntry(String key, Object value, int timetolive, boolean refreshing)
  {
    cache.putEntry(key, value, timetolive, refreshing);
  }

  public void putEntry(String key, Object value, long expiration, int timetolive, boolean refreshing)
    throws CacheException
  {
    cache.putEntry(key, value, expiration, timetolive, refreshing);
  }

  /**
   * @deprecated as of NW04.
   */
  public ICacheEntry getEntry(String key) throws CacheException
  {
    MemoryCacheEntryIndexedByString cacheEntry = new MemoryCacheEntryIndexedByString(key);
    getEntry(cacheEntry);
    return cacheEntry;
  }

  /**
   * @deprecated as of NW04.
   */
  public void getEntry(MemoryCacheEntryIndexedByString memoryCacheEntry) throws CacheException
  {
    cache.getEntry(memoryCacheEntry.getStringKeyMutableMemoryCacheEntry());
  }

  public Object getEntryValue(String key) throws CacheException
  {
    return cache.getEntryValue(key);
  }

  /**
   * @deprecated as of NW04.
   */
  public boolean removeEntry(ICacheEntry entry) throws CacheException
  {
    return cache.removeEntry(entry.getKey());
  }

  public boolean removeEntry(String key) throws CacheException
  {
    return cache.removeEntry(key);
  }

  /**
   * @deprecated as of NW04.
   */
  public boolean removeEntriesStartingWith(String prefix) throws CacheException
  {
    return cache.removeEntriesStartingWith(prefix);
  }

  /**
   * @deprecated as of NW04.
   */
  public boolean removeEntriesOlderThan(long timestamp) throws CacheException
  {
    return cache.removeEntriesOlderThan(timestamp);
  }

  /**
   * @deprecated as of NW04.
   */
  public boolean containsEntry(String key) throws CacheException
  {
    return cache.getEntryValue(key) != null;
  }

  private final class IterEnumeration implements java.util.Enumeration
  {
    private final Iterator iter;

    public IterEnumeration(Iterator iter)
    {
      this.iter = iter;
    }

    public boolean hasMoreElements()
    {
      return this.iter.hasNext();
    }

    public Object nextElement()
    {
      return this.iter.next();
    }
  }

  /**
   * @deprecated as of NW04.
   */
  public Enumeration keys() throws CacheException
  {
    synchronized (this)
    {
      return new IterEnumeration(keySet().iterator());
    }
  }

  /**
   * @deprecated as of NW04.
   */
  public Set keySet() throws CacheException
  {
    return cache.getKeys();
  }

  /**
   * @deprecated as of NW04.
   */
  public CacheEntryList elements() throws CacheException
  {
    CacheEntryList cacheEntryList = new CacheEntryList();
    for (Iterator iter = cache.getEntries().iterator(); iter.hasNext();)
    {
      StringKeyMutableMemoryCacheEntry entry = (StringKeyMutableMemoryCacheEntry)iter.next();
      cacheEntryList.add(new MemoryCacheEntryIndexedByString(entry));
    }
    return cacheEntryList;
  }

  /**
   * @deprecated as of NW04.
   */
  public void clearCache() throws CacheException
  {
    cache.clear();
  }

  /**
   * @deprecated as of NW04.
   */
  public void refresh() throws CacheException
  {
    cache.refresh();
  }

  public int getCapacity()
  {
    return cache.getCapacity();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getMaxEntrySize()
  {
    return maxEntrySize;
  }

  /**
   * @deprecated as of NW04.
   */
  public long getEntryCount()
  {
    return cache.getActEntryCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getMaximumEntryCount()
  {
    return cache.getMaxEntryCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getAddCount()
  {
    return cache.getPutCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getInsertCount()
  {
    return cache.getPutNewCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getRemoveCount()
  {
    return cache.getRemoveCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getGetCount()
  {
    return cache.getGetCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getHitCount()
  {
    return cache.getGetHitCount();
  }

  /**
   * @deprecated as of NW04.
   */
  public long getSize()
  {
    return cache.getActEntryCount() * defaultEntrySize;
  }

  /**
   * @deprecated as of NW04.
   */
  public boolean isSizeFullyDetermined()
  {
    return false;
  }

  /**
   * @deprecated as of NW04.
   */
  public void resetCounters()
  {
    cache.resetStatistic();
  }

  public void getStatistic(CacheStatistic statistic)
  {
    cache.getStatistic(statistic);
  }

  public static void mainFunction() throws CacheException
  {
    Properties properties = new Properties();
    properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(3));
    MemoryLRUCacheIndexedByString cache = new MemoryLRUCacheIndexedByString("Function", properties);

    cache.putEntry("1", "1", 0, 0, false);
    cache.putEntry("2", "2", 0, 0, false);
    cache.removeEntry("3");
    cache.removeEntry("2");
    cache.putEntry("3", "3", 0, 0, false);
    cache.putEntry("4", "4", 0, 0, false);

    System.out.println(cache.getEntryValue("12"));
    System.out.println(cache.getEntryValue("7"));
    System.out.println(cache.getEntryValue("4"));
    System.out.println(cache.getEntryValue("3"));
    System.out.println(cache.getEntryValue("2"));
    System.out.println(cache.getEntryValue("1"));

    cache.putEntry("4", "4", 0, 0, false);
    cache.putEntry("5", "5", 0, 0, false);

    System.out.println(cache.getEntryValue("5"));
    System.out.println(cache.getEntryValue("4"));
    System.out.println(cache.getEntryValue("2"));
    System.out.println(cache.getEntryValue("3"));
    System.out.println(cache.getEntryValue("1"));

    CacheStatistic statistic = new CacheStatistic();
    cache.getStatistic(statistic);
    System.out.println(statistic.getStatisticAsString());
    System.out.println(statistic.getRecommendationsAsString());
  }

  public static void mainPerformance() throws CacheException
  {
    long startTime;
    long durationTime;
    long startMem1;
    long startMem2;
    long usedMem1;
    long usedMem2;

    try
    {
      // Create cache and raw data
      System.gc();
      Thread.yield();
      System.gc();
      startMem1 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      Properties properties = new Properties();
      properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
      MemoryLRUCacheIndexedByString cache = new MemoryLRUCacheIndexedByString("Performance", properties);
      startMem2 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      Object[] objs = new Object[CacheStatistic.SIZE];
      for (int i = 0; i < objs.length; i++)
      {
        objs[i] = new Integer(i).toString();
      }

      startTime = new Date().getTime();
      for (int j = 0; j < objs.length; j++)
      {
        cache.putEntry((String)objs[j], objs[j], 0, 0, false);
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          cache.putEntry((String)objs[j], objs[j], 0, 0, false);
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for re-adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntryValue((String)objs[j]) != objs[j])
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering existent cache entries: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntryValue(objs[j] + "x") != null)
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering non-existent cache entries: " + durationTime);

      // Dump used memory
      usedMem1 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);
      startTime = new Date().getTime();
      System.gc();
      Thread.yield();
      System.gc();
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for GC: " + durationTime);
      usedMem1 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 = (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);

      CacheStatistic statistic = new CacheStatistic();
      cache.getStatistic(statistic);
      System.out.println(statistic.getStatisticAsString());
      System.out.println(statistic.getRecommendationsAsString());

      Thread.sleep(10000);
    }
    catch (Throwable throwable)
    {
      System.err.println("Error!");
      throwable.printStackTrace();
    }
  }

  public static void mainThreading() throws CacheException
  {
    Properties properties = new Properties();
    properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
    final MemoryLRUCacheIndexedByString cache = new MemoryLRUCacheIndexedByString("Threading", properties);
    final Random random = new Random(System.currentTimeMillis());

    // Populate
    for (int i = 0; i < CacheStatistic.SIZE; i++)
    {
      // Add cache entry
      Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
      cache.putEntry((String)obj, obj, 0, 0, false);
    }

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Add cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.putEntry((String)obj, obj, 0, 0, false);
            // System.currentTimeMillis() + random.nextInt() % (60 * 1000));

            // Go to sleep
            Thread.sleep(10);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Remove cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.removeEntry((String)obj);

            // Go to sleep
            Thread.sleep(100);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.getEntry((String)obj);

            // Go to next
            Thread.yield();
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.getEntryValue((String)obj);

            // Go to next
            Thread.yield();
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            CacheStatistic statistic = new CacheStatistic();
            cache.getStatistic(statistic);
            System.out.println(statistic.getStatisticAsString());
            System.out.println(statistic.getRecommendationsAsString());

            // Go to sleep
            Thread.sleep(1000);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();
  }

  public static void mainLocking()
  {
    // Safeguard operations
    try
    {
      Properties properties = new Properties();
      properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
      final MemoryLRUCacheIndexedByString cache = new MemoryLRUCacheIndexedByString("Threading", properties);
      final Random random = new Random(System.currentTimeMillis());

      // Populate
      for (int i = 0; i < CacheStatistic.SIZE; i++)
      {
        // Add cache entry
        Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
        cache.putEntry((String)obj, obj, 0, 0, false);
      }

      // Prepare data
      Thread[] readers = new Thread[CacheStatistic.READERS];
      Thread[] writers = new Thread[CacheStatistic.WRITERS];
      cache.resetCounters();
      System.gc();
      Thread.yield();
      System.gc();

      // Create readers
      for (int i = 0; i < CacheStatistic.READERS; i++)
      {
        readers[i] = new Thread(new Runnable()
        {
          public void run()
          {
            long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
            while (System.currentTimeMillis() < exitTime)
            {
              // Safeguard operations
              try
              {
                // Get cache entry
                Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
                cache.getEntry((String)obj);
              }
              catch (Exception exception)
              {
                // Handle caught exception
                System.err.println("Error!");
                exception.printStackTrace();
              }
            }
          }
        });
      }

      // Create writers
      for (int i = 0; i < CacheStatistic.WRITERS; i++)
      {
        writers[i] = new Thread(new Runnable()
        {
          public void run()
          {
            long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
            while (System.currentTimeMillis() < exitTime)
            {
              // Safeguard operations
              try
              {
                // Put cache entry
                Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
                cache.addEntry((String)obj, obj);
              }
              catch (Exception exception)
              {
                // Handle caught exception
                System.err.println("Error!");
                exception.printStackTrace();
              }
            }
          }
        });
      }

      // Start readers
      for (int i = 0; i < CacheStatistic.READERS; i++)
      {
        readers[i].start();
      }

      // Start writers
      for (int i = 0; i < CacheStatistic.WRITERS; i++)
      {
        writers[i].start();
      }

      // Join readers
      try
      {
        Thread.sleep(CacheStatistic.REP);
        for (int i = 0; i < CacheStatistic.READERS; i++)
        {
          readers[i].join();
        }

        // Join writers
        for (int i = 0; i < CacheStatistic.WRITERS; i++)
        {
          writers[i].join();
        }
      }
      catch (Exception exception)
      {
        // Handle caught exception
        System.err.println("Error!");
        exception.printStackTrace();
      }

      // Output throughput
      System.out.println(
        "Locking Test Throughput: Gets: "
          + (cache.getGetCount() / 1000)
          + " K   Puts: "
          + (cache.getAddCount() / 1000)
          + " K");
    }
    catch (Exception exception)
    {
      // Handle caught exception
      System.err.println("Error!");
      exception.printStackTrace();
    }
  }

  public static void main(String[] args)
  {
    try
    {
      if (CacheStatistic.TEST_FUNCTION)
      {
        mainFunction();
      }
      if (CacheStatistic.TEST_PERFORMANCE)
      {
        mainPerformance();
      }
      if (CacheStatistic.TEST_THREADING)
      {
        mainThreading();
      }
      if (CacheStatistic.TEST_LOCKING)
      {
        mainLocking();
      }
    }
    catch (Exception exception)
    {
      exception.printStackTrace();
    }
  }
}
