/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache.memory;

import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.cache.memory.optimized.*;

import java.util.*;

/**
 * TBD: Description of the class.
 */
public class MemoryLRUCache implements ICache, ICacheStatistics, IExtendedCacheStatistics {

  private final static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.cache.memory.MemoryLRUCache.class);

  private final String id;

  private int capacity;
  private long maxCacheSize;
  private long maxEntrySize;
  private long averageEntrySize;
  private int defaultTimeToLive;
  private boolean autoDelayExpiration;

  private Map cache;
  private Map dependencies;
  private Map identities;
  private MemoryCacheEntry[] entries;
  private MemoryCacheEntry[] removals;
  private int[] next;
  private int[] prev;

  private int head;
  private int tail;
  private int free;

  private long size;// cache size (in bytes)
  private boolean sizeFullyDetermined;
  private int entryCount;

  private long maxEntryCount;
  private long addCount;
  private long insertCount;
  private long removeCount;
  private long getCount;
  private long hitCount;

  // -- construct --

  public MemoryLRUCache(String cacheID, Properties properties)
    throws CacheException {
    this.id = cacheID;
    this.initCache(properties);
  }

  /**
   * Initializes this cache instance from the properties.
   *
   * @param properties TBD: Description of the incoming method parameter
   * @exception CacheException Exception raised in failure situation
   */
  public void initCache(Properties properties)
    throws CacheException {

    int capacity = ICache.DEFAULT_CAPACITY;
    long maxCacheSize = ICache.DEFAULT_MAX_CACHE_SIZE;
    long maxEntrySize = ICache.DEFAULT_MAX_ENTRY_SIZE;
    long averageEntrySize = ICache.DEFAULT_AVG_ENTRY_SIZE;
    int defaultTimeToLive = ICache.DEFAULT_TIME_TO_LIVE;
    boolean autoDelayExpiration = ICache.DEFAULT_AUTO_DELAY_EXPIRATION;

    if (properties == null) {
      s_log.warningT("initCache(84)", "properties for cache " + this.id + " missing in WCM configuration");
    }
    else {
      try {
        capacity = Integer.parseInt(properties.getProperty(CacheFactory.CFG_CAPACITY_KEY));
      }
      catch (Exception e) {
        s_log.warningT("initCache(91)", "property " + CacheFactory.CFG_CAPACITY_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try {
        maxCacheSize = Long.parseLong(properties.getProperty(CacheFactory.CFG_MAX_CACHE_SIZE_KEY));
      }
      catch (Exception e) {
        s_log.warningT("initCache(97)", "property " + CacheFactory.CFG_MAX_CACHE_SIZE_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try {
        maxEntrySize = Long.parseLong(properties.getProperty(CacheFactory.CFG_MAX_ENTRY_SIZE_KEY));
      }
      catch (Exception e) {
        s_log.warningT("initCache(103)", "property " + CacheFactory.CFG_MAX_ENTRY_SIZE_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try {
        averageEntrySize = Long.parseLong(properties.getProperty(CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY));
      }
      catch (Exception e) {
        s_log.warningT("initCache(109)", "property " + CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try {
        defaultTimeToLive = Integer.parseInt(properties.getProperty(CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY));
      }
      catch (Exception e) {
        s_log.warningT("initCache(115)", "property " + CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY + " for cache " + this.id + " missing in WCM configuration");
      }
      try {
        autoDelayExpiration = new Boolean(properties.get(CacheFactory.CFG_AUTO_DELAY_EXPIRATION_KEY).toString()).booleanValue();
      }
      catch (Exception e) {
        s_log.debugT(e.getMessage());
      }
    }

    synchronized (this) {
      this.capacity = capacity;
      this.maxCacheSize = maxCacheSize;
      this.maxEntrySize = maxEntrySize;
      this.averageEntrySize = averageEntrySize;
      this.defaultTimeToLive = defaultTimeToLive;
      this.autoDelayExpiration = autoDelayExpiration;

      this.cache = new HashMap(this.capacity);
      this.dependencies = new HashMap(this.capacity);
      this.identities = new HashMap(this.capacity);
      this.entries = new MemoryCacheEntry[this.capacity];
      this.removals = new MemoryCacheEntry[this.capacity];

      this.next = new int[this.capacity];
      this.prev = new int[this.capacity];

      this.clearCache();
      this.resetCounters();
    }
  }


  // -- interfaces --

  public String getID() {
    return this.id;
  }

  public void addEntry(ICacheEntry entry)
    throws CacheException {
    localAddEntry(entry.getKey(), entry.getObject(), entry.getTimeToLive(), entry.getExpirationTime(), entry.getSize(), entry.isAutoDelaying());
  }

  public void addEntry(String key, Object object)
    throws CacheException {
    this.sizeFullyDetermined = false;
    addEntry(key, object, this.defaultTimeToLive, this.averageEntrySize);
  }

  public void addEntry(String key, Object object, int timeToLive)
    throws CacheException {
    this.sizeFullyDetermined = false;
    addEntry(key, object, timeToLive, this.averageEntrySize);
  }

  public void addEntryAutoDelay(String key, Object object, int timeToLive)
    throws CacheException {
    this.sizeFullyDetermined = false;
    addEntryAutoDelay(key, object, timeToLive, this.averageEntrySize);
  }

  public void addEntry(String key, Object object, int timeToLive, long size)
    throws CacheException {
    localAddEntry(key, object, timeToLive,
      (timeToLive == 0) ? 0 : System.currentTimeMillis() + (long)timeToLive * 1000, size, this.autoDelayExpiration);
  }

  public void addEntryAutoDelay(String key, Object object, int timeToLive, long size)
    throws CacheException {
    localAddEntry(key, object, timeToLive,
      (timeToLive == 0) ? 0 : System.currentTimeMillis() + (long)timeToLive * 1000, size, true);
  }

  public ICacheEntry getEntry(String key)
    throws CacheException {
    /* if (s_log.beDebug()) {
      s_log.debugT("getEntry(191)", "cache " + this.id + " get " + key);
    } */

    synchronized (this) {
      this.getCount++;

      Object cachedObject = this.cache.get(key);
      if (cachedObject == null) {
        return null;
      }
      MemoryCacheEntry entry = (MemoryCacheEntry)cachedObject;
      long now = System.currentTimeMillis();

      if (entry.isExpired(now)) {
        if (s_log.beDebug()) {
          s_log.debugT("getEntry(205)", "cache " + this.id + " initiale removal " + key + " (expired)");
        }
        removeEntry(entry);
        return null;
      }

      if (entry.isAutoDelaying()) {
        if (s_log.beDebug()) {
          s_log.debugT("getEntry(213)", "cache " + this.id + " delay " + key);
        }
        entry.delayExpiration(now);
      }

      moveToFront(entry.getIndex());

      this.hitCount++;

      return entry;
    }
  }

  public boolean removeEntry(ICacheEntry entry)
    throws CacheException {
    synchronized (this) {
      String key = entry.getKey();
      MemoryCacheEntry existing = (MemoryCacheEntry)this.cache.remove(key);
      if (existing != entry) {
        // entry was already removed
        this.cache.put(key, existing);
        return false;
      }

      localRemoveEntry(existing, false, true);
      return true;
    }
  }

  public boolean removeEntry(String key)
    throws CacheException {
    /* if (s_log.beDebug()) {
      s_log.debugT("removeEntry(245)", "cache " + this.id + " remove " + key);
    } */

    synchronized (this) {
      Object cachedObject = this.cache.get(key);
      if (cachedObject == null) {
        return false;
      }

      localRemoveEntry((MemoryCacheEntry)cachedObject, false, true);
      return true;
    }
  }

  public boolean removeEntriesStartingWith(String prefix)
    throws CacheException {
    /*
    boolean result = false;

    synchronized (this) {
      List entriesToRemove = null;
      for (int i = this.head; i >= 0; i = this.next[i]) {
        MemoryCacheEntry entry = this.entries[i];
        if (entry != null) {
          String key = entry.getKey();
          if (key != null && key.startsWith(prefix)) {
            if (entriesToRemove == null) {
              entriesToRemove = new ArrayList(50);
            }
            entriesToRemove.add(entry);
            result = true;
          }
        }
      }

      if (entriesToRemove != null) {
        final boolean debug = s_log.beDebug();
        for (int i = 0, n = entriesToRemove.size(); i < n; ++i) {
          MemoryCacheEntry entry = (MemoryCacheEntry)entriesToRemove.get(i);
          // if (debug) {
          //  s_log.debugT("removeEntriesStartingWith(284)", "cache " + this.id + " initiale removal " + entry.getKey() + " (expired)");
          // }
          localRemoveEntry(entry, true, true);
        }
      }
    }

    return result;
    */
    synchronized( this ) {
      int r = 0;
      int i = this.head;
      while( i >= 0 ) {
        if(   ( this.entries[i] != null )
           && ( this.entries[i].getKey() != null )
           && ( this.entries[i].getKey().startsWith(prefix) )
          ) {
          this.removals[r] = this.entries[i];
          r += 1;
        }
        i = this.next[i];
      }
      if( r == 0 ) {
        return false; // nothing found
      }
      while( r > 0 ) {
        r -= 1;
        localRemoveEntry(this.removals[r], true, true);
        this.removals[r] = null;
      }
      return true;
    }
  }

  public boolean removeEntriesOlderThan(long timestamp)
                                 throws CacheException {

    synchronized( this ) {
      int r = 0;
      int i = this.tail;
      while( i >= 0 ) {
        if( this.entries[i] != null ) {
          if( this.entries[i].getModificationTime() > timestamp ) {
            i = -1; // done
          } else {
            this.removals[r] = this.entries[i];
            r += 1;
            i = this.prev[i];
          }
        } else {
          i = this.prev[i];
        }
      }
      if( r == 0 ) {
        return false; // nothing found
      }
      while( r > 0 ) {
        r -= 1;
        localRemoveEntry(this.removals[r], true, true);
        this.removals[r] = null;
      }
      return true;
    }

  }

  public boolean containsEntry(String key)
    throws CacheException {
    synchronized (this) {
      return this.cache.containsKey(key);
    }
  }

  // used only for deprecated method keys()
  //
  /**
   * TBD: Description of the class.
   */
  private final class IterEnumeration implements java.util.Enumeration {

    private final Iterator iter;

    public IterEnumeration(Iterator iter) {
      this.iter = iter;
    }

    public boolean hasMoreElements() {
      return this.iter.hasNext();
    }

    public Object nextElement() {
      return this.iter.next();
    }
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @exception CacheException Exception raised in failure situation
   * @deprecated as of NW04.
   */
  public Enumeration keys()
    throws CacheException {
    synchronized (this) {
      return new IterEnumeration(this.cache.keySet().iterator());
    }
  }

  public Set keySet() {
    synchronized (this) {
      return this.cache.keySet();
    }
  }

  public CacheEntryList elements()
    throws CacheException {
    synchronized (this) {
      CacheEntryList list = new CacheEntryList();
      for (int i = this.head; i >= 0; i = this.next[i]) {
        list.add((ICacheEntry)this.entries[i]);
      }
      return list;
    }
  }

  public void clearCache()
    throws CacheException {
    if (s_log.beDebug()) {
      s_log.debugT("clearCache(355)", "cache " + this.id + " clear");
    }

    synchronized (this) {
      this.cache.clear();

      this.size = 0;
      this.entryCount = 0;
      this.sizeFullyDetermined = true;

      this.head = -1;
      this.tail = -1;

      // init free list
      //
      this.free = 0;
      for (int i = 0; i < this.capacity; ++i) {
        this.next[i] = i + 1;
      }
      this.next[this.capacity - 1] = -1;
    }
  }

  public void refresh()
    throws CacheException {
    /*
    long now = System.currentTimeMillis();
    List entriesToRemove = null;

    synchronized (this) {
      for (int i = this.head; i >= 0; i = this.next[i]) {
        MemoryCacheEntry entry = this.entries[i];
        if (entry != null && entry.getKey() != null && entry.isExpired(now)) {
          if (entriesToRemove == null) {
            entriesToRemove = new ArrayList(50);
          }
          entriesToRemove.add(entry);
        }
      }
    }

    if (entriesToRemove != null) {
      final boolean debug = s_log.beDebug();
      for (int i = 0, n = entriesToRemove.size(); i < n; ++i) {
        MemoryCacheEntry entry = (MemoryCacheEntry)entriesToRemove.get(i);
        // if (debug) {
        //  s_log.debugT("refresh(400)", "cache " + this.id + " initiale removal " + entry.getKey() + " (expired)");
        // }
        localRemoveEntry(entry, true, false);
      }
    }
    */
    long now = System.currentTimeMillis();
    synchronized( this ) {
      int r = 0;
      int i = this.head;
      while( i >= 0 ) {
        if(   ( this.entries[i] != null )
           && ( this.entries[i].getKey() != null )
           && ( this.entries[i].isExpired(now) )
          ) {
          this.removals[r] = this.entries[i];
          r += 1;
               }
        i = this.next[i];
      }
      if( r == 0 ) {
        return; // nothing found
      }
      while( r > 0 ) {
        r -= 1;
        localRemoveEntry(this.removals[r], true, false);
        this.removals[r] = null;
      }
    }
  }

  public int getCapacity() {
    return this.capacity;
  }

  public long getMaxEntrySize() {
    return this.maxEntrySize;
  }

  public long getEntryCount() {
    return this.entryCount;
  }

  public long getMaximumEntryCount() {
    return this.maxEntryCount;
  }

  public long getAddCount() {
    return this.addCount;
  }

  public long getInsertCount() {
    return this.insertCount;
  }

  public long getRemoveCount() {
    return this.removeCount;
  }

  public long getGetCount() {
    return this.getCount;
  }

  public long getHitCount() {
    return this.hitCount;
  }

  public long getSize() {
    return this.size;
  }

  public boolean isSizeFullyDetermined() {
    return this.sizeFullyDetermined;
  }

  public void resetCounters() {
    this.maxEntryCount = 0;
    this.addCount = 0;
    this.insertCount = 0;
    this.removeCount = 0;
    this.getCount = 0;
    this.hitCount = 0;
  }

  // ------------------------- private ------------------------------

  /**
   * Removes entry from the list, puts it on the free list and returns the index
   * of the next entry. If removeKey is true, the key is also removed from the
   * internal hashmap
   *
   * @param entry to remove
   * @param check TBD: Description of the incoming method parameter
   * @param invalid TBD: Description of the incoming method parameter
   * @exception CacheException Exception raised in failure situation
   */
  private synchronized void localRemoveEntry(MemoryCacheEntry entry, boolean check, boolean invalid)
    throws CacheException {
    String key = entry.getKey();
    if (check) {
      MemoryCacheEntry existing = (MemoryCacheEntry)this.cache.remove(key);
      if (existing != entry) {
        // entry was already removed
        this.cache.put(key, existing);
        return;
      }
    }
    else {
      this.cache.remove(key);
    }

    List keysToRemove = removeDependencies(entry, invalid);

    this.size -= entry.getSize();
    entry.setObject(null);
    entry.setKey(null);

    remove(entry.getIndex());

    if (keysToRemove != null) {
      for (int i = 0, n = keysToRemove.size(); i < n; ++i) {
        String skey = (String)keysToRemove.get(i);
        MemoryCacheEntry sentry = (MemoryCacheEntry)this.cache.get(skey);
        if (sentry != null) {
          localRemoveEntry(sentry, false, true);
        }
      }
    }
  }

  private void localAddEntry(String key, Object object, long timeToLive,
    long expirationTime, long size, boolean autoDelay)
    throws CacheException {
    if (key == null) {
      throw new NullPointerException();
    }

    // too large?
    if (this.maxEntrySize > 0 && size > this.maxEntrySize) {
      return;
    }

    synchronized (this) {
      MemoryCacheEntry entry = (MemoryCacheEntry)this.cache.get(key);
      if (entry != null) {
        /* if (s_log.beDebug()) {
          s_log.debugT("localAddEntry(517)", "cache " + this.id + " add " + key + " (exists)");
        } */

        // overwrite
        removeDependencies(entry, false);

        this.size -= entry.getSize();

        entry.setObject(object);
        entry.setTimeToLive(timeToLive);
        entry.setExpirationTime(expirationTime);
        entry.updateModificationTime();
        entry.setSize(size);
        entry.setAutoDelay(autoDelay);

        moveToFront(entry.getIndex());
      }
      else {
        if (this.free < 0) {
          if (s_log.beDebug()) {
            s_log.debugT("localAddEntry(537)", "cache " + this.id + " add " + key + " replacing " + this.entries[this.tail].getKey());
          }

          // remove oldest
          localRemoveEntry(this.entries[this.tail], false, false);
        }

        if (this.free >= 0) {
          /* if (s_log.beDebug()) {
            s_log.debugT("localAddEntry(546)", "cache " + this.id + " add " + key + " (new)");
          } */

          // get from free list and insert at head
          int index = this.free;
          this.free = this.next[this.free];

          insertAtHead(index);

          if (this.entries[this.head] == null) {
            this.entries[this.head] = new MemoryCacheEntry(this.head);
          }
        }
        else {
          s_log.warningT("localAddEntry(560)", "shoult not happen, free list empty after removal of one element");
          return;
        }

        entry = this.entries[this.head];
        entry.setKey(key);
        entry.setObject(object);
        entry.setTimeToLive(timeToLive);
        entry.setExpirationTime(expirationTime);
        entry.updateModificationTime();
        entry.setSize(size);
        entry.setAutoDelay(autoDelay);

        this.cache.put(key, entry);
        this.insertCount++;
      }

      addDependencies(entry);
      this.size += size;

      // drop entries to stay under max size
      while (this.maxCacheSize > 0 && this.size > this.maxCacheSize) {
        if (s_log.beDebug()) {
          s_log.debugT("localAddEntry(582)", "cache " + this.id + " initiale removal " + this.entries[this.tail].getKey() + " (cache full)");
        }
        localRemoveEntry(this.entries[this.tail], false, false);
      }

      this.addCount++;
    }
  }

  private void insertAtHead(int index) {
    if (this.head < 0) {
      this.head = this.tail = index;
      this.next[index] = this.prev[index] = -1;
      this.entryCount = 1;
    }
    else {
      this.prev[this.head] = index;
      this.prev[index] = -1;
      this.next[index] = this.head;
      this.head = index;
      this.entryCount++;
    }
    if (this.entryCount > this.maxEntryCount) {
      this.maxEntryCount = this.entryCount;
    }
  }

  private List removeDependencies(MemoryCacheEntry entry, boolean invalid) {
    List keysToRemove = null;
    if (entry.keepsDependencies()) {
      // entry is handling dependencies on other entries.
      // If this entry is invalid, remove all depending entries.
      // Remove all dependency information.
      //
      String key = entry.getKey();
      List slaves = entry.getSlaveKeys();
      List slaveKeys = (List)this.dependencies.get(key);
      if (invalid && slaveKeys != null && !slaveKeys.isEmpty()) {
        // remove all depending entries.
        keysToRemove = new ArrayList(slaveKeys);

        slaveKeys.removeAll(slaves);
        if (slaveKeys.isEmpty()) {
          this.dependencies.remove(key);
        }
      }

      // Remove dependency information maintained by this entry from
      // the dependency map.
      //
      List masters = entry.getMasterKeys();
      for (int i = 0, n = masters.size(); i < n; ++i) {
        String mkey = (String)masters.get(i);
        List masterKeys = (List)this.dependencies.get(mkey);
        if (masterKeys != null) {
          masterKeys.remove(key);
          if (masterKeys.isEmpty()) {
            this.dependencies.remove(mkey);
          }
        }
      }

      // Remove identity information and, iff invalid, remove
      // all entries with the same identity
      //
      String id = entry.getIdentity();
      if (id != null) {
        List idkeys = (List)this.identities.get(id);
        if (idkeys != null) {
          idkeys.remove(key);
          if (invalid) {
            if (keysToRemove == null) {
              keysToRemove = new ArrayList(idkeys);
            }
            else {
              keysToRemove.addAll(idkeys);
            }
            idkeys.clear();
          }
          if (idkeys.isEmpty()) {
            this.identities.remove(id);
          }
        }
      }
    }
    return keysToRemove;
  }

  private void addDependencies(MemoryCacheEntry entry) {
    if (entry.keepsDependencies()) {
      String key = entry.getKey();
      List slaves = entry.getSlaveKeys();
      if (!slaves.isEmpty()) {
        List mydep = (List)this.dependencies.get(key);
        if (mydep != null) {
          mydep.addAll(slaves);
        }
        else {
          this.dependencies.put(key, new ArrayList(slaves));
        }
      }

      // Remove dependency information maintained by this entry from
      // the dependency map.
      //
      List masters = entry.getMasterKeys();
      for (int i = 0, n = masters.size(); i < n; ++i) {
        String mkey = (String)masters.get(i);
        List masterKeys = (List)this.dependencies.get(mkey);
        if (masterKeys == null) {
          masterKeys = new ArrayList(5);
          this.dependencies.put(mkey, masterKeys);
        }
        masterKeys.add(key);
      }

      String id = entry.getIdentity();
      if (id != null) {
        List idkeys = (List)this.identities.get(id);
        if (idkeys == null) {
          idkeys = new ArrayList(2);
          this.identities.put(id, idkeys);
        }
        idkeys.add(key);
      }
    }
  }

  private void moveToFront(int index) {
    if (this.head != index) {
      int previndex = this.prev[index];
      int nextindex = this.next[index];
      if (previndex >= 0) {
        this.next[previndex] = nextindex;
        if (this.tail == index) {
          this.tail = previndex;
        }
      }
      if (nextindex >= 0) {
        this.prev[nextindex] = previndex;
      }

      if (this.head >= 0) {
        this.prev[this.head] = index;
      }

      this.prev[index] = -1;
      this.next[index] = this.head;
      this.head = index;
    }
  }

  /**
   * Removes ith entry from the list, puts it on the free list and returns the
   * index of the next entry.
   *
   * @param index TBD: Description of the incoming method parameter
   * @return index of next entry
   */
  private int remove(int index) {
    // remove from double-linked list
    final int previndex = this.prev[index];
    final int nextindex = this.next[index];
    if (previndex >= 0) {
      this.next[previndex] = nextindex;
    }
    if (nextindex >= 0) {
      this.prev[nextindex] = previndex;
    }

    // update head and tail
    if (this.head == index) {
      this.head = nextindex;
    }
    if (this.tail == index) {
      this.tail = previndex;
    }

    // add index to free list
    this.next[index] = this.free;
    this.free = index;

    this.entryCount--;
    this.removeCount++;

    return nextindex;
  }

  public static void mainFunction() throws CacheException
  {
    Properties properties = new Properties();
    properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(3));
    MemoryLRUCache cache = new MemoryLRUCache("Function", properties);

    cache.addEntry("1", "1");
    cache.addEntry("2", "2");
    cache.removeEntry("3");
    cache.removeEntry("2");
    cache.addEntry("3", "3");
    cache.addEntry("4", "4");

    System.out.println(cache.getEntry("12"));
    System.out.println(cache.getEntry("7"));
    System.out.println(cache.getEntry("4"));
    System.out.println(cache.getEntry("3"));
    System.out.println(cache.getEntry("2"));
    System.out.println(cache.getEntry("1"));

    cache.addEntry("4", "4");
    cache.addEntry("5", "5");

    System.out.println(cache.getEntry("5"));
    System.out.println(cache.getEntry("4"));
    System.out.println(cache.getEntry("2"));
    System.out.println(cache.getEntry("3"));
    System.out.println(cache.getEntry("1"));
  }

  public static void mainPerformance() throws CacheException
  {
    long startTime;
    long durationTime;
    long startMem1;
    long startMem2;
    long usedMem1;
    long usedMem2;

    try
    {
      // Create cache and raw data
      System.gc();
      Thread.yield();
      System.gc();
      startMem1 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      Properties properties = new Properties();
      properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
      MemoryLRUCache cache = new MemoryLRUCache("Performance", properties);
      startMem2 = Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory();
      Object[] objs = new Object[CacheStatistic.SIZE];
      for (int i = 0; i < objs.length; i++)
      {
        objs[i] = new Integer(i).toString();
      }

      startTime = new Date().getTime();
      for (int j = 0; j < objs.length; j++)
      {
        cache.addEntry((String)objs[j], objs[j]);
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          cache.addEntry((String)objs[j], objs[j]);
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for re-adding: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntry((String)objs[j]).getObject() != objs[j])
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering existent cache entries: " + durationTime);

      startTime = new Date().getTime();
      for (int i = 0; i < CacheStatistic.REP; i++)
      {
        for (int j = 0; j < objs.length; j++)
        {
          if (cache.getEntry(objs[j] + "x") != null)
          {
            throw new Exception("Error!");
          }
        }
      }
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for quering non-existent cache entries: " + durationTime);

      // Dump used memory
      usedMem1 =
        (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 =
        (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);
      startTime = new Date().getTime();
      System.gc();
      Thread.yield();
      System.gc();
      durationTime = new Date().getTime() - startTime;
      System.out.println("Time needed for GC: " + durationTime);
      usedMem1 =
        (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem1;
      System.out.println("Memory needed in sum (excluding data structure alloc): " + usedMem1);
      usedMem2 =
        (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) - startMem2;
      System.out.println("Memory needed in sum (including data structure alloc): " + usedMem2);

      Thread.sleep(10000);
    }
    catch (Throwable throwable)
    {
      System.err.println("Error!");
      throwable.printStackTrace();
    }
  }

  public static void mainThreading() throws CacheException
  {
    Properties properties = new Properties();
    properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
    final MemoryLRUCache cache = new MemoryLRUCache("Threading", properties);
    final Random random = new Random(System.currentTimeMillis());

    // Populate
    for (int i = 0; i < CacheStatistic.SIZE; i++)
    {
      // Add cache entry
      Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
      cache.addEntry((String)obj, obj);
    }

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Add cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.addEntry((String)obj, obj);
            // System.currentTimeMillis() + random.nextInt() % (60 * 1000));

            // Go to sleep
            Thread.sleep(10);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Remove cache entry
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.removeEntry((String)obj);

            // Go to sleep
            Thread.sleep(100);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.getEntry((String)obj);

            // Go to next
            Thread.yield();
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
            cache.getEntry((String)obj);

            // Go to next
            Thread.yield();
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();

    new Thread(new Runnable()
    {
      public void run()
      {
        while (true)
        {
          // Safeguard operations
          try
          {
            // Dump statistic
            System.out.println("Still living!");

            // Go to sleep
            Thread.sleep(1000);
          }
          catch (Exception exception)
          {
            // Handle caught exception
            System.err.println("Error!");
            exception.printStackTrace();
          }
        }
      }
    }).start();
  }

  public static void mainLocking()
  {
    // Safeguard operations
    try
    {
      Properties properties = new Properties();
      properties.put(CacheFactory.CFG_CAPACITY_KEY, Integer.toString(CacheStatistic.SIZE));
      final MemoryLRUCache cache = new MemoryLRUCache("Threading", properties);
      final Random random = new Random(System.currentTimeMillis());

      // Populate
      for (int i = 0; i < CacheStatistic.SIZE; i++)
      {
        // Add cache entry
        Object obj = Long.toString(random.nextLong() % (CacheStatistic.SIZE));
        cache.addEntry((String)obj, obj);
      }

      // Prepare data
      Thread[] readers = new Thread[CacheStatistic.READERS];
      Thread[] writers = new Thread[CacheStatistic.WRITERS];
      cache.resetCounters();
      System.gc();
      Thread.yield();
      System.gc();

      // Create readers
      for (int i = 0; i < CacheStatistic.READERS; i++)
      {
        readers[i] = new Thread(new Runnable()
        {
          public void run()
          {
            long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
            while (System.currentTimeMillis() < exitTime)
            {
              // Safeguard operations
              try
              {
                // Get cache entry
                Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
                cache.getEntry((String)obj);
              }
              catch (Exception exception)
              {
                // Handle caught exception
                System.err.println("Error!");
                exception.printStackTrace();
              }
            }
          }
        });
      }

      // Create writers
      for (int i = 0; i < CacheStatistic.WRITERS; i++)
      {
        writers[i] = new Thread(new Runnable()
        {
          public void run()
          {
            long exitTime = System.currentTimeMillis() + CacheStatistic.REP;
            while (System.currentTimeMillis() < exitTime)
            {
              // Safeguard operations
              try
              {
                // Put cache entry
                Object obj = Long.toString(random.nextLong() % (10 * CacheStatistic.SIZE));
                cache.addEntry((String)obj, obj);
              }
              catch (Exception exception)
              {
                // Handle caught exception
                System.err.println("Error!");
                exception.printStackTrace();
              }
            }
          }
        });
      }

      // Start readers
      for (int i = 0; i < CacheStatistic.READERS; i++)
      {
        readers[i].start();
      }

      // Start writers
      for (int i = 0; i < CacheStatistic.WRITERS; i++)
      {
        writers[i].start();
      }

      // Join readers
      try
      {
        Thread.sleep(CacheStatistic.REP);
        for (int i = 0; i < CacheStatistic.READERS; i++)
        {
          readers[i].join();
        }

        // Join writers
        for (int i = 0; i < CacheStatistic.WRITERS; i++)
        {
          writers[i].join();
        }
      }
      catch (Exception exception)
      {
        // Handle caught exception
        System.err.println("Error!");
        exception.printStackTrace();
      }

      // Output throughput
      System.out.println(
        "Locking Test Throughput: Gets: "
          + (cache.getGetCount() / 1000)
          + " K   Puts: "
          + (cache.getAddCount() / 1000)
          + " K");
    }
    catch (Exception exception)
    {
      // Handle caught exception
      System.err.println("Error!");
      exception.printStackTrace();
    }
  }

  public static void main(String[] args)
  {
    try
    {
      if (CacheStatistic.TEST_FUNCTION)
      {
        mainFunction();
      }
      if (CacheStatistic.TEST_PERFORMANCE)
      {
        mainPerformance();
      }
      if (CacheStatistic.TEST_THREADING)
      {
        mainThreading();
      }
      if (CacheStatistic.TEST_LOCKING)
      {
        mainLocking();
      }
    }
    catch (Exception exception)
    {
      exception.printStackTrace();
    }
  }
}
