/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache.memory;

import com.sapportals.wcm.util.cache.*;
import java.util.Collections;

import java.util.List;

/**
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Dirk Sabiwalsky
 * @author stefan.eissing@greenbytes.de
 * @version $Id: MemoryCacheEntry.java,v 1.3 2002/09/24 10:48:48 sei Exp $
 */
public final class MemoryCacheEntry implements ICacheEntry {

  private final int index;

  private String key = null;
  private Object object = null;
  private long size = 0;
  private long timeToLive = 0;// 0: never expires
  private long expirationTime = 0;// 0: never expires
  private long modificationTime;
  private boolean autoDelay = false;
  private boolean keepsDependencies = false;

  protected MemoryCacheEntry(int index) {
    this.index = index;
    updateModificationTime();
  }

  // -- interface ICacheEntry --

  public String getKey() {
    return this.key;
  }

  public Object getObject() {
    return this.object;
  }

  public long getSize() {
    return this.size;
  }

  public long getTimeToLive() {
    return this.timeToLive;
  }

  public long getExpirationTime() {
    return this.expirationTime;
  }

  public long getModificationTime() {
    return this.modificationTime;
  }

  public boolean isExpired() {
    return isExpired(System.currentTimeMillis());
  }

  public boolean isAutoDelaying() {
    return this.autoDelay;
  }

  // -- package --

  protected void setKey(String key) {
    this.key = key;
  }

  protected void setObject(Object object) {
    this.object = object;
    this.keepsDependencies = (this.object instanceof ICacheNode);
  }

  protected void setSize(long size) {
    this.size = size;
  }

  protected void setTimeToLive(long timeToLive) {
    this.timeToLive = timeToLive;
  }

  protected void setExpirationTime(long expirationTime) {
    this.expirationTime = expirationTime;
  }

  protected void updateModificationTime() {
    this.modificationTime = System.currentTimeMillis();
  }

  protected int getIndex() {
    return this.index;
  }

  protected void setAutoDelay(boolean newState) {
    this.autoDelay = newState;
  }

  protected boolean delayExpiration() {
    return delayExpiration(System.currentTimeMillis());
  }

  protected boolean delayExpiration(long now) {
    if (this.expirationTime == 0) {
      return false;
    }
    this.expirationTime = now + this.timeToLive * 1000;
    return true;
  }

  protected boolean isExpired(long now) {
    return this.expirationTime == 0 ? false : this.expirationTime < now;
  }

  protected boolean keepsDependencies() {
    return this.keepsDependencies;
  }

  protected List getMasterKeys() {
    if (this.keepsDependencies) {
      return ((ICacheNode)this.object).getMasterKeys();
    }
    return Collections.EMPTY_LIST;
  }

  protected List getSlaveKeys() {
    if (this.keepsDependencies) {
      return ((ICacheNode)this.object).getSlaveKeys();
    }
    return Collections.EMPTY_LIST;
  }

  protected String getIdentity() {
    if (this.keepsDependencies) {
      return ((ICacheNode)this.object).getIdentity();
    }
    return null;
  }
}
