/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Converter thread for wcm_app_acl_acl table. Due to the need of performance
 * enhancements the layout of the wcm_app_acl_acl table was improved. There a 10
 * new columns Seg1 to Seg10 wich store a hashed information of path segments of
 * an acl. This columns have to be populated at runtime. This is the purpose of
 * the class. <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author martin.boettcher@greenbytes.de
 * @version $Id: TableColumnPopulator.java,v 1.1 2003/02/03 10:52:39 mbo Exp $
 */
public class TableColumnPopulator extends Thread {
  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.acl.jdbc.TableColumnPopulator.class);
  private static boolean populatorRunning = false;
  private static TableColumnPopulator singleton = new TableColumnPopulator();
  List con = new ArrayList();

  private TableColumnPopulator(){} 
  /**
   * Only constructor.
   *
   * @return instance
   */
  public static TableColumnPopulator getInstance() {
    return singleton;
  }

  /**
   * Add a connection to the list of connections which have to be populated.
   *
   * @param con a connection to the database holding the table to be converted.
   *      Important: This connection is only for this thread at will be closed
   *      if the thread finishes.
   */
  public void addConnection ( Connection con ) {
    synchronized ( this.con ) {
      this.con.add ( con ) ;
    }
    this.start();
  }

  public void run() {
    synchronized ( singleton ) {
      if (populatorRunning) {
        log.infoT("Populator already running...");
        return;
      }
      populatorRunning = true;
    }
    boolean ok = true;
    do {
      if (!(ok = populate())) {
        try {
          Thread.sleep ( 2 * 60 * 1000 ) ;
        }
        catch (InterruptedException ignore) {
                      //$JL-EXC$
        }
      }
    } while(!ok);
  }

  private boolean populate() {
    synchronized ( this.con ) {
      for ( int i = 0 ; i < this.con.size() ; i++ ) {
        if ( !populate ((Connection)this.con.get ( i ))) {
          return false ;
        }
      }
    }
    return true ;
  }

  private boolean populate(Connection conn) {
    boolean ok = true;
    PreparedStatement st = null;
    ResultSet rs = null;
    try{
      List l = new ArrayList();
      log.infoT("collecting old acl information...");

      try {
        st = conn.prepareStatement ( "SELECT RID FROM "
          + JDBCDatabaseConnectionUncached.ACL_TBL
          + " WHERE Seg1 IS NULL");
        rs = st.executeQuery();
        while (rs.next()) {
          String s = rs.getString(1);
          if (( s != null ) && ( s.length() > 0 )) {
            l.add(s);
          }
        }
        rs.close() ;
        rs = null ;
        st.close() ;
        st = null ;
        conn.commit() ;
      }
      catch (SQLException e) {
        try {
          conn.rollback() ;
        }
        catch ( Exception ie ) {
            //$JL-EXC$          
        }
        log.errorT("Can't collect acl entries! - "
          + com.sapportals.wcm.util.logging.LoggingFormatter
          .extractCallstack(e));
        return false ;
      }
      if (l.size()>0) {
        log.infoT("starting conversion in background...");
        Iterator it = l.iterator();
        String s = "" ;
        try {
          st = conn.prepareStatement("UPDATE "
            + JDBCDatabaseConnectionUncached.ACL_TBL + " SET "
            + "Seg1=?, Seg2=?, Seg3=?, Seg4=?, Seg5=?, Seg6=?, Seg7=?, "
            + "Seg8=?, Seg9=?, Seg10=? WHERE ("
            + JDBCDatabaseConnectionUncached.ACL_HASH_COL
            + "=?) AND (RID=?) AND (Seg1 IS NULL)");
          while (it.hasNext()) {
            s = (String) it.next();
            Integer[] segs = JDBCDatabaseConnectionUncached.makeHashSegments(s);
            for (int i = 0; i < 10; i++) {
              if (segs[i] == null) {
                st.setNull ( i + 1, Types.INTEGER ) ;
              }
              else {
                st.setInt ( i + 1, segs[i].intValue()) ;
              }
            }
            st.setInt ( 11, s.hashCode()) ;
            st.setString ( 12, s ) ;
            st.executeUpdate() ;
            conn.commit() ;
            try {
              /*
              * take a short nap so that others can complete some work
              */
              Thread.sleep ( 10 * 1000 ) ;
            }
            catch ( InterruptedException ignore ) {
            }
          }
          st.close() ;
          st = null ;
        }
        catch (SQLException e) {
          try {
            conn.rollback() ;
          }
          catch ( Exception ie ) {}          
          log.warningT("Could not populate ACL entry: " + s + " - "
            + com.sapportals.wcm.util.logging.LoggingFormatter
            .extractCallstack(e));
          return false ;
        }
        log.infoT("conversion ended!");

        //Test whether still entries unconverted:
        try {
          st = conn.prepareStatement("SELECT COUNT(*) FROM "
            + JDBCDatabaseConnectionUncached.ACL_TBL
            + " WHERE Seg1 IS NULL");
          rs = st.executeQuery();
          if ( rs.next()) {
            ok = (rs.getInt(1)==0);
          }
          rs.close() ;
          rs = null ;
          conn.commit() ;
        }
        catch (SQLException e) {
          try {
            conn.rollback() ;
          }
          catch ( Exception ie ) {}
          log.warningT("Could not recount converted entries! - "
            + com.sapportals.wcm.util.logging.LoggingFormatter
            .extractCallstack(e));
          return false ;
        }
      }
    }
    finally {
      try {
        if (rs != null) {
          rs.close();
          rs = null ;
        }
      }
      catch (Exception ignore) {
      }
      try {
        if (st != null) {
          st.close();
          st = null ;
        }
      }
      catch (Exception ignore) {
      }
      try {
        conn.close();
      }
      catch (Exception ignore) {
      }
    }
    return ok;
  }
}
