/*
* Copyright (c) 2003 by SAP AG. All Rights Reserved.
*
* SAP, mySAP, mySAP.com and other SAP products and
* services mentioned herein as well as their respective
* logos are trademarks or registered trademarks of
* SAP AG in Germany and in several other countries all
* over the world. MarketSet and Enterprise Buyer are
* jointly owned trademarks of SAP AG and Commerce One.
* All other product and service names mentioned are
* trademarks of their respective companies.
*
* @version $Id$
*/

package com.sapportals.wcm.util.acl.jdbc ;

import com.sap.tc.logging.Location ;
import com.sapportals.portal.security.usermanagement.* ;
import com.sapportals.wcm.util.acl.* ;
import com.sapportals.wcm.WcmException ;
import com.sapportals.wcm.util.opensql.AutoID ;
import com.sapportals.wcm.util.cache.ICache ;

import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPool ;

import java.sql.* ;
import java.util.* ;

/*
* Uncached access to the ACL database persistence
*/
public final class OpenSQLDatabaseConnectionUncached
extends AbstractDatabaseConnectionUncached {

  /*
  * table names
  * NOTE: ACL_TBL is made availabe to the package
  * as the TableColumnPopulator needs it
  */
  protected final static String ACL_TBL = "KMC_ACL_ACL" ;
  private final static String ACL_MANAGER_TBL = "KMC_ACL_MANAGER" ;
  private final static String ACL_ENTRY_TBL = "KMC_ACL_ENTRY" ;
  private final static String ACL_OWNER_TBL = "KMC_ACL_OWNER" ;
  private final static String ACL_PERM_TBL = "KMC_ACL_PERM" ;
  private final static String ACL_SUP_PERM_TBL = "KMC_ACL_SUPP_PERM" ;
  private final static String ACL_PERM_MEM_TBL = "KMC_ACL_PERM_MEMB" ;
  private final static String ACL_RID_TBL = "KMC_ACL_RID" ;
  private final static String ACL_VERSION_TBL = "KMC_ACL_VERSION" ;

  private final static int MAX_IDS_PER_QUERY = 100 ;
  private final static int SEGCOLCNT = 7 ;
  private final static int SEGCOLLEN = 300 ;

  /*
  * needs to be static in order to ensure that we're not running
  * one cleanup thread per instance of this class; what we really
  * want is within the scope of a cluster, among all participating
  * java vms there should be just a single thread that runs this
  * table cleanup every once in a while
  *
  * one thread within a vm (or to be more specific, one thread per
  * class loader scope) is kind of achieved by locking on the class
  * and checking that no additional threads are kicked of.
  *
  * next, out of this set of single threads per vm spanning a cluster
  * only one survives the first wakeup; this is achieved by reading
  * the high value for the rid table from the autoid table; as each
  * vm instance "connects" to this autoid table and increases the
  * high value by one, we can derive that only a single vm is running
  * with this temporarily highest value; threads with lower values
  * just drop themselves dead on the floor after realizing their
  * misery.
  *
  * every once in while on of the java vms in a cluster will crash
  * or reboot because of NPEs or OOMs; the restarted vm will then
  * get the new highest id from the autoid table; this vm's cleanup
  * thread will be created and at least survive until its first
  * wakeup; meanwhile, there is one other thread sleeping somewhere
  * in the cluster; as soon as this spook wakes up it'll notice that
  * it's not the master of desaster any longer and therefore take it's
  * own turn towards doom.
  */
  private static CleanUpThread cleanUpThread = null ;

  private class CleanUpThread extends Thread {
    boolean active ;
    int highId = 0 ;
    String[] tableNames = { ACL_TBL, ACL_ENTRY_TBL, ACL_OWNER_TBL } ;
    final static int PARTITION_SIZE = 2048 ;
    int nPartitions = 0 ;
    boolean beDebug = false ;

    public CleanUpThread() {
      active = true ;
      setName ( "KM BC RF ACL OpenSQLDatabaseConnectionUncached Cleanup" ) ;
    }

    public void run() {
      Connection c = null ;
      while ( active ) {
        /*
        * delete from rid table where del marker > 1
        *
        * find min, max autoid in rid table and cluster
        * read entire cluster
        *
        * read distinct rid_id from various tables where rid_id in cluster range
        * remove found rid_ids from cluster;
        * 
        * increase del marker for remaining ids in rid table
        *
        * log message with number of deleted entries
        *
        * if entries == 0 then increase sleep time
        *
        * NOTE: only erase one cluster chunk per cycle!
        * 
        */
        try {
          long t0, t1, n, sleepTime = calculateSleepTime() ;
          sleep ( sleepTime ) ;
          /*
          * check for 'active' as we might have been forced to wake up
          * for shutting down; in that case 'active' is false and we
          * don't need to recalculate or check anything
          */
          beDebug = s_log.beDebug() ;
          if ( active ) {
            c = conPool.getConnection() ;
            c.setAutoCommit ( false ) ;
            if ( active = isMasterServer ( c )) {
              t0 = System.currentTimeMillis() ;
              nPartitions = balancePartitions ( c ) ;
              t1 = System.currentTimeMillis() ;
              if ( beDebug ) {
                s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                  "took " + (t1-t0) + "ms to rebalance into " + nPartitions
                  + " partitions" ) ;
              }

              /*
              * reset markers
              */
              t0 = System.currentTimeMillis() ;
              resetMarkers ( c ) ;
              t1 = System.currentTimeMillis() ;
              if ( beDebug ) {
                s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                  "took " + (t1-t0) + "ms to reset markers" ) ;
              }

              /*
              * pass 1 for all partitions
              */
              t0 = System.currentTimeMillis() ;
              markAclRids ( c ) ;
              t1 = System.currentTimeMillis() ;
              if ( beDebug ) {
                s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                  "took " + (t1-t0) + "ms to run 1st pass" ) ;
              }

              try {
                /*
                * sleep 2 minutes before running the second pass
                */
                sleep ( 2 * 60 * 1000 ) ;
              }
              catch ( InterruptedException ie ) {
                s_log.debugT(ie.getMessage());                
              }

              /*
              * pass 2
              */
              t0 = System.currentTimeMillis() ;
              markAclRids ( c ) ;
              t1 = System.currentTimeMillis() ;
              if ( beDebug ) {
                s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                  "took " + (t1-t0) + "ms to run 2nd pass" ) ;
              }

              /*
              * finally, drop unused rid table entries
              */
              t0 = System.currentTimeMillis() ;
              n = dropUnusedAclRids ( c ) ;
              t1 = System.currentTimeMillis() ;
              if ( beDebug ) {
                s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                  "took " + (t1-t0) + "ms to drop " + n + " unused acl rids" ) ;
              }
            }
          }
        }
        catch ( InterruptedException ie ) {
          s_log.debugT(ie.getMessage());
        }
        catch ( AclPersistenceException ae ) {
          active = false ;
          s_log.errorT ( "cleanup problem", ae.getMessage()) ;
        }
        catch ( Exception e ) {
          s_log.errorT ( "OpenSQLDatabaseConnectionUncached", e.getMessage()) ;
        }
        finally {
          if ( c != null ) {
            returnDBConnection ( c ) ;
            c = null ;
          }
        }
      }
    }

    public synchronized void setHighId ( long longHighId ) {
      int localHighId = (int)(longHighId >> 32L) ;
      if ( this.highId < localHighId ) {
        this.highId = localHighId ;
      }
    }
    
    private long calculateSleepTime() {
      /*
      * calculate a period of time so that the next run takes
      * place an 1:15 a.m. on next Sunday
      */
      Calendar now = Calendar.getInstance(), next = Calendar.getInstance() ;
      next.set ( Calendar.DAY_OF_WEEK, Calendar.SUNDAY ) ;
      next.set ( Calendar.HOUR_OF_DAY, 1 ) ;
      next.set ( Calendar.MINUTE, 15 ) ;
      /*
      * whatever random value 'next' might hold now we start moving back in
      * time until we end up somewhere in the past ...
      */
      while ( next.after ( now )) {
        next.add ( Calendar.HOUR, -7*24 ) ;
      }
      /*
      * .. from where we start moving towards the future until we pass the
      * present
      */
      while ( next.before( now )) {
        next.add ( Calendar.HOUR, 7*24 ) ;
      }
      return ( next.getTime().getTime() - now.getTime().getTime()) ;
    }

    private boolean isMasterServer ( Connection c )
    throws AclPersistenceException {
      PreparedStatement ps = null ;
      ResultSet rs = null ;
      try {
        ps = c.prepareStatement ( "SELECT RANGE FROM KMC_AUTOID WHERE NAME=?" );
        ps.setString ( 1, ACL_RID_TBL.trim().toLowerCase()) ;
        rs = ps.executeQuery() ;
        boolean found = false ;
        if ( !rs.next()) {
          return false ;
        }
        int dbHigh = rs.getInt ( 1 ) ;
        return ( dbHigh == highId ) ;
      }
      catch ( SQLException e ) {
        throw new AclPersistenceException ( e ) ;
      }
      finally {
        try {
          if ( rs != null ) {
            rs.close() ;
          }
          if ( ps != null ) {
            ps.close() ;
          }
        }
        catch ( SQLException e ) {
          throw new AclPersistenceException ( e ) ;
        }
      }
    }

    private long dropUnusedAclRids ( Connection c ) throws Exception {
      PreparedStatement ps = null ;
      long rv = 0 ;
      try {
        for ( int pid = 1 ; pid <= nPartitions ; pid++ ) {
          long n, t1, t0 = System.currentTimeMillis() ;
          ps = c.prepareStatement ( "DELETE FROM " + ACL_RID_TBL
            + " WHERE DEL_CNT>1 AND PID=?" ) ;
          ps.setInt ( 1, pid ) ;
          n = ps.executeUpdate() ;
          rv += n ;
          c.commit() ;
          ps.close() ;
          ps = null ;
          t1 = System.currentTimeMillis() ;
          if ( beDebug ) {
            s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
              "took " + (t1-t0) + "ms to drop " + n
              + " acl rids from partition " + pid ) ;
          }
        }
        return rv ;
      }
      finally {
        if ( ps != null ) {
          ps.close() ;
        }
      }
    }

    private void resetMarkers ( Connection c ) throws Exception {
      PreparedStatement ps = null ;
      try {
        for ( int pid = 1 ; pid <= nPartitions ; pid++ ) {
          long n, t1, t0 = System.currentTimeMillis() ;
          ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
            + " SET USE_CNT=0, DEL_CNT=0 WHERE PID=?" ) ;
          ps.setInt ( 1, pid ) ;
          n = ps.executeUpdate() ;
          c.commit() ;
          ps.close() ;
          ps = null ;
          t1 = System.currentTimeMillis() ;
          if ( beDebug ) {
            s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
              "took " + (t1-t0) + "ms to reset " + n
              + " markers in partition " + pid ) ;
          }
        }
      }
      finally {
        if ( ps != null ) {
          ps.close() ;
        }
      }
    }

    private void markAclRids ( Connection c ) throws Exception {
      PreparedStatement ps = null ;
      ResultSet rs = null ;
      long t0, t1, n, minId = 0, maxId = 0 ;
      try {
        for ( int pid = 1 ; pid <= nPartitions ; pid++ ) {
          ps = c.prepareStatement ( "SELECT MIN(AUTO_ID), MAX(AUTO_ID)"
            + " FROM " + ACL_RID_TBL + " WHERE PID=?" ) ;
          ps.setInt ( 1, pid ) ;
          rs = ps.executeQuery() ;
          if ( rs.next()) {
            minId = rs.getLong ( 1 ) ;
            maxId = rs.getLong ( 2 ) ;
          }
          rs.close() ;
          c.commit() ;
          ps.close() ;

          for ( int i = 0 ; i < tableNames.length ; i++ ) {
            t0 = System.currentTimeMillis() ;
            ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
              + " SET USE_CNT=USE_CNT+1 WHERE PID=? AND AUTO_ID"
              + " IN (SELECT DISTINCT RID_ID FROM " + tableNames[i]
              + " WHERE RID_ID BETWEEN ? AND ?)" ) ;
            ps.setInt ( 1, pid ) ;
            ps.setLong ( 2, minId ) ;
            ps.setLong ( 3, maxId ) ;
            n = ps.executeUpdate() ;
            c.commit() ;
            ps.close() ;
            t1 = System.currentTimeMillis() ;
            if ( beDebug ) {
              s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                "took " + (t1-t0) + "ms to mark " + n
                + " acl rids in partition " + pid + " and table "
                + tableNames[i]) ;
            }
          }

          t0 = System.currentTimeMillis() ;
          ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
            + " SET DEL_CNT=DEL_CNT+1 WHERE USE_CNT=0 AND PID=?" ) ;
          ps.setInt ( 1, pid ) ;
          n = ps.executeUpdate() ;
          c.commit() ;
          ps.close() ;
          t1 = System.currentTimeMillis() ;
          if ( beDebug ) {
            s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
              "took " + (t1-t0) + "ms to accumulate " + n
              + " acl rids markers in partition " + pid ) ;
          }
        }
      }
      finally {
        if ( rs != null ) {
          rs.close() ;
        }
        if ( ps != null ) {
          ps.close() ;
        }
      }
    }

    private int balancePartitions ( Connection c ) throws Exception {
      PreparedStatement ps = null ;
      ResultSet rs = null ;
      int cnt = 0 ;
      long t1, t0 = System.currentTimeMillis() ;
      try {
        ps = c.prepareStatement ( "SELECT COUNT(*) FROM " + ACL_RID_TBL ) ;
        rs = ps.executeQuery() ;
        if ( rs.next()) {
          cnt = rs.getInt ( 1 ) ;
        }
        rs.close() ;
        ps.close() ;
        if ( cnt < PARTITION_SIZE ) {
          /*
          * if less entries than would fit into one partition then merge
          * into partition number 1
          */
          t0 = System.currentTimeMillis() ;
          if ( cnt > 0 ) {
            ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL + " SET PID=1" ) ;
            ps.executeUpdate() ;
            ps.close() ;
          }
          c.commit() ;
          t1 = System.currentTimeMillis() ;
          if ( beDebug ) {
            s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
              "only " + cnt + " acl rids, no rebalancing; took " + (t1-t0)
              + "ms to move them into partition 1" ) ;
          }
          return 1 ;
        }
        int maxpid = ((cnt%PARTITION_SIZE)!=0)
          ?((cnt/PARTITION_SIZE)+1):(cnt/PARTITION_SIZE) ;
        /*
        * move all newbies into the highest partition
        */
        t0 = System.currentTimeMillis() ;
        ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
          + " SET PID=? WHERE PID=0" ) ;
        ps.setInt ( 1, maxpid ) ;
        int rv = ps.executeUpdate() ;
        ps.close() ;
        c.commit() ;
        t1 = System.currentTimeMillis() ;
        if ( beDebug ) {
          s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
            "took " + (t1-t0) + "ms to move " + rv + " new entries into"
            + " partition " + maxpid ) ;
        }
        int pid = 1 ;
        while ( pid < maxpid ) {
          long i, n_old, n = 0 ;
          t0 = System.currentTimeMillis() ;
          ps = c.prepareStatement ( "SELECT COUNT(*) FROM " + ACL_RID_TBL
            + " WHERE PID=?" ) ;
          ps.setInt ( 1, pid ) ;
          rs = ps.executeQuery() ;
          if ( rs.next()) {
            n = rs.getLong ( 1 ) ;
          }
          rs.close() ;
          ps.close() ;
          c.commit() ;
          t1 = System.currentTimeMillis() ;
          if ( beDebug ) {
            s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
              "took " + (t1-t0) + "ms to find " + n + " entries in"
              + " partition " + pid ) ;
          }
          /*
          * adjust all but the last partition which is achieved by the outer
          * loop stopping one before the maximum partition id
          */
          int passCnt = 1 ;
          while ( n != PARTITION_SIZE ) {
            i = 0 ;
            n_old = n ;
            t0 = System.currentTimeMillis() ;
            if ( n < PARTITION_SIZE ) {
              /*
              * select the lowest id of the next cluster and move l .. l+n into
              * the current cluster
              */
              ps = c.prepareStatement ( "SELECT MIN(AUTO_ID) FROM "
                + ACL_RID_TBL + " WHERE PID>?" ) ;
              ps.setInt ( 1, pid ) ;
              rs = ps.executeQuery() ;
              if ( rs.next()) {
                i = rs.getLong ( 1 ) ;
              }
              rs.close() ;
              ps.close() ;
              /*
              * 'i' cannot have gone down as this thread is the only way to
              * drop entries
              */
              ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
                + " SET PID=? WHERE PID>? AND AUTO_ID<?" ) ;
              ps.setInt ( 1, pid ) ;
              ps.setInt ( 2, pid ) ;
              ps.setLong ( 3, i + ( PARTITION_SIZE - n )) ;
              n += ps.executeUpdate() ;
            }
            else {
              /*
              * select the highest id of the cluster and move h-n .. h out
              * of the way
              */
              ps = c.prepareStatement ( "SELECT MAX(AUTO_ID) FROM "
                + ACL_RID_TBL + " WHERE PID=?" ) ;
              ps.setInt ( 1, pid ) ;
              rs = ps.executeQuery() ;
              if ( rs.next()) {
                i = rs.getLong ( 1 ) ;
              }
              rs.close() ;
              ps.close() ;
              ps = c.prepareStatement ( "UPDATE " + ACL_RID_TBL
                + " SET PID=? WHERE PID=? AND AUTO_ID>?" ) ;
              ps.setInt ( 1, pid + 1 ) ;
              ps.setInt ( 2, pid ) ;
              ps.setLong ( 3, i - ( n - PARTITION_SIZE )) ;
              n -= ps.executeUpdate() ;
            }
            ps.close() ;
            c.commit() ;
            t1 = System.currentTimeMillis() ;
            if ( beDebug ) {
              s_log.debugT ( "OpenSQLDatabaseConnectionUncached",
                "took " + (t1-t0) + "ms change partition " + pid + " to "
                + n + " entries, relocating " + (n-n_old)
                + " entries in pass #" + passCnt ) ;
            }
            passCnt++ ;
            /*
            * proceed to next partition only if the current partition has
            * been filled, or if no entries were relocated
            */
            if ( n == n_old ) {
              break ;
            }
          }
          if ( n == PARTITION_SIZE ) {
            pid++ ;
          }
        }
        return maxpid ;
      }
      catch ( Exception e ) {
        throw e ;
      }
    }
  }

  public void terminate() {
    boolean beDebug = s_log.beDebug() ;
    if ( beDebug ) {
      s_log.debugT( "OpenSQLDatabaseConnectionUncached.terminate entry" ) ;
    }
    synchronized ( OpenSQLDatabaseConnectionUncached.class ) {
      if ( cleanUpThread != null ) {
        if ( beDebug ) {
          s_log.debugT( "OpenSQLDatabaseConnectionUncached.terminate ",
            "tearing down cleanup thread" ) ;
        }
        cleanUpThread.active = false ;
        cleanUpThread.interrupt() ;
        try {
          cleanUpThread.join() ;
          if ( beDebug ) {
            s_log.debugT( "OpenSQLDatabaseConnectionUncached.terminate ",
              "cleanup thread joined" ) ;
          }
        }
        catch ( InterruptedException ie ) {
          s_log.debugT ( "OpenSQLDatabaseConnectionUncached.terminate ",
            ie.getMessage());
        }
        cleanUpThread = null ;
      }
      else if ( beDebug ){
        s_log.debugT ( "OpenSQLDatabaseConnectionUncached.terminate ",
          "cleanup thread already gone" ) ;
      }
      if ( sm != null ) {
        sm = null ;
        s_log.debugT ( "OpenSQLDatabaseConnectionUncached.terminate ",
          "segmentor reset to null" ) ;
      }
      if ( ridCache != null ) {
        ridCache = null ;
        if ( beDebug ) {
          s_log.debugT ( "OpenSQLDatabaseConnectionUncached.terminate ",
            "ridCache reset to null" ) ;
        }
      }
      else if ( beDebug ) {
        s_log.debugT ( "OpenSQLDatabaseConnectionUncached.terminate ",
          "ridCache already reset" ) ;
      }
    }
  }

  private class Segmentor {
    private final int MaxEntries = 401 * 32 ;
    private final int DropEntries = MaxEntries / 10 ;
    int[] segstartndx ;
    int[] segendndx ;
    private char[] cb = new char[1024<<2] ;
    private LinkedList trimMatchLRU = new LinkedList() ;
    private LinkedList untrimLRU = new LinkedList() ;
    private Map trimMatchCache = new HashMap() ;
    private Map untrimCache = new HashMap() ;
    final int SHORTENTRYLEN = ( SEGCOLCNT * SEGCOLLEN ) / 3 ;

    Segmentor() {
      segstartndx = new int[SEGCOLCNT] ;
      segendndx = new int[SEGCOLCNT] ;
      for ( int i = 0 ; i < segstartndx.length ; i++ ) {
        segstartndx[i] = i * SEGCOLLEN ;
        segendndx[i] = (( i + 1 ) * SEGCOLLEN ) ;
      }
    }

    /*
    * return the number of segments that s spans across
    */
    public int getSegCount ( String s ) {
      int sl = s.length() ;
      int rv = sl / SEGCOLLEN ;
      if (( sl % SEGCOLLEN ) > 0 ) {
        rv++ ;
      }
      return rv ;
    }

    public String getTrimMatchString ( String s ) {
      String rv = null ;
      if ( s == null ) {
        return rv ;
      }
      int l, cbi = 0, si = 0, ei, sl = s.length(), sli = sl - 1 ;
      boolean useCache = s.length() < SHORTENTRYLEN ;
      if ( useCache ) {
        synchronized ( trimMatchCache ) {
          rv = (String)trimMatchCache.get ( s ) ;
        }
        if ( rv != null ) {
          return rv ;
        }
      }
      for ( char ch = s.charAt ( sli ) ; ( ch == ' ' ) || ( ch == '\t' ) ;
        ch = s.charAt ( --sli )) ;
      sl = sli + 1 ;
      synchronized ( cb ) {
        while ( si < sl ) {
          l = Math.min ( SEGCOLLEN - 1, sl - si ) ;
          ei = si + l ;
          s.getChars ( si, ei, cb, cbi ) ;
          cbi += l ;
          cb[cbi++] = '.' ;
          si = ei ;
        }
        rv = new String ( cb, 0, cbi ) ;
      }
      if ( useCache ) {
        synchronized ( trimMatchCache ) {
          if ( trimMatchLRU.size() >= MaxEntries ) {
            for ( int i = DropEntries ; i > 0 ; i-- ) {
              trimMatchCache.remove ( trimMatchLRU.removeLast()) ;
            }
          }
          trimMatchCache.put ( s, rv ) ;
          trimMatchLRU.add ( s ) ;
        }
      }
      return rv ;
    }

    public String getUntrimString ( String s ) {
      String rv = null ;
      if ( s == null ) {
        return rv ;
      }
      boolean useCache = s.length() < SHORTENTRYLEN ;
      if ( useCache ) {
        synchronized ( untrimCache ) {
          rv = (String)untrimCache.get ( s ) ;
        }
        if ( rv != null ) {
          return rv ;
        }
      }
      int l = s.lastIndexOf ( '.' ) ;
      rv = s.substring ( 0, l ) ;
      if ( useCache ) {
        synchronized ( untrimCache ) {
          if ( untrimLRU.size() >= MaxEntries ) {
            for ( int i = DropEntries ; i > 0 ; i-- ) {
              untrimCache.remove ( untrimLRU.removeLast()) ;
            }
          }
          
          untrimCache.put ( s, rv ) ;
          untrimLRU.add ( s ) ;
        }
      }
      return rv ;
    }

    public String getTrimLikeString ( String s ) {
      int l, cbi = 0, si = 0, ei, sl = s.length(), sli = sl - 1 ;
      for ( char ch = s.charAt ( sli ) ; ( ch == ' ' ) || ( ch == '\t' ) ;
        ch = s.charAt ( --sli )) ;
      boolean endsWithSlash = s.charAt ( sli ) == '/' ;
      sl = sli + 1 ;
      synchronized ( cb ) {
        while ( si < sl ) {
          l = Math.min ( SEGCOLLEN - 1, sl - si ) ;
          ei = si + l ;
          s.getChars ( si, ei, cb, cbi ) ;
          cbi += l ;
          if (( ei < sl ) || ( l == ( SEGCOLLEN - 1 ))) {
            cb[cbi++] = '.' ;
          }
          si = ei ;
        }
        if ( !endsWithSlash ) {
          cb[cbi++] = '/' ;
        }
        cb[cbi++] = '%' ;
        return new String ( cb, 0, cbi ) ;
      }
    }

    public String getSegment ( String s, int sn ) {
      if (( sn < 0 ) || ( sn >= SEGCOLCNT )) {
        throw new ArrayIndexOutOfBoundsException (
          "segment index not between 0 and " + SEGCOLCNT ) ;
      }
      /*
      * fast path for the non-segmentation case, e.g. the
      * trimmed string s fits into the first segment
      */
      int sl = s.length() ;
      if (( sn == 0 ) && ( sl < segendndx[0] )) {
        return s ;
      }
      int si = segstartndx[sn] ;
      int ei = Math.min ( segendndx[sn], sl ) ;
      String rv = s.substring ( si, ei ) ;
      return rv ;
    }
  }

  private static Long emptyEntry = new Long ( RidCache.RIDCACHE_NULL_ENTRY ) ;
    
  private class RidCache {
    /*
    * bucket head object should be as large as a cache line of the
    * underlying machine architecture to avoid cache line contention
    * when taking locks on MP machines
    */
    /*
    * primes: 401, 1009, 2003 ;
    */
    final static int NBUCKETS = 401 ;
    final static int BUCKETSIZE = 32 ;
    final static int BDROPSIZE = BUCKETSIZE / 4 ;
    final static int RIDCACHE_NO_ENTRY = -2 ;
    final static int RIDCACHE_NULL_ENTRY = -1 ;
        
    class Bucket {
      LinkedList lru = null ;
      Map entries = null ;
      Bucket() {
        lru = new LinkedList() ;
        entries = new HashMap ( BUCKETSIZE ) ;
      }

      long get ( String rid ) {
        Long entry = (Long)entries.get ( rid ) ;
        if ( entry != null ) {
          lru.remove ( rid ) ;
          lru.addFirst ( rid ) ;
          return ( entry.longValue()) ;
        }
        return ( RIDCACHE_NO_ENTRY ) ;
      }

      void put ( long id, String rid ) {
        Long entry = (Long)entries.get ( rid ) ;
        if (( entry != null ) && ( entry.longValue() == id )) {
          lru.remove ( rid ) ;
          lru.addFirst ( rid ) ;
          return ;
        }
        if ( lru.size() >= BUCKETSIZE ) {
          for ( int i = BDROPSIZE ; i > 0 ; i-- ) {
            entries.remove ( lru.removeLast()) ;
          }
        }
        if ( id == RIDCACHE_NULL_ENTRY ) {
          entries.put ( rid, emptyEntry ) ;
        }
        else {
          entries.put ( rid, new Long ( id )) ;
        }       
        lru.addFirst ( rid ) ;
      }
      
      void flush() {
        lru = new LinkedList() ;
        entries = new HashMap ( BUCKETSIZE ) ;
      }
    }

    Bucket[] buckets = null ;

    RidCache() {
      buckets = new Bucket[NBUCKETS] ;
      for ( int i = 0 ; i < buckets.length ; i++ ) {
        buckets[i] = new Bucket() ;
      }
    }

    long get ( String rid ) {
      int i = ((i=rid.hashCode())<0)?(-i % NBUCKETS):(i % NBUCKETS) ;
      Bucket b = buckets[i] ;
      synchronized ( b ) {
        return b.get ( rid ) ;
      }
    }

    void put ( long id, String rid ) {
      int i = ((i=rid.hashCode())<0)?(-i % NBUCKETS):(i % NBUCKETS) ;
      Bucket b = buckets[i] ;
      synchronized ( b ) {
        b.put ( id, rid ) ;
      }
    }

    void flush() {
      for ( int i = 0 ; i < buckets.length ; i++ ) {
        Bucket b = buckets[i] ;
        synchronized ( b ) {
          b.flush() ;
        }
      }
    }
  }

  private final static String[] SN ;
  private final static String[] RIDMATCH ;
  private final static String[] RIDLIKE ;
  private final static String[] UNCOMMITTEDGETACLRID_PS ;
  private final static String[] UNCOMMITTEDADDORGETACLRID_PS ;
  private final static String[] UNCOMMITTEDGETACLRIDSLIKE_PS ;
  private final static String[] GETDESCENDANTSWITHACL_PS ;
  private final static String[] UNCOMMITTEDAREACLSUPTODATEINDB_PS ;
  private final static String[] UNCOMMITTEDGETACLSFROMDB_PS ;
  private final static String[] UNCOMMITTEDGETOWNERS_PS2 ;
  private final static String[] UNCOMMITTEDGETACLENTRIES_PS2 ;
  private final static String[] UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS ;
  private final static String[] UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2 ;
  private final static String[] UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3 ;
  private final static String OPENSQLDATABASECONNECTIONUNCACHED_PS1 ;
  private final static String OPENSQLDATABASECONNECTIONUNCACHED_PS2 ;
  private final static String HASDESCENDANTSWITHACL_PS ;
  private final static String UNCOMMITTEDGETACL_PS ;
  private final static String UNCOMMITTEDGETINTERNALACLIDFROMENTRYID_PS ;
  private final static String UNCOMMITTEDISACLUPTODATE_PS ;
  private final static String UNCOMMITTEDINCREASEACLCHANGELEVEL_PS ;
  private final static String UNCOMMITTEDSETACLENTRYSORTINDEX_PS ;
  private final static String UNCOMMITTEDCHECKACL_PS ;
  private final static String UNCOMMITTEDCHECKACLENTRIES_PS ;
  private final static String UNCOMMITTEDCHECKPERMISSION_PS ;
  private final static String UNCOMMITTEDGETACLENTRIES_PS ;
  private final static String UNCOMMITTEDADDACL_PS ;
  private final static String UNCOMMITTEDADDACLENTRY_PS ;
  private final static String UNCOMMITTEDREMOVEACLENTRY_PS ;
  private final static String UNCOMMITTEDISOWNER_PS ;
  private final static String UNCOMMITTEDISLASTOWNER_PS ;
  private final static String UNCOMMITTEDGETOWNERS_PS ;
  private final static String UNCOMMITTEDHASOWNERSINDB_PS ;
  private final static String UNCOMMITTEDADDOWNER_PS ;
  private final static String UNCOMMITTEDREMOVEOWNER_PS ;
  private final static String UNCOMMITTEDREMOVEOWNER_PS2 ;
  private final static String UNCOMMITTEDGETPERMISSIONMEMBERS_PS ;
  private final static String UNCOMMITTEDGETALLPERMISSIONSNOTFULLCONTROL_PS ;
  private final static String UNCOMMITTEDREMOVEOWNERS_PS ;
  private final static String UNCOMMITTEDREMOVEACLENTRIES_PS ;
  private final static String UNCOMMITTEDREMOVEACLENTRIES_PS2 ;
  private final static String UNCOMMITTEDCHECKINITIALPERMISSIONS_PS ;
  private final static String UNCOMMITTEDCHECKINITIALSUPPORTEDPERMS_PS ;
  private final static String UNCOMMITTEDISSUPPORTEDPERMISSION_PS ;
  private final static String UNCOMMITTEDGETSUPPORTEDPERMISSIONS_PS ;
  private final static String UNCOMMITTEDGETPERMISSION_PS ;
  private final static String UNCOMMITTEDISUSEDPERMISSION_PS ;
  private final static String UNCOMMITTEDADDPERMISSION_PS ;
  private final static String UNCOMMITTEDADDSUPPORTEDPERMISSION_PS ;
  private final static String UNCOMMITTEDREMOVESUPPORTEDPERMISSION_PS ;
  private final static String UNCOMMITTEDADDPERMISSIONMEMBER_PS ;
  private final static String UNCOMMITTEDREMOVEPERMISSIONMEMBER_PS ;
  private final static String UNCOMMITTEDREMOVEPERMISSIONMEMBERS_PS ;
  private final static String UNCOMMITTEDISPREDEFINEDPERMISSION_PS ;
  private final static String UNCOMMITTEDREMOVEACL_PS ;
  private final static String UNCOMMITTEDLOCKACL_PS ;
  private final static String UNCOMMITTEDUNLOCKACL_PS ;
  private final static String UNCOMMITTEDISACLLOCKED_PS ;
  private final static String UNCOMMITTEDGETACLLOCKINGUSER_PS ;
  private final static String UNCOMMITTEDCHANGEACLID_IN_ACLTABLE_PS ;
  private final static String UNCOMMITTEDCHANGEACLID_IN_ACLENTRYTABLE_PS ;
  private final static String UNCOMMITTEDCHANGEACLID_IN_OWNERTABLE_PS ;
  private final static String UPDATEVERSION_PS ;
  private final static String CHECKVERSION_PS1 ;
  private final static String CHECKVERSION_PS2 ;
  private final static String GETDBVERSION_PS ;
  private final static String UNCOMMITTEDSETACLENTRYPROPAGATION_PS ;
  private final static String UNCOMMITTEDREMOVEPERMISSION_PS ;
  private final static String UNCOMMITTEDGETAFFECTEDACLIDS_PS ;

  static {
    OPENSQLDATABASECONNECTIONUNCACHED_PS1 = "SELECT AUTO_ID FROM "
      + ACL_MANAGER_TBL + " WHERE APP=? AND MANAGER=?" ;
    OPENSQLDATABASECONNECTIONUNCACHED_PS2 = "INSERT INTO " + ACL_MANAGER_TBL
      + " (AUTO_ID, APP, MANAGER) VALUES (?,?,?)" ;
    HASDESCENDANTSWITHACL_PS = "SELECT COUNT(*) FROM " + ACL_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDGETACL_PS = "SELECT OBJECT_TYPE, CHANGE_LEVEL FROM "
      + ACL_TBL + " WHERE OWNER_ID=? AND RID_ID = ?" ;
    UNCOMMITTEDGETINTERNALACLIDFROMENTRYID_PS = "SELECT T2.RID FROM "
      + ACL_ENTRY_TBL + " AS T1, " + ACL_RID_TBL
      + " AS T2 WHERE T1.AUTO_ID = ? AND T1.RID_ID = T2.AUTO_ID" ;
    UNCOMMITTEDISACLUPTODATE_PS = "SELECT CHANGE_LEVEL FROM " + ACL_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDINCREASEACLCHANGELEVEL_PS = "UPDATE " + ACL_TBL
      + " SET CHANGE_LEVEL = CHANGE_LEVEL + 1"
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDSETACLENTRYSORTINDEX_PS = "UPDATE " + ACL_ENTRY_TBL
      + " SET SORT_INDEX = ? WHERE AUTO_ID = ?" ;
    UNCOMMITTEDCHECKACL_PS = "SELECT COUNT(*) FROM " + ACL_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDCHECKACLENTRIES_PS = "SELECT COUNT(*) FROM "
      + ACL_ENTRY_TBL + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDCHECKPERMISSION_PS = "SELECT COUNT(*) FROM " + ACL_PERM_TBL
      + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDGETACLENTRIES_PS = "SELECT AUTO_ID, PRINCIPAL_NAME,"
      + " PRINCIPAL_TYPE, SORT_INDEX, IS_NEGATIVE, IS_PROPAGATED,"
      + " PERMISSION_NAME FROM " + ACL_ENTRY_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ? ORDER BY SORT_INDEX" ;
    UNCOMMITTEDADDACL_PS = "INSERT INTO " + ACL_TBL + " ( AUTO_ID, OBJECT_TYPE,"
      + " CHANGE_LEVEL, OWNER_ID, RID_ID ) VALUES ( ?, ?, 0, ?, ? )" ;
    UNCOMMITTEDADDACLENTRY_PS = "INSERT INTO " + ACL_ENTRY_TBL
      + " ( AUTO_ID, PRINCIPAL_NAME, PRINCIPAL_TYPE, SORT_INDEX, IS_NEGATIVE,"
      + " IS_PROPAGATED, PERMISSION_NAME, OWNER_ID, RID_ID )"
      + " VALUES (?,?,?,?,?,?,?,?,?)" ;
    UNCOMMITTEDREMOVEACLENTRY_PS = "DELETE FROM " + ACL_ENTRY_TBL
      + " WHERE AUTO_ID = ?" ;
    UNCOMMITTEDISOWNER_PS = "SELECT COUNT(*) FROM " + ACL_OWNER_TBL
      + " WHERE PRINCIPAL_TYPE = ? AND PRINCIPAL_NAME = ? AND OWNER_ID = ?"
      + " AND RID_ID = ?" ;
    UNCOMMITTEDISLASTOWNER_PS = "SELECT PRINCIPAL_TYPE, PRINCIPAL_NAME"
      + " FROM " + ACL_OWNER_TBL + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDGETOWNERS_PS = "SELECT PRINCIPAL_TYPE, PRINCIPAL_NAME FROM "
      + ACL_OWNER_TBL + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDHASOWNERSINDB_PS = "SELECT COUNT(*) FROM "
      + ACL_OWNER_TBL + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDADDOWNER_PS = "INSERT INTO " + ACL_OWNER_TBL
      + " ( PRINCIPAL_NAME, PRINCIPAL_TYPE, OWNER_ID, RID_ID )"
      + " VALUES ( ?, ?, ?, ? )" ;
    UNCOMMITTEDREMOVEOWNER_PS = "DELETE FROM " + ACL_OWNER_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ? AND PRINCIPAL_NAME = ?"
      + " AND PRINCIPAL_TYPE = ?" ;
    UNCOMMITTEDREMOVEOWNER_PS2 = "DELETE FROM " + ACL_OWNER_TBL
      + " WHERE OWNER_ID = ? AND PRINCIPAL_NAME = ? AND PRINCIPAL_TYPE = ?" ;
    UNCOMMITTEDGETPERMISSIONMEMBERS_PS = "SELECT MEMBER_NAME FROM "
      + ACL_PERM_MEM_TBL + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDGETALLPERMISSIONSNOTFULLCONTROL_PS = "SELECT PERMISSION_NAME,"
      + " IS_PREDEFINED FROM " + ACL_PERM_TBL + " WHERE OWNER_ID = ?" ;
    UNCOMMITTEDREMOVEOWNERS_PS = "DELETE FROM " + ACL_OWNER_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDREMOVEACLENTRIES_PS = "DELETE FROM " + ACL_ENTRY_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDREMOVEACLENTRIES_PS2 = "DELETE FROM " + ACL_ENTRY_TBL
      + " WHERE OWNER_ID = ? AND PRINCIPAL_NAME = ? AND PRINCIPAL_TYPE = ?" ;
    UNCOMMITTEDCHECKINITIALPERMISSIONS_PS = "SELECT COUNT(*) FROM "
      + ACL_PERM_TBL + " WHERE OWNER_ID = ?" ;
    UNCOMMITTEDCHECKINITIALSUPPORTEDPERMS_PS = "SELECT COUNT(*) FROM "
      + ACL_SUP_PERM_TBL + " WHERE OWNER_ID = ?" ;
    UNCOMMITTEDISSUPPORTEDPERMISSION_PS = "SELECT COUNT(*) FROM "
      + ACL_SUP_PERM_TBL
      + " WHERE OWNER_ID = ? AND OBJECT_TYPE = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDGETSUPPORTEDPERMISSIONS_PS = "SELECT PERMISSION_NAME FROM "
      + ACL_SUP_PERM_TBL + " WHERE OWNER_ID = ? AND OBJECT_TYPE = ?" ;
    UNCOMMITTEDGETPERMISSION_PS = "SELECT IS_PREDEFINED FROM "
      + ACL_PERM_TBL + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDISUSEDPERMISSION_PS = "SELECT COUNT(*) FROM "
      + ACL_ENTRY_TBL + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDADDPERMISSION_PS = "INSERT INTO " + ACL_PERM_TBL
      + " ( OWNER_ID, PERMISSION_NAME, IS_PREDEFINED ) VALUES ( ?, ?, ? )" ;
    UNCOMMITTEDADDSUPPORTEDPERMISSION_PS = "INSERT INTO " + ACL_SUP_PERM_TBL
      + " ( OWNER_ID, PERMISSION_NAME, OBJECT_TYPE ) VALUES ( ?, ?, ? )" ;
    UNCOMMITTEDREMOVESUPPORTEDPERMISSION_PS = "DELETE FROM " + ACL_SUP_PERM_TBL
      + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ? AND OBJECT_TYPE = ?" ;
    UNCOMMITTEDADDPERMISSIONMEMBER_PS = "INSERT INTO " + ACL_PERM_MEM_TBL
      + " ( OWNER_ID, PERMISSION_NAME, MEMBER_NAME ) VALUES ( ?, ?, ? )" ;
    UNCOMMITTEDREMOVEPERMISSIONMEMBER_PS = "DELETE FROM " + ACL_PERM_MEM_TBL
      + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ? AND MEMBER_NAME = ?" ;
    UNCOMMITTEDREMOVEPERMISSIONMEMBERS_PS = "DELETE FROM "
      + ACL_PERM_MEM_TBL + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDISPREDEFINEDPERMISSION_PS = "SELECT IS_PREDEFINED FROM "
      + ACL_PERM_TBL + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDREMOVEACL_PS = "DELETE FROM " + ACL_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDLOCKACL_PS = "UPDATE " + ACL_TBL
      + " SET LOCKING_USER = ? WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDUNLOCKACL_PS = "UPDATE " + ACL_TBL
      + " SET LOCKING_USER = '.' WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDISACLLOCKED_PS = "SELECT LOCKING_USER FROM " + ACL_TBL
      + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDGETACLLOCKINGUSER_PS = "SELECT LOCKING_USER FROM "
      + ACL_TBL + " WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDCHANGEACLID_IN_ACLTABLE_PS = "UPDATE " + ACL_TBL
      + " SET RID_ID = ? WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDCHANGEACLID_IN_ACLENTRYTABLE_PS = "UPDATE " + ACL_ENTRY_TBL
      + " SET RID_ID = ? WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UNCOMMITTEDCHANGEACLID_IN_OWNERTABLE_PS = "UPDATE " + ACL_OWNER_TBL
      + " SET RID_ID = ? WHERE OWNER_ID = ? AND RID_ID = ?" ;
    UPDATEVERSION_PS = "UPDATE " + ACL_VERSION_TBL
      + " SET COUNTER = COUNTER + 1" ;
    CHECKVERSION_PS1 = "SELECT COUNTER FROM " + ACL_VERSION_TBL ;
    CHECKVERSION_PS2 = "INSERT INTO " + ACL_VERSION_TBL
      + " (COUNTER) VALUES (1)" ;
    GETDBVERSION_PS = "SELECT COUNTER FROM " + ACL_VERSION_TBL ;

    SN = new String[SEGCOLCNT] ;
    for ( int i = 0 ; i < SEGCOLCNT ; i++ ) {
      SN[i] = "S" + i ;
    }
    RIDMATCH = new String[SEGCOLCNT+1] ;
    RIDMATCH[1] = SN[0] + " = ?" ;
    for ( int i = 2 ; i <= SEGCOLCNT ; i++ ) {
      RIDMATCH[i] = RIDMATCH[i-1] + " AND " + SN[i-1] + " = ?" ;
    }
    for ( int i = 1 ; i < SEGCOLCNT ; i++ ) {
      RIDMATCH[i] += " AND " + SN[i] + " IS NULL" ;
    }
    RIDLIKE = new String[SEGCOLCNT+1] ;
    RIDLIKE[1] = SN[0] + " LIKE ?" ;
    RIDLIKE[2] = SN[0] + " = ?" ;
    for ( int i = 3 ; i <= SEGCOLCNT ; i++ ) {
      RIDLIKE[i] = RIDLIKE[i-1] + " AND " + SN[i-2] + " = ?" ;
    }
    for ( int i = 2 ; i <= SEGCOLCNT ; i++ ) {
      RIDLIKE[i] += " AND " + SN[i-1] + " LIKE ?" ;
    }
    UNCOMMITTEDGETACLRID_PS = new String[SEGCOLCNT+1] ;
    for ( int i = 1 ; i <= SEGCOLCNT ; i++ ) {
      UNCOMMITTEDGETACLRID_PS[i] = "SELECT AUTO_ID FROM " + ACL_RID_TBL
        + " WHERE " + RIDMATCH[i] ;
    }
    UNCOMMITTEDADDORGETACLRID_PS = new String[SEGCOLCNT+1] ;
    for ( int i = 1 ; i <= SEGCOLCNT ; i++ ) {
      UNCOMMITTEDADDORGETACLRID_PS[i] = "INSERT INTO " + ACL_RID_TBL
        + " ( USE_CNT, DEL_CNT, AUTO_ID, RID" ;

      String s0 = "", s1 = "" ;
      for ( int j = 0 ; j < i ; j++ ) {
        s0 += ", " + SN[j] ;
        s1 += ",?" ;
      }
      UNCOMMITTEDADDORGETACLRID_PS[i] += s0 += ") VALUES ( 0, 0, ?, ?"
        + s1 + ")" ;
    }

    UNCOMMITTEDGETACLRIDSLIKE_PS = new String[SEGCOLCNT+1] ;
    for ( int i = 1 ; i <= SEGCOLCNT ; i++ ) {
      UNCOMMITTEDGETACLRIDSLIKE_PS[i] = "SELECT AUTO_ID FROM " + ACL_RID_TBL
        + " WHERE " + RIDLIKE[i] ;
    }

    GETDESCENDANTSWITHACL_PS = new String[SEGCOLCNT+1] ;
    for ( int i = 1 ; i <= SEGCOLCNT ; i++ ) {
      GETDESCENDANTSWITHACL_PS[i] = "SELECT T2.RID FROM " + ACL_TBL
        + " AS T1, " + ACL_RID_TBL
        + " AS T2 WHERE T1.RID_ID = T2.AUTO_ID AND T1.OWNER_ID = ? AND "
        + RIDLIKE[i] ;
    }
    UNCOMMITTEDAREACLSUPTODATEINDB_PS = new String[MAX_IDS_PER_QUERY+1] ;
    for ( int i = 1 ; i <= MAX_IDS_PER_QUERY ; i++ ) {
      UNCOMMITTEDAREACLSUPTODATEINDB_PS[i] = "SELECT RID_ID, CHANGE_LEVEL FROM "
        + ACL_TBL + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      for ( int j = 1 ; j < i ; j++ ) {
        UNCOMMITTEDAREACLSUPTODATEINDB_PS[i] += ",?" ;
      }
      UNCOMMITTEDAREACLSUPTODATEINDB_PS[i] += ")" ;
    }

    UNCOMMITTEDGETACLSFROMDB_PS = new String[MAX_IDS_PER_QUERY+1] ;
    for ( int i = 1 ; i <= MAX_IDS_PER_QUERY ; i++ ) {
      UNCOMMITTEDGETACLSFROMDB_PS[i] = "SELECT RID_ID, CHANGE_LEVEL,"
        + " OBJECT_TYPE FROM " + ACL_TBL
        + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      for ( int j = 1 ; j < i ; j++ ) {
        UNCOMMITTEDGETACLSFROMDB_PS[i] += ",?" ;
      }
      UNCOMMITTEDGETACLSFROMDB_PS[i] += ")" ;
    }

    UNCOMMITTEDGETOWNERS_PS2 = new String[MAX_IDS_PER_QUERY+1] ;
    for ( int i = 1 ; i <= MAX_IDS_PER_QUERY ; i++ ) {
      UNCOMMITTEDGETOWNERS_PS2[i] = "SELECT RID_ID, PRINCIPAL_TYPE,"
        + " PRINCIPAL_NAME FROM " + ACL_OWNER_TBL
        + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      for ( int j = 1 ; j < i ; j++ ) {
        UNCOMMITTEDGETOWNERS_PS2[i] += ",?" ;
      }
      UNCOMMITTEDGETOWNERS_PS2[i] += ")" ;
    }

    UNCOMMITTEDGETACLENTRIES_PS2 = new String[MAX_IDS_PER_QUERY+1] ;
    for ( int i = 1 ; i <= MAX_IDS_PER_QUERY ; i++ ) {
      UNCOMMITTEDGETACLENTRIES_PS2[i] = "SELECT AUTO_ID, PRINCIPAL_NAME,"
        + " PRINCIPAL_TYPE, SORT_INDEX, IS_NEGATIVE, IS_PROPAGATED,"
        + " PERMISSION_NAME, OWNER_ID, RID_ID FROM " + ACL_ENTRY_TBL
        + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      for ( int j = 1 ; j < i ; j++ ) {
        UNCOMMITTEDGETACLENTRIES_PS2[i] += ",?" ;
      }
      UNCOMMITTEDGETACLENTRIES_PS2[i] += ") ORDER BY SORT_INDEX" ;
    }

    UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS = new String[MAX_IDS_PER_QUERY+1] ;
    UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2 = new String[MAX_IDS_PER_QUERY+1] ;
    UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3 = new String[MAX_IDS_PER_QUERY+1] ;
    for ( int i = 1 ; i <= MAX_IDS_PER_QUERY ; i++ ) {
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS[i] = "DELETE FROM "
        + ACL_TBL + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2[i] = "DELETE FROM "
        + ACL_ENTRY_TBL + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3[i] = "DELETE FROM "
        + ACL_OWNER_TBL + " WHERE OWNER_ID = ? AND RID_ID IN (?" ;
      for ( int j = 1 ; j < i ; j++ ) {
        UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS[i] += ",?" ;
        UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2[i] += ",?" ;
        UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3[i] += ",?" ;
      }
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS[i] += ")" ;
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2[i] += ")" ;
      UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3[i] += ")" ;
    }

    UNCOMMITTEDSETACLENTRYPROPAGATION_PS = "UPDATE " + ACL_ENTRY_TBL
      + " SET IS_PROPAGATED = ? WHERE AUTO_ID = ?" ;
    UNCOMMITTEDREMOVEPERMISSION_PS = "DELETE FROM " + ACL_PERM_TBL
      + " WHERE OWNER_ID = ? AND PERMISSION_NAME = ?" ;
    UNCOMMITTEDGETAFFECTEDACLIDS_PS = "SELECT RID FROM " + ACL_RID_TBL
      + " WHERE AUTO_ID IN (SELECT DISTINCT T2.AUTO_ID FROM " + ACL_ENTRY_TBL
      + " AS T1, " + ACL_RID_TBL + " AS T2 WHERE T1.OWNER_ID = ? AND"
      + " T1.RID_ID = T2.AUTO_ID AND T1.PERMISSION_NAME = ?)" ;
  }

  private class LocalAutoID {
    private Map autoIDs ;
    
    LocalAutoID() {
      autoIDs = new HashMap() ;
    }
    
    long getNextID ( String tbl ) throws WcmException {
      synchronized ( autoIDs ) {
        AutoID a = (AutoID)autoIDs.get ( tbl ) ;
        if ( a == null ) {
          a = AutoID.getAutoID ( tbl ) ;
          if ( a == null ) {
            throw new WcmException ( "autoID not found" ) ;
          }
          autoIDs.put ( tbl, a ) ;
        }
        return a.getNextID() ;
      }
    }
  }

  private LocalAutoID localAutoIDs = null ;
  private static Segmentor sm = null ;
  private static RidCache ridCache = null ;

  static {
    s_log = Location.getLocation ( OpenSQLDatabaseConnectionUncached.class ) ;
  } ;

  /**
   * Construct
   *
   * @param applicationID TBD: Description of the incoming method parameter
   * @param managerID TBD: Description of the incoming method parameter
   * @param poolID TBD: Description of the incoming method parameter
   * @param permissionCache TBD: Description of the incoming method parameter
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   */

  public OpenSQLDatabaseConnectionUncached ( String applicationID,
  String managerID, String poolID, ICache permissionCache,
  JDBCConnectionPool p ) throws AclLoadClassException, AclPersistenceException {
    if (( applicationID == null ) || ( managerID == null )
    || ( poolID == null )) {
      throw new java.lang.IllegalArgumentException() ;
    }
    if ( localAutoIDs == null ) {
      localAutoIDs = new LocalAutoID() ;
    }
    synchronized ( OpenSQLDatabaseConnectionUncached.class ) {
      if ( sm == null ) {
        sm = new Segmentor() ;
      }
      if ( ridCache == null ) {
        ridCache = new RidCache() ;
      }
      if ( cleanUpThread == null ) {
        cleanUpThread = new CleanUpThread() ;
        cleanUpThread.setDaemon ( true ) ;
        cleanUpThread.start() ;
      }
    }
    m_applicationID = applicationID ;
    m_managerID = managerID ;
    m_poolID = poolID ;
    conPool = p ;
    if ( s_log.beDebug()) {
      s_log.debugT("OpenSQLDatabaseConnectionUncached", "applicationID "
        + applicationID + ", managerID " + managerID + ", poolID " + poolID
        + ", permissionCache " + permissionCache.getID());
    }
    /*
    * due to the unfortunate design of this block buster and the
    * restriction of OpenSQL to hold at any point in time a maximum
    * of one open connection per thread, we need to sync our memory
    * representations of the ACL auto_id objects with the database
    * before opening the one-and-only connection
    *
    * NOTE that this wastes one auto_id per table but then, who cares
    */
    try {
      String[] tl = { ACL_OWNER_TBL, ACL_TBL, ACL_MANAGER_TBL, ACL_ENTRY_TBL,
        ACL_PERM_TBL, ACL_SUP_PERM_TBL, ACL_PERM_MEM_TBL, ACL_RID_TBL } ;
      for ( int i = 0 ; i < tl.length ; i++ ) {
        long aid = localAutoIDs.getNextID ( tl[i] ) ;
        if ( tl[i].equals ( ACL_RID_TBL )) {
          cleanUpThread.setHighId ( aid ) ;
        }
      }
    }
    catch ( WcmException we ) {
      throw new AclPersistenceException ( "cannot init auto IDs; " + we ) ;
    }
    m_permissionCache = permissionCache ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    String trimmedApplicationID = sm.getTrimMatchString ( m_applicationID ) ;
    String trimmedManagerID = sm.getTrimMatchString ( m_managerID ) ;
    Connection c = null ;
    try {
      c = getDBConnection() ;
      ps = c.prepareStatement ( OPENSQLDATABASECONNECTIONUNCACHED_PS1 ) ;
      ps.setString ( 1, trimmedApplicationID ) ;
      ps.setString ( 2, trimmedManagerID ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        ownerId = rs.getLong ( 1 ) ;
        rs.close() ;
        rs = null ;
      }
      else {
        rs.close() ;
        rs = null ;
        ps.close() ;
        ownerId = localAutoIDs.getNextID ( ACL_MANAGER_TBL ) ;
        ps = c.prepareStatement ( OPENSQLDATABASECONNECTIONUNCACHED_PS2 ) ;
        ps.setLong ( 1, ownerId ) ;
        ps.setString ( 2, trimmedApplicationID ) ;
        ps.setString ( 3, trimmedManagerID ) ;
        ps.executeUpdate() ;
      }
      ps.close() ;
      ps = null ;
      checkVersion ( c ) ;
      checkInsertInitialPermissions ( c ) ;
    }
    catch ( WcmException we ) {
      throw new AclPersistenceException ( we ) ;
    }
    catch ( SQLException e ) {
      throw new AclPersistenceException ( e ) ;
    }
    finally {
      try {
        if ( rs != null ) {
          rs.close() ;
        }
        if ( ps != null ) {
          ps.close() ;
        }
        if ( c != null ) {
          returnDBConnection ( c ) ;
        }
      }
      catch ( SQLException e ) {
        throw new AclPersistenceException ( e ) ;
      }
    }
  }

  /**
   * @param owner TBD: Description of the incoming method parameter
   * @param externalAclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @return a new ACL (create in the database)
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public JDBCAcl createAcl ( IUMPrincipal owner, String externalAclID,
  IObjectType objectType) throws AclPersistenceException, AclExistsException {
    if (( owner == null ) || ( externalAclID == null )
    || ( objectType == null )) {
      throw new java.lang.IllegalArgumentException() ;
    }
    if ( s_log.beDebug()) {
      s_log.debugT ( "createAcl(991)", "owner " + owner.getId()
        + ", externalAclID " + externalAclID + ", objectType " + objectType ) ;
    }
    Connection c = null ;
    try {
      c = getDBConnection() ;
      synchronized ( m_main_connection_lock ) {
        try {
          if ( uncommittedCheckAcl ( c, externalAclID )) {
            throw new AclExistsException() ;
          }
          /*
          * in case the database is inconsistent, entries may exist
          * for a removed acl - remove them
          */
          uncommittedRemoveAclEntries(c, externalAclID);
          /*
          * in case the database is inconsistent, owners may exist
          * for a removed acl - remove them
          */
          uncommittedRemoveOwners(c, externalAclID);
          if (!uncommittedAddOwner(c, externalAclID, owner)) {
            throw new Exception("can't create owner");
          }
          if (!uncommittedAddAcl(c, externalAclID, objectType)) {
            throw new Exception("can't create acl");
          }
          c.commit();
        }
        catch (AclExistsException e) {
          try {
            c.rollback();
          }
          catch ( SQLException sqlException ) {
            s_log.debugT ( com.sapportals.wcm.util.logging.LoggingFormatter
              .extractCallstack ( sqlException )) ;
          }
          throw e;
        }
        catch (Exception e) {
          try {
            c.rollback();
          }
          catch ( SQLException sqlException ) {
            s_log.debugT ( com.sapportals.wcm.util.logging.LoggingFormatter
              .extractCallstack ( sqlException )) ;
          }
          try {
            reopenMainConnection();
          }
          catch (SQLException sqlException) {
            s_log.debugT ( com.sapportals.wcm.util.logging.LoggingFormatter
              .extractCallstack ( sqlException )) ;
          }
          throw new AclPersistenceException(e);
        }
        return getAcl(externalAclID, false); // COOKED
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "connection problem ", se.getMessage()) ;
      throw new AclPersistenceException ( se ) ;
    }
    finally {
      if ( c != null ) {
        returnDBConnection ( c ) ;
      }
    }
  }

  protected boolean hasDescendantsWithAcl ( Connection con,
  String externalAclIDRoot ) throws SQLException {
    long ridId = uncommittedGetAclRid ( con, externalAclIDRoot ) ;
    if ( ridId == -1 ) {
      return false ;
    }
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      ps = con.prepareStatement ( HASDESCENDANTSWITHACL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  protected List getDescendantsWithAcl ( Connection con,
  String externalAclIDRoot ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      String trimmedRid = sm.getTrimLikeString ( externalAclIDRoot ) ;
      int sc = sm.getSegCount ( trimmedRid ) ;
      String stmt = GETDESCENDANTSWITHACL_PS[sc] ;
      List rv = new LinkedList() ;
      ps = con.prepareStatement ( stmt ) ;
      ps.setLong ( 1, ownerId ) ;
      for ( int i = 0 ; i < sc ; i++ ) {
        String ss = sm.getSegment ( trimmedRid, i ) ;
        ps.setString ( 2 + i, ss ) ;
      }
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        String rid = sm.getUntrimString ( rs.getString ( 1 )) ;
        rv.add ( rid ) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  protected void uncommittedRemoveDescendantsWithAcl ( Connection con,
  String externalAclIDRoot ) throws SQLException {
    List children = uncommittedGetAclRidsLike ( con, externalAclIDRoot ) ;
    if (( children == null ) || ( children.size() == 0 )) {
      return ;
    }
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      long[] lva = new long[MAX_IDS_PER_QUERY] ;
      for ( int i = 0, ls = children.size() ; i < ls ; i += MAX_IDS_PER_QUERY ){
        String stmt = null ;
        int js = Math.min ( MAX_IDS_PER_QUERY, ls - i )  ;
        for ( int i0 = 0 ; i0 < 3 ; i0++ ) {
          switch ( i0 ) {
            case 0 : //ACL table
              for ( int j = 0 ; j < js ; j++ ) {
                lva[j] = ((Long)children.get ( i + j )).longValue() ;
              }
              stmt = UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS[js] ;
              break;
            case 1 : //ENTRY table
              stmt = UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS2[js] ;
              break ;
            case 2 : //OWNER table
              stmt = UNCOMMITTEDREMOVEDESCENDANTSWITHACL_PS3[js] ;
              break ;
          }
          ps = con.prepareStatement ( stmt ) ;
          ps.setLong ( 1, ownerId ) ;
          for ( int j = 0 ; j < js ; j++ ) {
            ps.setLong ( 2 + j, lva[j] ) ;
          }
          if ( ps.executeUpdate() < 0 ) {
            s_log.debugT ( "run", "uncommittedRemoveDescendantsWithAcl ( "
              + externalAclIDRoot + ") [stmt=" + stmt + "]" ) ;
          }
          ps.close() ;
          ps = null ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param externalAclIDRoot TBD: Description of the incoming method parameter
   * @return all ACL IDs (strings) in a list which start with the specified
   *      prefix
   * @exception SQLException Exception raised in failure situation
   */
  protected List uncommittedGetDescendantsWithAcl ( Connection con,
  String externalAclIDRoot ) throws SQLException {
    return getDescendantsWithAcl ( con, externalAclIDRoot ) ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param externalAclID TBD: Description of the incoming method parameter
   * @return an ACL object for an ACL in the database (may be null)
   * @exception SQLException Exception raised in failure situation
   */
  protected AclRec uncommittedGetAcl ( Connection con, String externalAclID )
  throws SQLException {
    AclRec rv = null ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      /*
      * use id cache here, just like urimapper cache in index management
      */
      long ridId = uncommittedGetAclRid ( con, externalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDGETACL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        String objectType = sm.getUntrimString ( rs.getString ( 1 )) ;
        long changeList = rs.getLong ( 2 ) ;
        rv = new AclRec ( externalAclID, ridId, objectType, changeList ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param entryID TBD: Description of the incoming method parameter
   * @return the ACL ID which an ACE is attached to
   * @exception SQLException Exception raised in failure situation
   */
  protected String uncommittedGetInternalAclIDFromEntryID ( Connection con,
  long entryID ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDGETINTERNALACLIDFROMENTRYID_PS ) ;
      ps.setLong ( 1, entryID ) ;
      rs = ps.executeQuery() ;
      String rv = null ;
      if ( rs.next()) {
        rv = sm.getUntrimString ( rs.getString ( 1 )) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  protected AclRec[] uncommittedGetAclsFromDB ( Connection con,
  String[] aclIDs ) throws SQLException {
    /*
    * lookup rids, object type, and change level for the given
    * list of internal ACL ids; for each row from the result set
    * create an ACL record with the associated external ACL id
    */
    if ( aclIDs.length == 0 ) {
      return new AclRec[0] ;
    }
    long[] ridIds = uncommittedGetAclRids ( con, aclIDs ) ;
    AclRec[] rv = new AclRec[aclIDs.length] ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      String stmt = null ;
      int js0 = -1 ;
      for ( int i = 0 ; i < aclIDs.length ; i += MAX_IDS_PER_QUERY ) {
        int ndx, j, js = Math.min ( aclIDs.length - i, MAX_IDS_PER_QUERY ) ;
        if ( js0 != js ) {
          if ( ps != null ) {
            ps.close() ;
          }
          stmt = UNCOMMITTEDGETACLSFROMDB_PS[js] ;
          ps = con.prepareStatement ( stmt ) ;
          js0 = js ;
        }
        ps.setLong ( 1, ownerId ) ;
        for ( j = 0 ; j < js ; j++ ) {
          ps.setLong ( 2 + j, ridIds[j + i] ) ;
        }
        j = i ;
        rs = ps.executeQuery() ;
        while ( rs.next()) {
          long ridId = rs.getLong ( 1 ) ;
          boolean found = false ;
          for ( ndx = j ; ndx < i + js ; ndx++ ) {
            if ( found = ( ridIds[ndx] == ridId )) {
              if ( ndx == j ) {
                j++ ;
              }
              break ;
            }
          }
          if ( !found ) {
            continue ;
          }
          long cl = rs.getLong ( 2 ) ;
          String objectType = sm.getUntrimString ( rs.getString ( 3 )) ;
          rv[ndx] = new AclRec ( aclIDs[ndx], ridIds[ndx], objectType, cl ) ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param changeLevel TBD: Description of the incoming method parameter
   * @return true iff the change level of the ACL in the database is not larger
   *      than the specified one
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsAclUpToDate ( Connection con,
  String internalAclID, long changeLevel ) throws SQLException {
    /*
    * retrieve change level for a given ACL rid and match against
    * passed change level
    */
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      boolean rv = false ;
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDISACLUPTODATE_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = changeLevel == rs.getLong ( 1 ) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param aclIDs TBD: Description of the incoming method parameter
   * @param changeLevels TBD: Description of the incoming method parameter
   * @return up to date flags for ACLs (check change level in the database)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean[] uncommittedAreAclsUpToDateInDB ( Connection con,
  String[] aclIDs, long[] changeLevels ) throws SQLException {
    if ( aclIDs.length == 0 ) {
      return new boolean[0] ;
    }
    long[] ridIds = uncommittedGetAclRids ( con, aclIDs ) ;
    boolean[] rv = new boolean[aclIDs.length] ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      int js0 = -1 ;
      String stmt = null ;
      for ( int i = 0 ; i < aclIDs.length ; i += MAX_IDS_PER_QUERY ) {
        int ndx, j, js = Math.min ( aclIDs.length - i, MAX_IDS_PER_QUERY ) ;
        if ( js != js0 ) {
          if ( ps != null ) {
            ps.close() ;
          }
          stmt = UNCOMMITTEDAREACLSUPTODATEINDB_PS[js] ;
          ps = con.prepareStatement ( stmt ) ;
          js0 = js ;
        }
        ps.setLong ( 1, ownerId ) ;
        for ( j = 0 ; j < js ; j++ ) {
          ps.setLong ( 2 + j, ridIds[j + i] ) ;
        }
        j = i ;
        rs = ps.executeQuery() ;
        while ( rs.next()) {
          long ridId = rs.getLong ( 1 ) ;
          boolean found = false ;
          for ( ndx = j ; ndx < i + js ; ndx++ ) {
            if ( found = ( ridIds[ndx] == ridId )) {
              if ( ndx == j ) {
                j++ ;
              }
              break ;
            }
          }
          if ( !found ) {
            continue ;
          }
          rv[ndx] = rs.getLong ( 2 ) == changeLevels[ndx] ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    return rv ;
  }

  /**
   * Increase the change level of the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedIncreaseAclChangeLevel ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDINCREASEACLCHANGELEVEL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      if ( ps.executeUpdate() > 0 ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL exists in the database
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckAcl ( Connection con, String internalAclID )
  throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDCHECKACL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Set the ACE sort index
   *
   * @param con TBD: Description of the incoming method parameter
   * @param aclEntryID TBD: Description of the incoming method parameter
   * @param sortIndex TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedSetAclEntrySortIndex ( Connection con,
  long aclEntryID, int sortIndex ) throws SQLException {
    PreparedStatement ps = null;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDSETACLENTRYSORTINDEX_PS ) ;
      ps.setInt ( 1, sortIndex ) ;
      ps.setLong ( 2, aclEntryID ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Define the propagate-flag of the ACE
   *
   * @param con TBD: Description of the incoming method parameter
   * @param aclEntryID TBD: Description of the incoming method parameter
   * @param propagate TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedSetAclEntryPropagation ( Connection con,
  long aclEntryID, boolean propagate ) throws SQLException {
    PreparedStatement ps = null;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDSETACLENTRYPROPAGATION_PS ) ;
      ps.setShort ( 1, (short)((propagate)?(1):(0))) ;
      ps.setLong ( 2, aclEntryID ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL has ACEs
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckAclEntries ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDCHECKACLENTRIES_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return true iff the permission exists
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckPermission ( Connection con,
  String permissionName ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDCHECKPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private class EntryData {
    public long m_aclEntryID ;
    public String m_internalAclID ;
    public String m_principalName ;
    public int m_principalType ;
    public int m_sortIndex ;
    public boolean m_negative ;
    public boolean m_propagated ;
    public String m_permissionName ;
  }
  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the ACEs of the ACL (principal may be null; otherwise only the
   *      entries belonging to the pricnipal are returned)
   * @exception SQLException Exception raised in failure situation
   */
  protected AclEntryList uncommittedGetAclEntries ( Connection con,
    String internalAclID,
    boolean raw ) throws SQLException {
    AclEntryList rv = new AclEntryList() ;
    PreparedStatement ps = null;
    ResultSet rs = null;
    List edl = new ArrayList ( 50 ) ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDGETACLENTRIES_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        EntryData ed = new EntryData() ;
        ed.m_aclEntryID = rs.getLong ( 1 ) ;
        ed.m_internalAclID = internalAclID ;
        ed.m_principalName = sm.getUntrimString ( rs.getString ( 2 )) ;
        ed.m_principalType = rs.getInt ( 3 ) ;
        ed.m_sortIndex = rs.getInt ( 4 ) ;
        ed.m_negative = rs.getShort ( 5 ) != 0 ;
        ed.m_propagated = rs.getShort ( 6 ) != 0 ;
        ed.m_permissionName = sm.getUntrimString ( rs.getString ( 7 )) ;
        edl.add ( ed ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    for ( int i = 0 ; i < edl.size() ; i++ ) {
      try {
        EntryData ed = (EntryData)edl.get ( i ) ;
        IUMPrincipal newPrincipal = getUMPrincipal ( ed.m_principalName,
          getPrincipalTypeFromTag ( ed.m_principalType ), raw ) ;
        if ( newPrincipal != null ) {
          /*
          * ignore users which are not configured in the user management
          */
          IAclPermission permission = uncommittedGetPermission ( con,
            ed.m_permissionName ) ;
          JDBCAclEntry ae = (raw && ( newPrincipal instanceof IUMPrincipalRaw ))
            ? new JDBCAclEntryRaw ( this, newPrincipal, ed.m_negative,
              permission, ed.m_sortIndex, ed.m_propagated )
            : new JDBCAclEntry ( this, newPrincipal, ed.m_negative, permission,
              ed.m_sortIndex, ed.m_propagated ) ;
          ae.setIDs ( ed.m_aclEntryID, ed.m_internalAclID ) ;
          rv.add ( ae ) ;
        }
      }
      catch ( InvalidClassException e ) {
        /*
        * will not be thrown; just to keep the exception from the
        * method prototype
        */
        s_log.debugT(e.getMessage());
      }
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param acls TBD: Description of the incoming method parameter
   * @return the ACEs of the ACLs (don't get them all at once to limit the size
   *      of the SQL statement) entries of the acls table may be null
   * @exception SQLException Exception raised in failure situation
   */
  protected AclEntryList[] uncommittedGetAclEntries ( Connection con,
    AclRec[] acls,
    boolean raw ) throws SQLException {
    if ( acls.length == 0 ) {
      return new AclEntryList[0] ;
    }
    AclEntryList[] rv = new AclEntryList[acls.length] ;
    int cls = acls.length ;
    for ( int i = 0 ; i < acls.length ; i++ ) {
      if ( acls[i] == null ) {
        cls-- ;
      }
    }
    int[] ndxmap = new int[cls] ;
    long[] idmap = new long[cls] ;
    List[] edmap = new ArrayList[cls] ;
    for ( int i = 0, j = 0 ; i < acls.length; i++ ) {
      if ( acls[i] != null ) {
        ndxmap[j] = i ;
        idmap[j] = acls[i].getAclRidID() ;
        edmap[j] = new ArrayList ( 64 ) ;
        rv[i] = new AclEntryList() ;
        j++ ;
      }
    }
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      int js0 = -1 ;
      String stmt = null ;
      for ( int i = 0 ; i < cls ; i += MAX_IDS_PER_QUERY ) {
        int ndx, js = Math.min ( cls - i, MAX_IDS_PER_QUERY ) ;
        if ( js0 != js ) {
          if ( ps != null ) {
            ps.close() ;
          }
          stmt = UNCOMMITTEDGETACLENTRIES_PS2[js] ;
          ps = con.prepareStatement ( stmt ) ;
          js0 = js ;
        }
        ps.setLong ( 1, ownerId ) ;
        for ( int j = 0 ; j < js ; j++ ) {
          ps.setLong ( 2 + j, idmap[j + i] ) ;
        }
        rs = ps.executeQuery() ;
        while ( rs.next()) {
          long ridId = rs.getLong ( 9 ) ;
          boolean found = false ;
          for ( ndx = i ; ndx < i + js ; ndx++ ) {
            if ( found = ( idmap[ndx] == ridId )) {
              break ;
            }
          }
          if ( !found ) {
            continue ;
          }
          EntryData ed = new EntryData() ;
          ed.m_aclEntryID = rs.getLong ( 1 ) ;
          ed.m_principalName = sm.getUntrimString ( rs.getString ( 2 )) ;
          ed.m_principalType = rs.getShort ( 3 ) ;
          ed.m_sortIndex = rs.getInt ( 4 ) ;
          ed.m_negative = rs.getShort ( 5 ) != 0 ;
          ed.m_propagated = rs.getShort ( 6 ) != 0 ;
          ed.m_permissionName = sm.getUntrimString ( rs.getString ( 7 )) ;
          edmap[ndx].add ( ed ) ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    /*
    * NOTE: the result set needs to be closed this early as some of the
    * methods called down below might issue their own db requests which
    * would, of course, fail as we haven't closed the result set here.
    *
    * at this point the entry table 'et' holds a list of pairs
    * of an ACL rid and the associated set of ACL entry table data;
    *
    * this data is finally turned into ACL entry objects now
    */
    for ( int i = 0 ; i < cls ; i++ ) {
      try {
        for ( int j = 0 ; j < edmap[i].size() ; j++ ) {
          EntryData ed = (EntryData)edmap[i].get ( j ) ;
          IUMPrincipal principal = getUMPrincipal ( ed.m_principalName,
            getPrincipalTypeFromTag ( ed.m_principalType ), raw) ;
          if ( principal != null ) {
            /*
            * ignore users which are not configured in the user management
            */
            IAclPermission permission = uncommittedGetPermission ( con,
              ed.m_permissionName ) ;
            JDBCAclEntry ae = ( raw && ( principal instanceof IUMPrincipalRaw ))
              ? new JDBCAclEntryRaw ( this, principal, ed.m_negative,
                permission, ed.m_sortIndex, ed.m_propagated )
              : new JDBCAclEntry ( this, principal, ed.m_negative, permission,
                ed.m_sortIndex, ed.m_propagated ) ;
            ae.setIDs ( ed.m_aclEntryID, ed.m_internalAclID ) ;
            rv[ndxmap[i]].add ( ae ) ;
          }
        }
      }
      catch ( InvalidClassException e ) {
        /*
        * will not be thrown; just to keep the exception from the
        * method prototype
        */
        s_log.debugT(e.getMessage());
      }
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return a list with the IDs of the ACLs (strings) which use the specified
   *      permissions in one of their ACEs
   * @exception SQLException Exception raised in failure situation
   */
  protected List uncommittedGetAffectedAclIDs ( Connection con,
  String permissionName ) throws SQLException {
    List rv = new LinkedList() ;
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDGETAFFECTEDACLIDS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        String s1 = sm.getUntrimString ( rs.getString ( 1 )) ;
        rv.add ( s1 ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    return rv ;
  }

  /**
   * Add a new ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddAcl ( Connection con, String internalAclID,
  IObjectType objectType ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      long aclId = localAutoIDs.getNextID ( ACL_TBL ) ;
      String trimmedObjectType = sm.getTrimMatchString ( objectType.getName()) ;
      long ridId = uncommittedAddOrGetAclRid ( con, internalAclID ) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDACL_PS ) ;
      ps.setLong ( 1, aclId ) ;
      ps.setString ( 2, trimmedObjectType ) ;
      ps.setLong ( 3, ownerId ) ;
      ps.setLong ( 4, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    catch ( WcmException we ) {
      throw new SQLException ( "problem with auto id: " + we ) ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private long uncommittedAddOrGetAclRid ( Connection con, String id )
  throws SQLException {
    long rv ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      if (( rv = uncommittedGetAclRid ( con, id )) != -1 ) {
        return rv ;
      }
      rv = localAutoIDs.getNextID ( ACL_RID_TBL ) ;
      String trimmedId = sm.getTrimMatchString ( id ) ;
      int sc = sm.getSegCount ( trimmedId ) ;
      String stmt = UNCOMMITTEDADDORGETACLRID_PS[sc] ;
      ps = con.prepareStatement ( stmt ) ;
      /*
      * NOTE: we can't use the segmentor trim-and-match method here as
      * it would insert delimitor symbols (.) at the segment boundaries
      * which isn't what we want for the whole rid
      *
      * Not sure whether it's correct to trim the id string??
      */
      ps.setLong ( 1, rv ) ;
//XXX:      ps.setString ( 2, id.trim() + "." ) ;
      ps.setString ( 2, id + "." ) ;
      for ( int i = 0 ; i < sc ; i++ ) {
        String ss = sm.getSegment ( trimmedId, i ) ;
        ps.setString ( 3 + i, ss ) ;
      }
      if ( ps.executeUpdate() <= 0 ) {
        throw new SQLException ( "failed to insert new rid" ) ;
      }
      ridCache.put ( rv, id ) ;
      return rv ;
    }
    catch ( WcmException we ) {
      throw new SQLException ( "problem with auto id: " + we ) ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }


  private long uncommittedGetAclRid ( Connection con, String id )
  throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      /*
      * check whether ID already exists
      */
      long rv = ridCache.get ( id ) ;
      if ( rv != RidCache.RIDCACHE_NO_ENTRY ) {
        return ( rv ) ;
      }
      String trimmedId = sm.getTrimMatchString ( id ) ;
      int sc = sm.getSegCount ( trimmedId ) ;
      String stmt = UNCOMMITTEDGETACLRID_PS[sc] ;
      ps = con.prepareStatement ( stmt ) ;
      for ( int i = 0 ; i < sc ; i++ ) {
        String ss = sm.getSegment ( trimmedId, i ) ;
        ps.setString ( 1 + i, ss ) ;
      }
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = rs.getLong ( 1 ) ;
      }
      else {
        rv = RidCache.RIDCACHE_NULL_ENTRY ;
      }
      ridCache.put ( rv, id ) ;
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private long[] uncommittedGetAclRids ( Connection con, String[] ids )
  throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    long[] rv = new long[ids.length] ;
    int i = 0 ;
    int sc0 = -1 ;
    try {
      for ( ; i < ids.length ; i++ ) {
        if (( rv[i] = ridCache.get ( ids[i] )) != RidCache.RIDCACHE_NO_ENTRY ) {
          /*
          * cached entry for ids[i] found; get on to the next one
          */
          continue ;
        }
        /*
        * even no empty mappig found; let's take a look at the database
        * and fill in the slot with the info we get from there
        */
        String trimmedId = sm.getTrimMatchString ( ids[i] ) ;
        int sc = sm.getSegCount ( trimmedId ) ;
        if ( sc != sc0 ) {
          /*
          * ayee; number of string segments has changed which is we need
          * to switch to another prepared statement to retrieve the db entry
          */
          String stmt = UNCOMMITTEDGETACLRID_PS[sc] ;
          if ( ps != null ) {
            ps.close() ;
          }
          ps = con.prepareStatement ( stmt ) ;
          sc0 = sc ;
        }
        for ( int j = 0 ; j < sc ; j++ ) {
          String ss = sm.getSegment ( trimmedId, j ) ;
          ps.setString ( 1 + j, ss ) ;
        }
        rs = ps.executeQuery() ;
        if ( rs.next()) {
          rv[i] = rs.getLong ( 1 ) ;
        }
        else {
          rv[i] = RidCache.RIDCACHE_NULL_ENTRY ;
        }
        ridCache.put ( rv[i], ids[i] ) ;
        rs.close() ;
        rs = null ;
      }
      return rv ;
    }
    finally {
      /*
      * just in case an exception throws us out of that loop we're
      * filling in some decent values
      */
      for ( ; i < ids.length ; i++ ) {
        rv[i] = -1 ;
      }
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private List uncommittedGetAclRidsLike ( Connection con, String rid )
  throws SQLException {
    /*
    * NOTE that there is not guarantee that the auto_id code doesn't
    * generate a key that equals '-1'; it's just very unlikely! It
    * requires 2^31 restarts of the java vm, and then, on the 2^31st
    * restart the java vm has to create 2^31 + 1 new auto_ids for the
    * given table, ACL_RID_TBL in this case; only in this situation
    * the + 1 to get to the next auto_id will generate an overflow
    */
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      /*
      * check whether ID already exists
      */
      String trimmedRid = sm.getTrimLikeString ( rid ) ;
      int sc = sm.getSegCount ( trimmedRid ) ;
      String stmt = UNCOMMITTEDGETACLRIDSLIKE_PS[sc] ;
      ps = con.prepareStatement ( stmt ) ;
      for ( int i = 0 ; i < sc ; i++ ) {
        String ss = sm.getSegment ( trimmedRid, i ) ;
        ps.setString ( 1 + i, ss ) ;
      }
      List rv = null ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = new ArrayList ( 64 ) ;
        rv.add ( new Long ( rs.getLong ( 1 ))) ;
        while ( rs.next()) {
          rv.add ( new Long ( rs.getLong ( 1 ))) ;
        }
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Add a new ACE
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param aclEntry TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected long uncommittedAddAclEntry ( Connection con, String internalAclID,
  JDBCAclEntry aclEntry ) throws SQLException {
    /*
    * add a new entry to the ACL_ENTRY_TBL, of course, with the RID going into
    * the ACL_RID_TBL table, and return the new ACL_ENTRY_TBL.AUTO_ID as the
    * id of the newly created entry
    */
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      IUMPrincipal principal = populate ( aclEntry.getPrincipal()) ;
      if ( principal == null ) {
        throw new java.lang.IllegalArgumentException() ;
      }
      JDBCPermission permission = (JDBCPermission)aclEntry.getPermission() ;
      if ( permission == null ) {
        throw new java.lang.IllegalArgumentException() ;
      }
      long ridId = uncommittedAddOrGetAclRid ( con, internalAclID ) ;
      long aeId = localAutoIDs.getNextID ( ACL_ENTRY_TBL ) ;
      String trimmedPrincipalName = sm.getTrimMatchString ( principal.getId()) ;
      String trimmedPermissionName = sm.getTrimMatchString (
        permission.getName()) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDACLENTRY_PS ) ;
      ps.setLong ( 1, aeId ) ;
      ps.setString ( 2, trimmedPrincipalName ) ;
      ps.setShort ( 3, (short)getPrincipalTypeTag ( principal )) ;
      ps.setInt ( 4, aclEntry.getSortIndex()) ;
      ps.setShort ( 5, (short)((aclEntry.isNegative())?(1):(0))) ;
      ps.setShort ( 6, (short)((aclEntry.isPropagated())?(1):(0))) ;
      ps.setString ( 7, trimmedPermissionName ) ;
      ps.setLong ( 8, ownerId ) ;
      ps.setLong ( 9, ridId ) ;
      if ( ps.executeUpdate() <= 0 ) {
        throw new SQLException ( "failed to insert new acl entry" ) ;
      }
      return aeId ;
    }
    catch ( AclPersistenceException ae ) {
      /* doesn't really happen with the acl entry getters */
      throw new SQLException ( "problem with acl entry: " + ae ) ;
    }
    catch ( WcmException we ) {
      throw new SQLException ( "problem with auto id: " + we ) ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove an ACE
   *
   * @param con TBD: Description of the incoming method parameter
   * @param aeID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveAclEntry ( Connection con, long aeID )
  throws SQLException {
    PreparedStatement ps = null ;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEACLENTRY_PS ) ;
      ps.setLong ( 1, aeID ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return true iff the specified principal is an owner of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsOwner ( Connection con, String internalAclID,
  IUMPrincipal owner ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      String trimmedOwner = sm.getTrimMatchString ( owner.getId()) ;
      ps = con.prepareStatement ( UNCOMMITTEDISOWNER_PS ) ;
      ps.setInt ( 1, getPrincipalTypeTag ( owner )) ;
      ps.setString ( 2, trimmedOwner ) ;
      ps.setLong ( 3, ownerId ) ;
      ps.setLong ( 4, ridId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param pOwnerId TBD: Description of the incoming method parameter
   * @param ownerType TBD: Description of the incoming method parameter
   * @return true iff the specified principal is the last owner of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsLastOwner ( Connection con,
  String internalAclID, String pOwnerId, int ownerType ) throws SQLException {
    /*
    * select 'all' owner entries for the given ACL rid belonging to
    * the given owner; return true if only one entry exists and this
    * entry matches the criteria
    */
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      boolean rv = false ;
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDISLASTOWNER_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        int principalType = rs.getInt ( 1 ) ;
        String dbOwnerId = sm.getUntrimString ( rs.getString ( 2 )) ;
        rv = (( pOwnerId.equals ( dbOwnerId ))
          && ( ownerType == getPrincipalTypeFromTag ( principalType ))
          && ( ! rs.next())) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the owners of the ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected UMPrincipalList uncommittedGetOwners ( Connection con,
    String internalAclID,
    boolean raw ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      UMPrincipalList rv = new UMPrincipalList() ;
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDGETOWNERS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        int principalType = rs.getInt ( 1 ) ;
        String principalName = sm.getUntrimString ( rs.getString ( 2 )) ;
        IUMPrincipal ump = getUMPrincipal ( principalName,
          getPrincipalTypeFromTag ( principalType ), raw ) ;
        if ( ump != null ) {
          rv.add ( ump ) ;
        }
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param acls TBD: Description of the incoming method parameter
   * @return the owners of the ACLs (don't get them all at once to limit the
   *      size of the SQL statement; entries of the ACL array may be null)
   * @exception SQLException Exception raised in failure situation
   */
  protected UMPrincipalList[] uncommittedGetOwners ( Connection con,
    AclRec[] acls,
    boolean raw ) throws SQLException {
    if ( acls.length == 0 ) {
      return new UMPrincipalList[0] ;
    }
    UMPrincipalList[] rv = new UMPrincipalList[acls.length] ;
    int cls = acls.length ;
    for ( int i = 0 ; i < acls.length; i++ ) {
      if ( acls[i] == null ) {
        cls-- ;
      }
    }
    int[] ndxmap = new int[cls] ;
    long[] idmap = new long[cls] ;
    for ( int i = 0, j = 0 ; i < acls.length; i++ ) {
      if ( acls[i] != null ) {
        ndxmap[j] = i ;
        idmap[j] = acls[i].getAclRidID() ;
        rv[i] = new UMPrincipalList() ;
        j++ ;
      }
    }
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      int js0 = -1 ;
      String stmt = null ;
      for ( int i = 0 ; i < cls ; i += MAX_IDS_PER_QUERY ) {
        int ndx, js = Math.min ( cls - i, MAX_IDS_PER_QUERY ) ;
        if ( js0 != js ) {
          if ( ps != null ) {
            ps.close() ;
          }
          stmt = UNCOMMITTEDGETOWNERS_PS2[js] ;
          ps = con.prepareStatement ( stmt ) ;
          js0 = js ;
        }
        ps.setLong ( 1, ownerId ) ;
        for ( int j = 0 ; j < js ; j++ ) {
          ps.setLong ( 2 + j, idmap[j + i] ) ;
        }
        rs = ps.executeQuery() ;
        while ( rs.next()) {
          long ridId = rs.getLong ( 1 ) ;
          boolean found = false ;
          for ( ndx = i ; ndx < i + js ; ndx++ ) {
            if ( found = ( idmap[ndx] == ridId )) {
              break ;
            }
          }
          if ( !found ) {
            continue ;
          }
          UMPrincipalList umpl = (UMPrincipalList)rv[ndxmap[ndx]] ;
          if ( umpl == null ) {
            continue ;
          }
          int principalType = rs.getInt ( 2 ) ;
          String principalName = sm.getUntrimString ( rs.getString ( 3 )) ;
          IUMPrincipal ump = getUMPrincipal ( principalName,
            getPrincipalTypeFromTag ( principalType ), raw ) ;
          if ( ump == null ) {
            continue ;
          }
          umpl.add ( ump ) ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    return rv ;
  }

  protected boolean uncommittedHasOwnersInDB ( Connection con,
  String internalAclID ) throws SQLException {
    /*
    * return false unless there is at least one owner for this id
    */
    boolean rv = false ;
    long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
    if ( ridId == -1 ) {
      return rv ;
    }
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDHASOWNERSINDB_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Add an owner to the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddOwner ( Connection con, String internalAclID,
  IUMPrincipal owner ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      owner = populate ( owner ) ;
      String trimmedOwner = sm.getTrimMatchString ( owner.getId()) ;
      int pt = getPrincipalTypeTag ( owner ) ;
      long ridId = uncommittedAddOrGetAclRid ( con, internalAclID ) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDOWNER_PS ) ;
      ps.setString ( 1, trimmedOwner ) ;
      ps.setInt ( 2, pt ) ;
      ps.setLong ( 3, ownerId ) ;
      ps.setLong ( 4, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove an owner from the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param principalName TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveOwner ( Connection con,
  String internalAclID, String principalName, int principalType )
  throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
/*
* XXX: good question! should we return 'true' as we've successfully
* reached the state of non-existance? Or should we return 'false' as
* we haven't run the 'remove' statement?
*/
        return false ;
      }
      String trimmedPrincipalName = sm.getTrimMatchString ( principalName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEOWNER_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      ps.setString ( 3, trimmedPrincipalName ) ;
      ps.setInt ( 4, getPrincipalTypeTag ( principalType )) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove the owner from all ACLs
   *
   * @param con TBD: Description of the incoming method parameter
   * @param principalName TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveOwner ( Connection con,
  String principalName, int principalType ) throws SQLException {
    PreparedStatement ps = null ;
    String trimmedPrincipalName = sm.getTrimMatchString ( principalName ) ;
    short principalTypeTag = (short)getPrincipalTypeTag ( principalType ) ;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEOWNER_PS2 ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPrincipalName ) ;
      ps.setShort ( 3, principalTypeTag ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return the members of the permission
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetPermissionMembers ( Connection con,
  String permissionName ) throws SQLException {
    if ( permissionName.equals ( IAclPermission.ACL_PERMISSION_FULL_CONTROL )) {
      return uncommittedGetAllPermissionsNotFullControl ( con ) ;
    }
    AclPermissionList rv = new AclPermissionList() ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    List ml = new ArrayList() ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDGETPERMISSIONMEMBERS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        String memberName = sm.getUntrimString ( rs.getString ( 1 )) ;
        ml.add ( memberName ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    for ( int i = 0 ; i < ml.size() ; i++ ) {
      String memberName = (String)ml.get ( i ) ;
      IAclPermission permission = uncommittedGetPermission ( con, memberName ) ;
      if ( permission != null ) {
        rv.add ( permission ) ;
      }
    }
    return rv ;
  }

  private class PermRec {
    public String permName ;
    public boolean isPredefined ;
  }
  /**
   * @param con TBD: Description of the incoming method parameter
   * @return a list of all permissions (WITHOUT fullcontrol)
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetAllPermissionsNotFullControl (
  Connection con ) throws SQLException {
    AclPermissionList rv = new AclPermissionList() ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    List ml = new ArrayList() ;
    try {
      ps = con.prepareStatement (
        UNCOMMITTEDGETALLPERMISSIONSNOTFULLCONTROL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        String permissionName = sm.getUntrimString ( rs.getString ( 1 )) ;
        if ( !IAclPermission.ACL_PERMISSION_FULL_CONTROL.equals (
        permissionName )) {
          PermRec pr = new PermRec() ;
          pr.permName = permissionName ;
          pr.isPredefined = rs.getShort ( 2 ) != 0 ;
          ml.add ( pr ) ;
        }
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    for ( int i = 0 ; i < ml.size() ; i++ ) {
      PermRec pr = (PermRec)ml.get ( i ) ;
      IAclPermissionList members = uncommittedGetPermissionMembers ( con,
        pr.permName ) ;
      JDBCPermission p = new JDBCPermission ( this, pr.permName,
        pr.isPredefined, members ) ;
      rv.add ( p ) ;
    }
    return rv ;
  }

  /**
   * Remove all owners of the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveOwners ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEOWNERS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove all ACE of the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveAclEntries ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEACLENTRIES_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove all ACE of the ACL which belong to the specified principal
   *
   * @param con TBD: Description of the incoming method parameter
   * @param principalName TBD: Description of the incoming method parameter
   * @param principalType TBD: Description of the incoming method parameter
   * @exception SQLException Exception raised in failure situation
   */
  protected void uncommittedRemoveAclEntries ( Connection con,
  String principalName, int principalType ) throws SQLException {
    PreparedStatement ps = null ;
    String trimmedPrincipalName = sm.getTrimMatchString ( principalName ) ;
    short principalTypeTag = (short)getPrincipalTypeTag ( principalType ) ;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEACLENTRIES_PS2 ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPrincipalName ) ;
      ps.setShort ( 3, principalTypeTag ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @return true iff permissions already have been created
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckInitialPermissions ( Connection con )
  throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDCHECKINITIALPERMISSIONS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @return true iff supported permissions already have been assigned
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedCheckInitialSupportedPermissions (
  Connection con ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDCHECKINITIALSUPPORTEDPERMS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return true iff the permission is supported for the object type
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsSupportedPermission ( Connection con,
  String permissionName, String objectTypeName ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      String trimmedObjectTypeName = sm.getTrimMatchString ( objectTypeName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDISSUPPORTEDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setString ( 3, trimmedObjectTypeName ) ;
      rs = ps.executeQuery() ;
      boolean rv = false ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return the supported permissions for the object type
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermissionList uncommittedGetSupportedPermissions (
  Connection con, String objectTypeName ) throws SQLException {
    /*
    * XXX: strange!? shouldn't we be reading from the cache here?
    * just like we do in the same call with the permission name as
    * a parameter?
    */
    AclPermissionList rv = new AclPermissionList() ;
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    List pnl = new ArrayList ( 20 ) ;
    try {
      String trimmedObjectTypeName = sm.getTrimMatchString ( objectTypeName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDGETSUPPORTEDPERMISSIONS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedObjectTypeName ) ;
      rs = ps.executeQuery() ;
      while ( rs.next()) {
        String permissionName = sm.getUntrimString ( rs.getString ( 1 )) ;
        pnl.add ( permissionName ) ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    for ( int i = 0 ; i < pnl.size() ; i++ ) {
      String permissionName = (String)pnl.get ( i ) ;
      IAclPermission permission = uncommittedGetPermission ( con,
        permissionName ) ;
      if ( permission != null ) {
        rv.add ( permission ) ;
      }
    }
    if ( rv.size() > 0 ) {
      cacheAddSupportedPermissions ( objectTypeName, rv ) ;
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return IAclPermission object for the specified permission name
   * @exception SQLException Exception raised in failure situation
   */
  protected IAclPermission uncommittedGetPermission ( Connection con,
  String permissionName ) throws SQLException {
    IAclPermission rv = cacheGetPermission ( permissionName ) ;
    if ( rv != null ) {
      return rv ;
    }
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    String pn = null ;
    boolean isPredefined = false ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDGETPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        pn = permissionName ;
        isPredefined = rs.getShort ( 1 ) != 0 ;
      }
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
    if ( pn != null ) {
      IAclPermissionList ml = uncommittedGetPermissionMembers ( con, pn ) ;
      rv = new JDBCPermission ( this, pn, isPredefined, ml ) ;
      cacheAddPermission ( rv ) ;
    }
    return rv ;
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return true iff the permission is user in some ACL
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsUsedPermission ( Connection con,
  String permissionName ) throws SQLException {
    PreparedStatement ps = null;
    ResultSet rs = null;
    try {
      /*
      * run through segmentor in order to append the delimiter
      * character
      */
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      boolean rv = false ;
      ps = con.prepareStatement ( UNCOMMITTEDISUSEDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = rs.getInt ( 1 ) > 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Create a new permission
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param isPredefined TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddPermission ( Connection con,
  String permissionName, boolean isPredefined ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setShort ( 3, (short)((isPredefined)?(1):(0))) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove a permission
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermission ( Connection con,
  String permissionName ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Assign a supported permission to an object type
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddSupportedPermission ( Connection con,
  String permissionName, String objectTypeName ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      String trimmedObjectTypeName = sm.getTrimMatchString ( objectTypeName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDSUPPORTEDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setString ( 3, trimmedObjectTypeName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove a supported permission assignment
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectTypeName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemoveSupportedPermission ( Connection con,
  String permissionName, String objectTypeName ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      String trimmedObjectTypeName = sm.getTrimMatchString ( objectTypeName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDREMOVESUPPORTEDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setString ( 3, trimmedObjectTypeName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Add a new member to a permisson
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param memberName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedAddPermissionMember ( Connection con,
  String permissionName, String memberName ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      String trimmedMemberName = sm.getTrimMatchString ( memberName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDADDPERMISSIONMEMBER_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setString ( 3, trimmedMemberName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove a member from a permission
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @param memberName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermissionMember ( Connection con,
  String permissionName, String memberName ) throws SQLException {
    PreparedStatement ps = null ;
    String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
    String trimmedMemberName = sm.getTrimMatchString ( memberName ) ;
    try {
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEPERMISSIONMEMBER_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      ps.setString ( 3, trimmedMemberName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove all members from a permission
   *
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedRemovePermissionMembers ( Connection con,
  String permissionName ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEPERMISSIONMEMBERS_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param permissionName TBD: Description of the incoming method parameter
   * @return true iff the permission is predefined (and thus cannot be changed)
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsPredefinedPermission ( Connection con,
  String permissionName ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      String trimmedPermissionName = sm.getTrimMatchString ( permissionName ) ;
      ps = con.prepareStatement ( UNCOMMITTEDISPREDEFINEDPERMISSION_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setString ( 2, trimmedPermissionName ) ;
      boolean rv = false ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = rs.getShort ( 1 ) != 0 ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Remove the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   */
  protected void uncommittedRemoveAcl ( Connection con, String internalAclID )
  throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        throw new SQLException ( "rid id not found" ) ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDREMOVEACL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Lock the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @param lockingUser TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedLockAcl ( Connection con, String internalAclID,
  IUMPrincipal lockingUser ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      lockingUser = populate ( lockingUser ) ;
      String trimmedLockingUser = sm.getTrimMatchString ( lockingUser.getId()) ;
      ps = con.prepareStatement ( UNCOMMITTEDLOCKACL_PS ) ;
      ps.setString ( 1, trimmedLockingUser ) ;
      ps.setLong ( 2, ownerId ) ;
      ps.setLong ( 3, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Unlock the ACL
   *
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedUnlockAcl ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDUNLOCKACL_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return true iff the ACL is locked
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedIsAclLocked ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return false ;
      }
      boolean rv = false ;
      ps = con.prepareStatement ( UNCOMMITTEDISACLLOCKED_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        String lockingUser = sm.getUntrimString ( rs.getString ( 1 )) ;
        rv = ( lockingUser != null ) && ( lockingUser.length() > 0 ) ;
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * @param con TBD: Description of the incoming method parameter
   * @param internalAclID TBD: Description of the incoming method parameter
   * @return the user which locks the ACL (or null if it is not locked)
   * @exception SQLException Exception raised in failure situation
   */
  protected IUser uncommittedGetAclLockingUser ( Connection con,
  String internalAclID ) throws SQLException {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      IUser rv = null ;
      long ridId = uncommittedGetAclRid ( con, internalAclID ) ;
      if ( ridId == -1 ) {
        return rv ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDGETACLLOCKINGUSER_PS ) ;
      ps.setLong ( 1, ownerId ) ;
      ps.setLong ( 2, ridId ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        String un = sm.getUntrimString ( rs.getString ( 1 )) ;
        if (( un != null ) && ( un.length() > 0 )) {
          rv = (IUser)getUMPrincipal ( un, IUMPrincipal.IUSER, false ) ;
        }
      }
      return rv ;
    }
    finally {
      if ( rs != null ) {
        rs.close() ;
      }
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  protected boolean uncommittedChangeAclID_in_AclTable ( Connection con,
  String oldID, String newID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      /*
      * well, the acl rid is not in the acl table itself any longer; right
      * now there is no ref counting in the acl rid table so we can't just
      * drop the entry from that table; instead, we simply add/lookup the
      * new acl rid and then update the rid auto_id in the acl table
      */
      long newRidId = uncommittedAddOrGetAclRid ( con, newID ) ;
      long oldRidId = uncommittedGetAclRid ( con, oldID ) ;
      if ( oldRidId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDCHANGEACLID_IN_ACLTABLE_PS ) ;
      ps.setLong ( 1, newRidId ) ;
      ps.setLong ( 2, ownerId ) ;
      ps.setLong ( 3, oldRidId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Change ACL ID in the ACEs table
   *
   * @param con TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedChangeAclID_in_AclEntryTable (
  Connection con, String oldID, String newID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      /*
      * well, the acl rid is not in the acl table itself any longer; right
      * now there is no ref counting in the acl rid table so we can't just
      * drop the entry from that table; instead, we simply add/lookup the
      * new acl rid and then update the rid auto_id in the acl table
      */
      long newRidId = uncommittedAddOrGetAclRid ( con, newID ) ;
      long oldRidId = uncommittedGetAclRid ( con, oldID ) ;
      if ( oldRidId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDCHANGEACLID_IN_ACLENTRYTABLE_PS ) ;
      ps.setLong ( 1, newRidId ) ;
      ps.setLong ( 2, ownerId ) ;
      ps.setLong ( 3, oldRidId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  /**
   * Change ACL ID in the owners table
   *
   * @param con TBD: Description of the incoming method parameter
   * @param oldID TBD: Description of the incoming method parameter
   * @param newID TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception SQLException Exception raised in failure situation
   */
  protected boolean uncommittedChangeAclID_in_OwnerTable ( Connection con,
  String oldID, String newID ) throws SQLException {
    PreparedStatement ps = null ;
    try {
      /*
      * well, the acl rid is not in the acl table itself any longer; right
      * now there is no ref counting in the acl rid table so we can't just
      * drop the entry from that table; instead, we simply add/lookup the
      * new acl rid and then update the rid auto_id in the acl table
      */
      long newRidId = uncommittedAddOrGetAclRid ( con, newID ) ;
      long oldRidId = uncommittedGetAclRid ( con, oldID ) ;
      if ( oldRidId == -1 ) {
        return false ;
      }
      ps = con.prepareStatement ( UNCOMMITTEDCHANGEACLID_IN_OWNERTABLE_PS ) ;
      ps.setLong ( 1, newRidId ) ;
      ps.setLong ( 2, ownerId ) ;
      ps.setLong ( 3, oldRidId ) ;
      boolean rv = ps.executeUpdate() > 0 ;
      if ( rv ) {
        ps.close() ;
        ps = null ;
        updateVersion ( con ) ;
      }
      return rv ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private final static long minAgeLimit = 1 * 1000 ; /* 1 second */
  private final static long maxAgeLimit = minAgeLimit << 5 ;
  private long curAgeLimit = minAgeLimit, lastReadTS = 0 ;
  private volatile long cachedVersion = -1 ;
  private Object ageLock = new Object() ;

  private void updateVersion ( Connection con ) throws SQLException {
    /*
    * could add logic to check against the database only after
    * cluster timeout period has passed
    */
    PreparedStatement ps = null ;
    try {
      ps = con.prepareStatement ( UPDATEVERSION_PS ) ;
      ps.executeUpdate() ;
      synchronized ( ageLock ) {
        /*
        * force re-reading of version counter
        */
        lastReadTS = 0 ;
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "getDBVersion", "Can't retrieve DB Version!" ) ;
    }
    finally {
      if ( ps != null ) {
        ps.close() ;
      }
    }
  }

  private void checkVersion ( Connection con ) {
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    try {
      ps = con.prepareStatement ( CHECKVERSION_PS1 ) ;
      rs = ps.executeQuery() ;
      if ( ! rs.next()) {
        rs.close() ;
        rs = null ;
        ps.close() ;
        ps = null ;
        ps = con.prepareStatement ( CHECKVERSION_PS2 ) ;
        ps.executeUpdate() ;
        synchronized ( ageLock ) {
          /*
          * force re-reading of version counter
          */
          lastReadTS = 0 ;
        }
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "checkVersion", "Can't check DB Version!" ) ;
    }
    finally {
      try {
        if ( rs != null ) {
          rs.close() ;
        }
        if ( ps != null ) {
          ps.close() ;
        }
      }
      catch ( SQLException se ) {
        s_log.debugT ( com.sapportals.wcm.util.logging.LoggingFormatter
          .extractCallstack ( se )) ;
      }
    }
  }

  /**
   * Returns the Modifycounter of the acl tables or -1 if not possible to
   * retrieve.
   *
   * @return dBVersion
   */
  public long getDBVersion() {
    long localCachedVersion ;
    synchronized ( ageLock ) {
      localCachedVersion = cachedVersion ;
      if ( System.currentTimeMillis() - lastReadTS < curAgeLimit ) {
        return cachedVersion ;
      }
    }
    PreparedStatement ps = null ;
    ResultSet rs = null ;
    long rv = -1 ;
    Connection c = null ;
    try {
      c = conPool.getConnection() ;
      if ( !c.getAutoCommit()) {
        c.setAutoCommit ( true ) ;
      }
      ps = c.prepareStatement ( GETDBVERSION_PS ) ;
      rs = ps.executeQuery() ;
      if ( rs.next()) {
        rv = rs.getLong ( 1 ) ;
      }
    }
    catch ( SQLException se ) {
      s_log.errorT ( "getDBVersion", "Can't retrieve DB Version!" ) ;
    }
    finally {
      try {
        if ( rs != null ) {
          rs.close() ;
        }
        if ( ps != null ) {
          ps.close() ;
        }
        if ( c != null ) {
          c.close() ;
        }
      }
      catch ( SQLException se ) {
        s_log.debugT ( com.sapportals.wcm.util.logging.LoggingFormatter
          .extractCallstack ( se )) ;
      }
    }
    synchronized ( ageLock ) {
      if ( localCachedVersion != cachedVersion ) {
        /*
        * looks like another thread walked in and updated the
        * cached value; let's trust that value for now
        */
        return cachedVersion ;
      }
      lastReadTS = System.currentTimeMillis() ;
      if ( rv == cachedVersion ) {
        if ( curAgeLimit < maxAgeLimit ) {
          curAgeLimit <<= 2 ;
        }
      }
      else {
        cachedVersion = rv ;
        curAgeLimit = minAgeLimit ;
      }
    }
    return rv ;
  }

  protected final void reopenMainConnection() throws SQLException,
  AclPersistenceException {
    /*
    * do nothing in case of OpenSQL as we can't hog a connection as
    * the thing that OpenSQL has handed out is just a proxy object
    * for the underlying real connection; and of course, every once
    * in a while OpenSQL will release the association between the
    * proxy object and the real connection which drives us to start
    * pumping mud
    */
  }

  protected final Connection getDBConnection() throws SQLException,
  AclPersistenceException {
    Connection con = conPool.getConnection() ;
    if ( con == null ) {
      throw new AclPersistenceException (
        "JDBCConnectionPool.getConnection() failed" ) ;
    }
    if (con.getAutoCommit()) con.setAutoCommit(false);
    if (con.getTransactionIsolation() != Connection.TRANSACTION_READ_COMMITTED) con.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
    return con ;
  }

  protected final void returnDBConnection ( Connection c ) {
    if ( c != null ) {
      try {
        c.close() ;
      }
      catch ( SQLException se ) {
        s_log.errorT ( "returnDBConnection", se.getMessage()) ;
      }
    }
  }
}
