/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl.jdbc;


import com.sapportals.wcm.util.acl.*;
import java.util.*;

/**
 * Default implementation of IAclPermission for database-persisted ACLs
 */
public final class JDBCPermission extends AclPermissionImpl {
  private final AbstractDatabaseConnectionUncached m_databaseConnection;
  private final String m_name;
  private final boolean m_isPredefined;
  private IAclPermissionList m_members = null;

  /**
   * Construct
   *
   * @param databaseConnection TBD: Description of the incoming method parameter
   * @param name TBD: Description of the incoming method parameter
   * @param isPredefined TBD: Description of the incoming method parameter
   * @param members TBD: Description of the incoming method parameter
   */
  protected JDBCPermission(AbstractDatabaseConnectionUncached databaseConnection, String name,
    boolean isPredefined, IAclPermissionList members) {
    if (databaseConnection == null || name == null || members == null) {
      throw new NullPointerException();
    }

    m_databaseConnection = databaseConnection;
    m_name = name;
    m_isPredefined = isPredefined;
    m_members = members;
  }

  /**
   * @return true iff the permission has no members
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isAtomic()
    throws AclPersistenceException {
    return m_members.size() == 0;
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the specified permission is a direct members
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isMember(IAclPermission permission)
    throws AclPersistenceException {
    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }

    IAclPermissionListIterator iterator = m_members.iterator();
    while (iterator != null && iterator.hasNext()) {
      IAclPermission listPermission = iterator.next();
      if (listPermission != null) {
        if (listPermission.equals(permission)) {
          return true;
        }
      }
    }

    return false;
  }

  /**
   * @return the permission members
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public IAclPermissionList getMembers()
    throws AclPersistenceException {
    return m_members;
  }

  /**
   * @param other TBD: Description of the incoming method parameter
   * @return true iff the specified permission is a direct or non-direct member
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isContained(IAclPermission other)
    throws AclPersistenceException {
    if (other == null) {
      throw new NullPointerException();
    }

    if (m_name.equals(IAclPermission.ACL_PERMISSION_FULL_CONTROL)) {
      return true;
    }

    if (m_name.equals(other.getName())) {
      return true;
    }

    if (this.isAtomic()) {
      // We are an atomic permission and not equal other, must be false
      //
      return false;
    }

    Set myAtomicPermissions = new HashSet(7);
    collectAtomicPermissions(this, myAtomicPermissions);

    if (other.isAtomic()) {
      // Other is atomic, see if it is contained in our member set
      //
      return myAtomicPermissions.contains(other);
    }
    else {
      // Both not atomic. other member set must be contained in this member set
      //
      Set hisAtomicPermissions = new HashSet(7);
      collectAtomicPermissions(other, hisAtomicPermissions);

      return myAtomicPermissions.containsAll(hisAtomicPermissions);
    }
  }

  /**
   * @return true permission name
   */
  public String getName() {
    return m_name;
  }

  /**
   * @return true iff the permission is predefined
   */
  public boolean isPredefined() {
    return m_isPredefined;
  }

  /**
   * Add a permission member
   *
   * @param permission member to be added
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PermissionExistsException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean addMember(IAclPermission permission)
    throws AclPersistenceException, PermissionExistsException, PredefinedPermissionException, PermissionUsedException {
    if (permission == null) {
      throw new NullPointerException();
    }

    if (m_isPredefined) {
      return false;
    }

    if (!m_databaseConnection.addPermissionMember(m_name, permission)) {
      return false;
    }

    m_members = m_databaseConnection.getPermissionMembers(m_name);
    return true;
  }

  /**
   * Remove a permission member
   *
   * @param permission TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception PredefinedPermissionException Exception raised in failure
   *      situation
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean removeMember(IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException {
    if (permission == null) {
      throw new NullPointerException();
    }

    if (m_isPredefined) {
      return false;
    }

    if (!m_databaseConnection.removePermissionMember(m_name, permission)) {
      return false;
    }

    m_members = m_databaseConnection.getPermissionMembers(m_name);
    return true;
  }

  public boolean equals(Object other) {
    if (other instanceof JDBCPermission) {
      return this.m_name.equals(((JDBCPermission)other).m_name);
    }
    return false;
  }

  public int hashCode() {
    return this.m_name.hashCode();
  }

  public String toString() {
    StringBuffer sb = new StringBuffer();
    sb.append("JDBCPermission[").append(m_name);
    if (m_members.size() > 0) {
      sb.append(", ").append(m_members);
    }
    sb.append("]");
    return sb.toString();
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @return true iff the permissions names are the same
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean equals(IAclPermission permission)
    throws AclPersistenceException {
    return m_name.equals(permission.getName());
  }

  /**
   * @param permission TBD: Description of the incoming method parameter
   * @param set TBD: Description of the incoming method parameter
   * @exception AclPersistenceException Exception raised in failure situation
   */
  private void collectAtomicPermissions(IAclPermission permission, Set set)
    throws AclPersistenceException {
    if (permission.isAtomic()) {
      set.add(permission);
      return;
    }

    IAclPermissionList members = permission.getMembers();
    IAclPermissionListIterator iterator = members.iterator();
    while (iterator.hasNext()) {
      IAclPermission memberPermission = iterator.next();
      collectAtomicPermissions(memberPermission, set);
    }
  }
}
